# This is a little small perl-script to execute EZStubs testcases
# automatically, it is not invoked directly but by invoking a make
# target %.sim_check or %.target_check in analogy to the make targets
# %.sim_gdb
# The script does not itself invoke the debugger directly but relies
# on another make target %.sim_run or %.target_run. This target invokes
# the debugger with a gdb.init-file specified by this script and
# requests the debugger to generate a log file. This log file is analyzed
# to find out whether the test was successful, or has failed
# Finally, as a testcase can also get stucked in an endless loop for
# example, a timeout has to be specified. In the case the testcase
# has not encountered a Panic() or a TestOK() before the timout
# is reached, the testcase is assumed to have failed

use File::Touch;
use File::Basename;
use POSIX;
use POSIX ":sys_wait_h";

my $target    = shift;
my $timeout   = shift;
my $remotecmd = shift;
my $report    = shift;
my $submissiondir = shift;
my $logprefix = shift;
my @testcases = @ARGV;

chdir($submissiondir);

logmsg("Building EZstubs library");
if(!-e 'tests/libsuccess') {
  my ($estatus, $to) = buildLib();
  if($estatus) {
    touch('tests/libfailed');
    logresult("failed");
  } else {
    touch('tests/libsuccess');
    logresult("success");
  }
} else {
  logresult("exists");
}

if(-e 'tests/libsuccess') {
  foreach my $testcase (@testcases) {
    # after this, we try to build the testcase
    if(-e "$testcase.build_failed") {
      unlink("$testcase.build_failed");
    }
    if(-e "$testcase.timeout") {
      unlink("$testcase.timeout");
    }

    logmsg($testcase);
    my ($estatus, $to) = buildTestcase($testcase);
    if($estatus) {
      touch("$testcase.build_failed");
      logresult("build failed");
    } else {
      ($has_sequence,$has_rightsequence) = hasSequence($testcase);
      genDebuggerScript($testcase,$has_sequence,$has_rightsequence,$port);
      runTestcase($testcase);
    }
  }
}

genReport($report,@testcases);

## END ##

# check whether a testcase employs a sequence to protocol the
# the proress of the testcase
sub hasSequence {
  my $testcase = shift;
  my $has_sequence = 0;
  my $has_rightsequence = 0;

  if(open(TESTCASE,"<$testcase.cc")) {
    while(<TESTCASE>) {
      if(/right_sequence/) {
        $has_rightsequence = 1;
      } elsif(/sequence/) {
        $has_sequence = 1;
      }

      if(($has_rightsequence == 1) && ($has_sequence == 1)) {
        last;
      }
    }

    close(TESTCASE);
  } else {
    die("hasSequence: unable to open file \"$testcase.cc\" for reading!\n");
  }
  return ($has_sequence,$has_rightsequence);
}

# generate a script for the gdb that queries
# the pc
# the adress of testok
# the right_sequence (if it exists)
# the sequence (if it exists)
sub genDebuggerScript {
  my $testcase = shift;
  my $has_sequence = shift;
  my $has_rightsequence = shift;

  if($remotecmd =~ m/<PORT>/) {
      my $port = `perl make/findport.pl`;
      $ENV{GDBPORT} = $port;
      $remotecmd =~ s/<PORT>/$port/;
  }

  if(open(DBG,">$testcase.gdb")) {      
    print DBG "$remotecmd\n";
    print DBG "load\n";
    print DBG "b BTestOK\n";
    print DBG "b BPanic\n";
    print DBG "cont\n";
    print DBG "echo RESULTS\\n\n";
    print DBG "echo PC:\n";
    print DBG "p /x \$pc\n";
    print DBG "echo BTESTOK:\n";
    print DBG "p /x &BTestOK\n";
    print DBG "echo MESSAGE:\n";
    print DBG "p __message__\n";

    if($has_rightsequence == 1) {
      print DBG "echo right_sequence:\n";
      print DBG "p right_sequence\n";
    }
    if($has_sequence == 1) {
      print DBG "echo sequence:\n";
      print DBG "p sequence\n";
    }

    print DBG "detach\n";
    print DBG "quit\n";

    close(DBG);
  }
  else {
    die("genDebuggerScript: unable to open file \"$testcase.gdb\" for writing!\n");
  }
}

# just invokes a specific make target
sub buildMakeTarget {
  my $maketarget = shift;
  my $timeout = shift;
  my $logfile = shift;
    print "buildmaketarget: " . $maketarget . "\n";
  return runchld($timeout, $logfile, "make", $maketarget);
}

# here we build the EZStubs-Lib, this means, we do a 'make all'
sub buildLib {
  return buildMakeTarget("all", 0, "buildlib.log");
}

# here we build the testcase
sub buildTestcase {
  my $testcase = shift;
  return buildMakeTarget("$testcase.elf", 0, $testcase."_build.log");
}

# invokes the appropriate make target and then waits until the
# testcase is finished or the timeout occurs
sub runTestcase {
  my $testcase = shift;
  my $estatus = undef;
  my $toterm = undef;

  ($estatus, $toterm) = buildMakeTarget("$testcase.".$target."_run", $timeout, "$testcase"."_run.log");
  if($toterm == 0) {
    logresult("normally terminated");
  } else {
  # timeout expired
    logresult("timeout exceeded");
    touch("$testcase.timeout");
  }
}

# generate a report from all the different testcases
sub genReport {
  my $reportfile = shift;
  my @tests = @_;

  logmsg("Generating report");
  if(open(REPORT,">$report")) {
    if(-e 'tests/libfailed') {
      print REPORT "Failed to build EZStubs-Library!\n";
      logresult("EZStubs library build failure");
    } else {
      print REPORT "Test\tfail/success\treason\tmessage\tright sequence\tsequence\n";

      my $suc=0, $to=0, $bf=0, $tf=0;
      foreach my $testcase (@tests) {
        print REPORT "$testcase\t";

        if(-e "$testcase.build_failed") {
          print REPORT "fail\tbuild"; $bf++;
        } elsif(-e "$testcase.timeout") {
          print REPORT "fail\ttimeout"; $to++;
        } else {
          my ($has_sequence,$has_rightsequence) = hasSequence($testcase);
          my ($pc,$testok,$message,$rseq,$seq) = parseDebuggerLog($testcase);

          if($pc == $testok + 4) {
            print REPORT "success"; $suc++;
          } else {
            print REPORT "fail\t";
            print REPORT "PC($pc) != TestOK($testok)\t";
            print REPORT "$message\t";
            print REPORT "$rseq\t";
            print REPORT "$seq";
            $tf++;
          }
        }
        print REPORT "\n";
      }
      logresult($suc+$to+$bf+$tf." tests, $suc ok, $bf build failures, $to timeout expirations, $tf runtime failures");
    }
    close(REPORT);
  } else {
    die("genReport: unable to open file \"$reportfile\" for writing!\n");
  }
}

sub parseDebuggerLog {
  my $testcase = shift;

  my $pc       = undef;
  my $testok   = undef;
  my $message  = undef;
  my $rseq     = undef;
  my $seq      = undef;

  if(open(LOG,"<$testcase.log")) {
    my $interesting_stuff = 0;

    while(<LOG>) {
      if(/RESULTS/) {
        $interesting_stuff = 1;
      } elsif($interesting_stuff == 0) {
        next;
      } else {
        if(/^PC:\$\d+\s*=\s*(0x[0-9a-fA-F]+)$/) {
          $pc = hex($1);
        } elsif(/^BTESTOK:\$\d+\s*=\s*(0x[0-9a-fA-F]+)$/) {
          $testok = hex($1);
        } elsif(/^sequence:\$\d+\s*=\s*"([^"]+)"$/) {
          $seq = $1;
        } elsif(/^right_sequence:\$\d+\s*=\s*(?:0x[0-9a-fA-F]+)\s*"([^"]+)"$/) {
          $rseq = $1;
        } elsif(/^MESSAGE:\$\d+\s*=\s*(?:0x[0-9a-fA-F]+)\s*"([^"]+)"$/) {
          $message = $1;
        }
      }
    }

    close(LOG);
  } else {
    die("parseDebuggerLog: unable to open file \"$testcase.log\" for reading!\n");
  }

  return ($pc,$testok,$message,$seq,$rseq);
}

# This procedure forks a child that is optionally killed after exceeding a timeout
# the output is redirected to the specified logfile
#   call runchld( timeout, logfile, cmdline, ... )
# where timeout is the maximum time to wait for the termination of the program,
# in seconds. 0 disables the timeout.
# logfile is a file to which all output (stdout+stderr) of the program is written
# to.
# Returns a tuple ($estatus, $toterm), where $estatus is the exitstatus of the
# called program and $toterm is set to 1 if the program was killed due to
# exceeding the timeout (0 if it terminated within the timeout)
sub runchld {
  my $timeout = shift;
  my $logfile = shift;
  my @cmdline = @_;
  my $kid;
  my $retval=0;
  my $exitstatus=undef;
  my $zombie;

  $kid = fork();
  defined($kid) || die "Failed to fork a new child";

  if($kid == 0) {
  # We move the child to a new session which allows us to
  # kill the whole clan at a single blow in case the child
  # does not quit within the timeout
    open(STDOUT, ">$logfile");
    open(STDERR, ">&STDOUT");
    POSIX::setsid();
    exec(@cmdline);
    die "Failed to launch @cmdline\n";
  }

  my $waitfor = $kid;

  if($timeout > 0) {
    my $timebomb = fork();
    defined($timebomb) || die "Failed to start the timeout process\n";
    if($timebomb == 0) {
      sleep($timeout);
      exit(0);
    }

    while(1) {
      $zombie = wait();
      if($zombie == $timebomb) {
      # Kill the ever-running child
        kill(-9, $kid);
        $retval++;
      } elsif ($zombie == $kid) {
        $exitstatus = $?;
      # Kid died in good time, disarm timebomb
        kill(15, $timebomb);
        $waitfor=$timebomb;
      } else { next; }
      last;
    }
  }

  # wait for kid to die
  do { $zombie = wait(); } until ($zombie == $waitfor);
  $exitstatus = $? unless (defined $exitstatus);

  return ($exitstatus>>8, $retval);
}

# Procedures for printing the progress
BEGIN {
my $indentlvl=1;
my $logmessage;
sub logmsg {
  my $msg = shift;
  my $i=0;
  $logmessage = "";
  while($i < $indentlvl) { $i++; $logmessage .= "\t"; }
  $logmessage .= "[$logprefix] ";
  $logmessage .= "$msg...";
}

sub logresult {
  my $msg = shift;
  print STDERR "$logmessage\t$msg\n";
}
}

