/* -*- c++ -*- ***************************************************************/
/* Echtzeitsysteme                                                           */
/*---------------------------------------------------------------------------*/
/*                                                                           */
/*                 I N T E R R U P T _ C O N T R O L L E R                   */
/*                                                                           */
/*---------------------------------------------------------------------------*/
/* Der Interrupt Controller des NDS			                                 */
/*---------------------------------------------------------------------------*/

#ifndef __devices_interrupt_controller_h__
#define __devices_interrupt_controller_h__

#include "infra/memory_mapped.h"
#include "infra/types.h"

/**
 * \brief Low level abstraction for the NDA's interrupt controller
 **/
class Interrupt_Controller {
  /**
   * \brief Interrupt Mask Enable Register
   **/
  Memory_Mapped< ezstubs_uint16,0x04000208 >  IME;

  /**
   * \brief Interrupt Enable Register
   **/
  Memory_Mapped< ezstubs_uint32,0x04000210 >  IE;

  /**
   * \brief Interrupt Request Flags
   **/
  Memory_Mapped< ezstubs_uint32,0x04000214 >  IF;

public:

  /**
   * \brief Enable all Interrupts
   **/
  void enable_int() {
    IME |= 0x1;
  }

  /**
   * \brief Disable all interrupts
   **/
  void disable_int() {
    IME &= ~0x1;
  }

  /**
   * \brief Mask a set of interupt vectors.
   * \param mask A bit mask of interrupt vectors to mask
   *
   * Mask an interrupt vector if the corresponding bit in
   * interrupt mask is set to one.
   **/
  void mask_int(ezstubs_uint32  mask) {
    IE &= ~mask;
  }

  /**
   * \brief Mask one specific vector
   * \param vector The vector to mask
   **/
  void mask_vector(ezstubs_uint32  vector) {
    IE &= ~(1 << vector);
  }

  /**
   * \brief Unmask a set of interupt vectors.
   * \param mask A bit mask of interrupt vectors to unmask
   *
   * Unmask an interrupt vector if the corresponding bit in
   * interrupt mask is set to one.
   **/
  void unmask_int(ezstubs_uint32 mask) {
    IE |= mask;
  }

  /**
   * \brief Unmask one specific vector
   * \param vector The vector to unmask
   **/
  void unmask_vector(ezstubs_uint32 vector) {
    IE |= (1 << vector);
  }

  /**
   * \brief Get the lowest vector that caused the interrupt
   * \return The lowest vector that caused the interrupt
   **/
  ezstubs_uint16 decode_int() {
    ezstubs_uint16 i = 0;

    while(!(IF & (1 << i))) i++;

    return i;
  }

  /**
   * \brief Acknowledge an interrupt request
   * \param vector The vector whose interrut request should be acknowledged
   **/
  void acknowledge_vector(ezstubs_uint32 vector) {
    IF |= (1 << vector);
  }

  /**
   * \brief Acknowledge the interrupt requests of a set of interrupt vectors
   * \param mask The interrupt vectors whose interrupt requests should 
   *             be acknowledged
   *
   * The interrupt vectors interrupt request will be acknowledged if the
   * corresponding bit is set to one
   **/
  void acknowlegde_int(ezstubs_uint32  mask) {
    IF |= mask;
  }

  /**
   * \brief Has a particular vector raised an interrupt request?
   * \param The vector to check.
   * \retval true The given has raised an interrupt request.
   * \retval false The given vector has not raised an interrupt request.
   **/
  bool vector_raised(ezstubs_uint32 vector) {
    return (IF & (1 << vector)) != 0;
  }
};

/**
 * \brief Global instance of the Interrupt Controller
 **/
extern Interrupt_Controller ic;

/**
 * \brief C-wrapper-function for the decode_int() method
 * \see Interrupt_Controller::decode_int()
 *
 * We need this wrapper, because we need to call it from
 * the interrupt handler written in assembler
 **/
extern "C" ezstubs_uint16 decode_int();

#endif /* __devices_interrupt_controller_h__ */
