/* -*- c++ -*- ***************************************************************/
/* Echtzeitsysteme                                                           */
/*---------------------------------------------------------------------------*/
/*                                                                           */
/*                    I N T E R R U P T _ T I M E R                          */
/*                                                                           */
/*---------------------------------------------------------------------------*/
/* Ein Timer, der einmal oder perdiodisch Interrupts generiert.              */
/*---------------------------------------------------------------------------*/

#ifndef __interrupt_timer_h__
#define __interrupt_timer_h__

#include "devices/timer.h"
#include "interrupt/gate.h"
#include "interrupt/plugbox.h"

/**
 * \class Interrupt_Timer
 * \brief A timer that can generate interrupts.
 */
class Interrupt_Timer : public Timer, public Gate {

protected:

  /**
   * \brief The period of interrupt generation
   **/
  us_type period_duration;

  /**
   * \brief Constructor
   * \param slot The interrupt vector of the interrupt generated by this
   *             timer.
   */
  Interrupt_Timer(unsigned int slot) {
    plugbox.assign(slot,this);
  }

public:

  /**
   * \brief Get the current period for generating interrupts.
   */
  us_type period() { return period_duration; } 

  /**
   * \brief Set the current period for generating interrupts.
   * \param period An interrupt should be generated every \arg period 
   *               microseconds
   *
   * Sets the period for generating interrupts. As a timer cannot generate
   * interrupts with arbitrary periods, a period similar to \arg period
   * is computed. To inform the user, the real period of one timer tick is 
   * returned.
   */
  virtual us_type period(us_type period) = 0;

  /**
   * \brief Get the maximal available interrupt period of this timer.
   * \return The maximal available interrupt period of this timer.
   **/
  virtual us_type max_period() = 0;

  /**
   * \brief Get the minimal available interrupt period of this timer.
   * \return The minimal available interrupt period of this timer.
   **/
  virtual ns_type min_period() = 0;

  /**
   * \brief Generate a single interrupt.
   *
   * Generate a single interrupt. In this case the period is regarded as
   * offset, not as a real period.
   */
  virtual void trigger_single() = 0;

  /**
   * \brief Generate interrupts periodically.
   *
   * Generate interrupts periodically with the period returned by
   * Interrupt_Timer::period().
   */
  virtual void trigger_periodical() = 0;

  /**
   * \brief Stop the generation of interrupts.
   */
  virtual void cancel() = 0;
};

#endif /* __interrupt_timer_h__ */
