/* -*- c++ -*- ***************************************************************/
/* Echtzeitsysteme                                                           */
/*---------------------------------------------------------------------------*/
/*                                                                           */
/*                               T I M E R                                   */
/*                                                                           */
/*---------------------------------------------------------------------------*/
/* Basisimplementierung eines Free Running Timers.                           */
/*---------------------------------------------------------------------------*/

#ifndef __timer_h__
#define __timer_h__

#include "infra/types.h"

/**
 * \class Timer
 * \brief A simple timer.
 **/
class Timer {

protected:

  /**
   * How many nanoseconds lasts one timer tick.
   **/
  ns_type tick_duration;

  /**
   * \brief The timer value captured at the beginning of a measurement
   **/
  tick_type start_value;

  /**
   * \brief The timer value captured at the end of a measurement
   **/
  tick_type stop_value; 

public:

  /**
   * \brief Destructor
   *
   * Only needed to prevent compiler warnings due to virtual methods
   **/  
  virtual ~Timer() {}

  /**
   * \brief How many nanoseconds lasts one timer tick.
   **/
  ns_type tick() { return tick_duration; }

  /**
   * \brief Sets the tick duration of the timer.
   * \param ns One timer tick should last \arg td nanoseconds.
   * \return How long one timer tick really lasts.
   *
   * Sets the tick duration of the timer. As a timer cannot generate ticks of
   * arbitrary durations, a duration similar to \arg td is computed. To inform
   * the user, the real duration of one timer tick is returned.
   **/
  virtual ns_type tick(ns_type ns) = 0;

  /**
   * \brief The maximal counter value
   * \return The maximal counter value
   **/
  virtual tick_type max_value() = 0;

  /**
   * \brief The current counter value.
   * \return The current counter value.
   * \attention Be aware that the counter value that is returned may be already
   *            out of date, when it is being read.
   **/
  virtual tick_type value() = 0;

  /**
   * \brief Start timer operation.
   **/
  virtual void start() = 0;

  /**
   * \brief Stop timer operation.
   **/
  virtual void stop() = 0;

  /**
   * \brief Start time measurement.
   * \attention May only be called after start.
   * \see start
   **/
  void start_time() { start_value = value(); }

  /**
   * \brief Stop time measurement.
   * \attention May only be called after start.
   * \see stop
   **/
  void stop_time() { stop_value = value(); }

  /**
   * \brief Return the measured time in microseconds.
   * \return Measured time.
   * \attention May only be called after start_time() and stop_time()
   **/
  us_type get_time();
};

#endif /* __timer_h__ */
