package de.fau.spicsim.gui

import java.awt.Color
import java.awt.event.ActionEvent
import java.awt.event.ActionListener
import de.fau.spicsim.Main
import de.fau.spicsim.SpicSim
import de.fau.spicsim.dev.DiscreteLowPass
import de.fau.spicsim.dev.PWMLowPass
import de.fau.spicsim.dev.PwmMon
import de.fau.spicsim.dev.Button
import de.fau.spicsim.dev.Adc
import de.fau.spicsim.interfaces.DevObserver
import de.fau.spicsim.interfaces.PinInterface
import de.fau.spicsim.interfaces.PinTristate
import javax.swing.AbstractButton
import javax.swing.JButton
import javax.swing.JToggleButton
import javax.swing.JSlider
import javax.swing.JFileChooser
import javax.swing.JOptionPane
import javax.swing.filechooser.FileNameExtensionFilter
import javax.swing.event.ChangeEvent
import javax.swing.event.ChangeListener

class ButtonMonitor(jb: AbstractButton, btn: Button) {

	val bModel = jb.getModel

	val cl = new ChangeListener {
		def stateChanged(cEvt: ChangeEvent) {
			if(jb.isInstanceOf[JToggleButton]){
				btn.pressed = bModel.isSelected()
			}else{
				btn.pressed = bModel.isPressed()
			}
		}
	}

	jb.addChangeListener(cl)
}

class SlideMonitor(js: JSlider, pc: Adc) {

	val cl = new ChangeListener {
		def stateChanged(cEvt: ChangeEvent) {
			pc.level = js.getValue
		}
	}
	js.addChangeListener(cl)

	val devo = new DevObserver {
		def notify(subject: Any, data: Any) {
			js.setValue(pc.level.min(js.getMaximum()))
		}
	}

	devo.notify(null, null)

	pc.addObserver(devo)

}

class SSegMon(ss: SevenSeg, el: Int, input: PwmMon) extends DevObserver {
	val col = Color.red

	val dlp = new DiscreteLowPass(new PWMLowPass(input), 256)
	dlp.addObserver(this)

	notify(dlp, null)

	private def getC(level: Int) = {
		val mult = (level.toFloat * 2 / 255).min(1).max(0.2f)
		new Color((col.getRed * mult).toInt, (col.getGreen * mult).toInt, (col.getBlue() * mult).toInt, col.getAlpha())
	}

	def notify(subject: Any, data: Any) {
		subject match {
			case dlp: DiscreteLowPass => ss.setSegment(el, getC(dlp.level))
			case _ => throw new Exception("Did not get a DLP")
		}
	}
}

class LedMon(led: Led, input: PwmMon) extends DevObserver {
	def col = led.color

	val dlp = new DiscreteLowPass(new PWMLowPass(input), 256)
	dlp.addObserver(this)
	notify(dlp, null)

	private def getC(level: Int) = {
		val mult = (level.toFloat * 2 / 255).min(1).max(0.2f)
		new Color((col.getRed * mult).toInt, (col.getGreen * mult).toInt, (col.getBlue() * mult).toInt, col.getAlpha())
	}

	def notify(subject: Any, data: Any) {
		subject match {
			case dlp: DiscreteLowPass => led.setForeground(getC(dlp.level))
			case _ => throw new Exception("Did not get a DLP")
		}
	}
}

class Gui(ssim: SpicSim, skin: String) {
	//Create Main Window
	private val mw = Main.skinManager.windowForSkin(skin)

	var lastpath = "."

	val bm = List(
		{ new ButtonMonitor(mw.btn(0), ssim.buttondev.btns(0)) },
		{ new ButtonMonitor(mw.btn(1), ssim.buttondev.btns(1)) }
	)

	//Connect Adc
	val sm = List(
		{ new SlideMonitor(mw.adc(0), ssim.adcdev.adc(0)) },
		{ new SlideMonitor(mw.adc(1), ssim.adcdev.adc(1)) }
	)

	val ssegm = for (sseg <- 0 to 1) yield {
		for (seg <- 0 to 6) yield {
			val smon = new SSegMon(mw.sSeg(sseg), seg, ssim.segwatch.segc(sseg)(seg))
			smon
		}
	}

	val ledm = for (i <- 0 to 7) yield {
		val led = new LedMon(mw.leds(i), ssim.ledwatch.leds(i))
		led
	}

	mw.btnLoad.addActionListener(new ActionListener {
		def actionPerformed(e: ActionEvent) {
			openFile
		}
	})

	mw.btnReset.addActionListener(new ActionListener {
		def actionPerformed(e: ActionEvent) {
			reset
		}
	})

	def openFile() {
		val chooser = new JFileChooser(lastpath);
		var filter = new FileNameExtensionFilter("ELF - File", "elf");
    	chooser.setFileFilter(filter);

		if (chooser.showOpenDialog(mw.frame) != JFileChooser.APPROVE_OPTION) return ;
		lastpath = chooser.getSelectedFile.getParentFile.toString
		val file = chooser.getSelectedFile

		if(ssim.checkFileExtension(file)){
			if (! file.exists() || ! file.isFile()) {
				Console.err.println("File not found")
				JOptionPane.showMessageDialog(mw.frame,
				    "File not found.",
				    "Failed to load",
				    JOptionPane.WARNING_MESSAGE)
				return
			}
		}else{
			Console.err.println("Wrong file extension")
			JOptionPane.showMessageDialog(mw.frame,
    			"Wrong file extension. Select an ELF file.",
    			"Failed to load",
    			JOptionPane.WARNING_MESSAGE)
			return
		}

		ssim.load(file)
		ssim.start
		/*if(sp != null) sp.dispose
		if(stackF != null) stackF.dispose*/

		/*sp = new StackPanel
		sim.addStackmon(sp)*/
	}

	def reset() {
		ssim.restart
	}

}