#include <stddef.h>
#include <stdbool.h>
#include <avr/io.h>
#include <avr/interrupt.h>
#include <avr/sleep.h>

#include "7seg.h"
#include "button.h"
#include "display.h"
#include "led.h"

#ifndef BAUD_RATE
#define BAUD_RATE      1200UL
#endif

#define COUNT(A) (sizeof(A)/sizeof(A[0]))
#define ADCSAMPLES 8

_Static_assert(ADCSAMPLES * 1023 < INT16_MAX, "ADCSAMPLES zu gross");

// Calculate ticks to skip for the 8 bit timer
#define TICKS(FACTOR)	((F_CPU) / (FACTOR * 8UL * BAUD_RATE))

// Choose best timer divisor
#if TICKS(1) < 1
#error BAUD_RATE zu hoch - Timer kann das nicht abbilden!
#elif TICKS(1) <= 65535
#define TCCR_MASK	(1 << CS40)
#define OCR_VALUE	TICKS(1)
#elif TICKS(8) <= 65535
#define TCCR_MASK	(1 << CS41)
#define OCR_VALUE	TICKS(8)
#elif TICKS(64) <= 65535
#define TCCR_MASK	((1 << CS41) | (1 << CS40))
#define OCR_VALUE	TICKS(64)
#elif TICKS(256) <= 65535
#define TCCR_MASK	(1 << CS12)
#define OCR_VALUE	TICKS(256)
#elif TICKS(1024) <= 65535
#define TCCR_MASK	((1 << CS42) | (1 << CS40))
#define OCR_VALUE	TICKS(1024)
#else
#error BAUD_RATE zu gering - Timer kann das nicht abbilden!
#endif

enum status {
	STATUS_DISABLED = -1,
	STATUS_ENABLED = 0,
	STATUS_READY = 1,
	STATUS_ACTIVE = 2,
};

static volatile enum status recvstatus = STATUS_ENABLED;
static volatile enum status sendstatus = STATUS_DISABLED;


static volatile uint8_t recvbuf[255];
static volatile uint16_t recvsize = 0;

static volatile uint8_t sendbuf[16] = {0xff, 0x00, 0xff, 0xff, 0xff, 0x00, 0xff, 0x00, 0x00, 0x00, 0x00, 0xff, 0xff, 0xff, 0xff, 0xff};

static volatile bool senddraw = false;
static volatile bool recvdraw = false;

static const char colName[] = { ' ', 'A', '0', '1', '2', '3', '4', '5', '6', '7', 'P', 'E' };

static const uint8_t binsym[2] = { 0x00, 0xff };

static volatile uint16_t adcValue = 0;

static void buttonCallback(BUTTON btn, BUTTONEVENT eve){
	if (eve == ONPRESS){
		uint8_t val, par = 0;
		switch (btn){
			case BUTTON1:
				recvstatus = STATUS_ENABLED;
				break;
			case BUTTON0:
				val = adcValue / 4;
				for (uint8_t i = 0; i < 8 ; i++){
					uint8_t bit = (val >> i) & 1;
					sendbuf[2+i] = binsym[bit];
					par ^= bit;
				}
				sendbuf[10] = binsym[par];
				sendstatus = STATUS_READY;
				break;
			default:
				return;
		}

		if ((TIMSK4 & OCIE4A) == 0){
			//ADCSRA &= ~((1 << ADIE) | (1<<ADEN));
			TIMSK4 |= (1 << OCIE4A);
		}
	}
}

ISR(TIMER4_COMPA_vect) {
	static uint8_t recvbit, recvbyte, sendbit;
	static uint16_t recvpos;

	switch (recvstatus){
		case STATUS_ENABLED:
			recvbyte = PIND & (1 << PD0);
			recvbuf[0] = binsym[recvbyte];
			recvsize = recvpos = 1;
			recvstatus = STATUS_READY;
			sb_led_off(RED0);
			sb_led_on(YELLOW0);
			break;
		case STATUS_READY:
			// Warte auf Änderung
			if (recvbyte == (PIND & (1 << PD0)))
				break;
			else {
				// Init
				recvbit = 0;
				recvbyte = 0;
				recvstatus = STATUS_ACTIVE;
				sb_led_off(YELLOW0);
				sb_led_on(GREEN0);
			}
			// no break
		case STATUS_ACTIVE:
			recvbyte |= (PIND & (1 << PD0)) << recvbit;
			if (++recvbit >= 8){
				recvbuf[recvpos++] = recvbyte;
				// Abbruchbedingung
				if (recvpos >= COUNT(recvbuf)){
					recvstatus = STATUS_DISABLED;
					sb_led_off(GREEN0);
					sb_led_on(RED0);
					recvdraw = true;
					// Trim
					for (uint16_t i = recvpos - 2; i > 12 && recvbuf[i] == recvbyte; i--)
						if (i < recvpos - 3)
							recvpos--;
					recvsize = recvpos;
				}
				recvbit = 0;
				recvbyte = 0;
			}
			break;
		default:
			break;
	}

	switch (sendstatus){
		case STATUS_ENABLED:
		case STATUS_READY:
			sendbit = 0;
			sendstatus = STATUS_ACTIVE;
			sb_led_off(RED1);
			sb_led_on(GREEN1);
			// no break;
		case STATUS_ACTIVE:
			if ((sendbuf[sendbit / 8 ] >> (sendbit % 8)) & 1)
				PORTD |= (1 << PD1);
			else
				PORTD &= ~(1<< PD1);
			if (++sendbit >= 8 * sizeof(sendbuf)){
				sendstatus = STATUS_DISABLED;
				sb_led_off(GREEN1);
				sb_led_on(RED1);
				senddraw = true;
			}
			break;
		default:
			break;
	}

	if (recvstatus == STATUS_DISABLED && sendstatus == STATUS_DISABLED){
		TIMSK4 &= ~(1 << OCIE4A);
		//ADCSRA |= (1 << ADIE) | (1<<ADEN) |  (1<<ADSC);
	}
}

ISR(ADC_vect){
	static int16_t result = 0;
	static uint8_t counter = 0;
	result += ADCW;
	if (++counter >= ADCSAMPLES){
		int16_t div = adcValue - result / ADCSAMPLES;
		if (div > 2 || div < -2)
			adcValue = result / ADCSAMPLES;
		counter = 0;
		result = 0;
	}
}

static char toHex(uint8_t n){
	uint8_t v = n & 0x0f;
	return v  + (v > 9 ? '7' : '0');
}

// Graph Buffer
static uint8_t graph[128];

static const char * symbolParity[3] = { "\x03", "!", "?" };
static const char * symbolFrame[3] = { "\x03", "!", "\x13" };
static const uint8_t delimiter[16] = { 0x2A, 0x7E, 0x3E, 0x14, 0x14, 0x14, 0x14, 0x14, 0x14, 0x14, 0x3E, 0x1C, 0x7E, 0x2A, 0x2A, 0x2A};

static uint8_t overviewBytesPerPage = 0;

static void drawOverview(){
	uint16_t size = recvsize;
	if (size > 0){
		overviewBytesPerPage = (size+127)/128;
		uint8_t combined = 0; 
		for (uint16_t bit = 0; bit < size; bit++){
			// Hohe Pegel zählen
			uint8_t highs = 0;
			for (uint8_t pos = 2; pos < 6; pos++)
				if ((recvbuf[bit] >> pos) & 1)
					highs++;
			if (highs == 2)
				combined = 100; 
			else if (highs > 2)
				combined++;
			if (bit % overviewBytesPerPage + 1 == overviewBytesPerPage){
				graph[bit/overviewBytesPerPage] = combined == 0 ? 0x40 : (combined == overviewBytesPerPage ? 0x10 : 0x70);
				combined = 0;
			}
		}
	} else 
		overviewBytesPerPage = 0;
	// Rest der Zeile loeschen
	for (uint8_t i = size/overviewBytesPerPage; i < 128; i++)
		graph[i] = 0x00;

	// Zeichne Graph
	sb_display_drawBitmap(4, 0, 1, 128, graph);
}

static void drawRecv(uint16_t start){
	static bool init = false;
	uint8_t frame = 0;
	uint8_t parity = 2;
	uint16_t size = recvsize;
	uint8_t recvByte = 0;
	uint8_t recvPar = 0;
	bool recvClear = true;
	
	char strbuf[4] = { 0, 0, 0, 0 };
	uint8_t prev = (start == 0 ? recvbuf[0] : (recvbuf[start - 1] >> 7)) & 1;
	for (uint8_t bit = 0; bit < 16; bit++){
		if (!init){
			// Trenner zeichnen
			sb_display_draw(1, 8 * bit, delimiter[bit]);
			sb_display_draw(3, 8 * bit, delimiter[bit]);

			// und beschriften
			if (bit < COUNT(colName)){
				strbuf[0] = colName[bit];
				sb_display_showStringSmall(1, 8 * bit + 3, strbuf);
			}
		}

		// Hohe Pegel zählen
		uint8_t highs = 0;
		if (bit + start >= size){
			for (uint8_t pos = 0; pos < 8; pos++)
				graph[bit*8+pos] = 0x00;
			highs = 2; 
		} else {
			for (uint8_t pos = 0; pos < 8; pos++){
			
				uint8_t val = (recvbuf[bit + start] >> pos) & 1;

				// Erstelle Graphpunkt
				graph[bit*8+pos] = prev == val ? (val ? 0x01 : 0x80) : (val ? 0x7F : 0xFE);

				// vorherigen Wert sichern für nächsten Graphpunkt
				prev = val;

				// Nur mittleren Werte werden ausgwertet
				if (pos > 1 && pos < 6 && val)
					highs++;
			}
		}
		// Invalides signal
		if (highs == 2){
			strbuf[0] = ' ';
			if (bit == 1 || bit == 11)
				frame++;
			else if (bit > 1 && bit < 10)
				recvClear = false;
		} else {
			bool ishigh = highs > 2;
			// Startbit
			if (bit == 1)
				frame += ishigh ? 1 : 0;
			// Paritätsbit korrekt?
			else if (bit == 10)
				parity = recvPar != ishigh;
			// Stoppbit korrekt?
			else if (bit == 11)
				frame += ishigh ? 0 : 1;
			// Empfangenen Wert zusammenbauen
			else if (ishigh && bit > 1 && bit < 10){
				recvByte |= 1 << (bit - 2);
				recvPar ^= 1;
			}

			// Prepariere ausgabe des logischen Werts
			strbuf[0] = ishigh ? '1' : '0';
		}
		// Schreibe Logischer Wert
		sb_display_showStringSmall(3, 8 * bit + 3, strbuf);
	}

	if (!init){
		// Status line
		sb_display_showString(0, 1, "Recv --  P   S");
		init = true;
	}
	// Zeichne Graph
	sb_display_drawBitmap(2, 0, 1, 128, graph);
	// Schreibe Wert
	if (recvClear){
		strbuf[0] = 'x';
		strbuf[1] = toHex(recvByte >> 4);
		strbuf[2] = toHex(recvByte);
	} else {
		strbuf[0] = '?';
		strbuf[1] = '?';
		strbuf[2] = ' ';
		parity = 2;
	}
	sb_display_showString(0, 37, strbuf);
	// Parität
	sb_display_showString(0, 85, symbolParity[parity]);
	// Frame
	sb_display_showString(0, 117, symbolFrame[frame]);
	// Overview
	uint8_t lower = start/overviewBytesPerPage;
	uint8_t upper = (start+16)/overviewBytesPerPage;
	for (uint8_t i = 0; i < 128; i++)
		graph[i] = i < lower || i > upper ? 0x00 : 0x03;
	sb_display_drawBitmap(5, 0, 1, 128, graph);
}

static void drawSend(){
	uint8_t prev = 0;
	char strbuf[2] = { 0, 0 };

	for (uint8_t bit = 0; bit < 16; bit++){
		// Hohe Pegel zählen
		uint8_t highs = 0;
		for (uint8_t pos = 0; pos < 8; pos++){
			uint8_t val = (sendbuf[bit] >> pos) & 1;

			// Erstelle Graphpunkt
			graph[bit*8+pos] = (bit == 0 && pos == 0) || prev == val ? (val ? 0x01 : 0x80) : (val ? 0x7F : 0xFE);

			// vorherigen Wert sichern für nächsten Graphpunkt
			prev = val;

			// Nur mittleren Werte werden ausgwertet
			if (pos > 1 && pos < 6 && val)
				highs++;
		}

		// Schreibe Logischer Wert
		sb_display_draw(7, 8 * bit, delimiter[bit]);
		strbuf[0] = highs == 2 ? ' ' : (highs > 2 ? '1' : '0');
		sb_display_showStringSmall(7, 8 * bit + 3, strbuf);
	}
	// Zeichne Graph
	sb_display_drawBitmap(6, 0, 1, 128, graph);
}

void main(){
	DDRC &= ~(1<< PC0);
	ADMUX = (1<<REFS0);
	ADCSRB = 0;
	ADCSRA = (1<<ADPS2) | (1<<ADPS1) | (1<<ADPS0) | (1<<ADATE) | (1 << ADIE) | (1<<ADEN);
	ADCSRA |= (1<<ADSC);

	TCCR4A &= ~((1 << WGM41) | (1 << WGM40));
	TCCR4B |= (1 << WGM42) | TCCR_MASK;
	OCR4A = OCR_VALUE;
	TIMSK4 |= (1 << OCIE4A);

	// RX als Eingang
	DDRD &= ~(1 << PD0);
	// RX Pullup Widerstand
	PORTD |= 1<< PD0;

	// TX als Ausgang
	DDRD |= 1 << PD1;
	PORTD |= (1 << PD1);

	if (sb_display_enable() == 0){
		// Init
		sb_display_fillScreen(NULL);
		sb_button_registerCallback(BUTTON0, ONPRESS, buttonCallback);
		sb_button_registerCallback(BUTTON1, ONPRESS, buttonCallback);

		sei();
		uint16_t currentValue = adcValue;
		uint16_t prevStart = 1025;

		while (1){
			uint16_t start = recvsize < 10 ? 0 : (((recvsize - 11UL) * currentValue) / 1024UL);
			sb_7seg_showHexNumber(currentValue / 4);

			if (senddraw){
				drawSend();
				senddraw = false;
			}
			if (recvdraw){
				drawOverview();
				drawRecv(0);
				recvdraw = false;
			} else if (start != prevStart){
				drawRecv(start);
				prevStart = start;
			}

			cli();
			while (adcValue == currentValue && senddraw == false && recvdraw == false){
				/* Schlafen */
				sleep_enable();
				sei();
				sleep_cpu();
				sleep_disable();
				cli();
			}
			currentValue = adcValue;
			sei();
		}
	}
	while(1);
}
