#include <avr/pgmspace.h>
#include <stdint.h>
#include <stdbool.h>
#include <stdlib.h>

// From libspicboard
#include <adc.h>
#include <button.h>
#include <display.h>
#include <led.h>

// macros to make life a bit easier
#define COUNT(X) (sizeof(X)/sizeof(X[0]))
#define BIT(B,D) (((D) >> (B)) & 1)
#define BTN0(B) (((B) & (1)) != 0)
#define BTN1(B) (((B) & (2)) != 0)
#define ABOVE field[8]
#define BELOW field[9]

static const __flash struct {
	uint16_t rule[2];
	const char name[15];
	} world[]= {
	{ { 0x0008, 0x0008 }, "   G3 Welt    " },
	{ { 0x0008, 0x000A }, "   1G3 Welt   " },
	{ { 0x0008, 0x000C }, "2G3 / Cornway " },
	{ { 0x0008, 0x0018 }, "   4G3 Welt   " },
	{ { 0x0008, 0x0028 }, "   5G3 Welt   " },
	{ { 0x0008, 0x004C }, "26G3 (explode)" },
	{ { 0x0028, 0x002A }, "1G35 (wie 1G3)" },
	{ { 0x0008, 0x003E }, "1234G3 (Laby) " },
	{ { 0x00AA, 0x00AA }, "G1357 Kopierer" },
	{ { 0x0028, 0x0014 }, "  24/35 Welt  " },
	{ { 0x001F, 0x000F }, "blink. Flecken" },
	{ { 0x019F, 0x01DF }, " Anti-Cornway " },
};

static const __flash uint8_t splash[128*8] = { 0xfc, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x84, 0x84, 0x84, 0x7c, 0x40, 0x40, 0x7c, 0x84, 0x84, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0xfc, 0x00, 0x00, 0xb8, 0xb8, 0xb8, 0x00, 0x00, 0x00, 0x80, 0xe0, 0xf0, 0xf8, 0xf8, 0xf8, 0x00, 0x00, 0x00, 0x80, 0xc0, 0xc0, 0xc0, 0x60, 0x60, 0xe0, 0xc0, 0x00, 0xc0, 0xc0, 0xe0, 0xe0, 0x60, 0x60, 0x60, 0x60, 0xc0, 0xc0, 0x00, 0x60, 0x60, 0x60, 0x00, 0x00, 0x00, 0x00, 0x80, 0x80, 0xc0, 0xe0, 0x60, 0xe0, 0xe0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x40, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xc0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xff, 0x00, 0x40, 0x60, 0x50, 0xd0, 0x08, 0x08, 0x04, 0x02, 0x1e, 0xe0, 0x01, 0x06, 0x04, 0xfc, 0x04, 0x04, 0xfc, 0x08, 0x08, 0x10, 0x21, 0x42, 0x82, 0x04, 0x18, 0x60, 0x80, 0x00, 0x00, 0x00, 0x00, 0xff, 0x00, 0x00, 0xff, 0xff, 0xff, 0x00, 0x3c, 0x3f, 0x3f, 0x3f, 0x39, 0xff, 0xff, 0xff, 0x38, 0x38, 0x00, 0xc3, 0xc7, 0xce, 0xcc, 0xcc, 0xfc, 0x78, 0x30, 0x00, 0x00, 0xff, 0xff, 0x1d, 0x0c, 0x0e, 0x06, 0x07, 0x03, 0x01, 0x18, 0x7e, 0xfe, 0xe0, 0x60, 0x70, 0x78, 0xfe, 0xc7, 0xc3, 0xc1, 0xc0, 0x61, 0x71, 0x31, 0x10, 0x88, 0x8e, 0x72, 0x20, 0x20, 0x10, 0xfc, 0x1c, 0x4c, 0x62, 0x22, 0x3a, 0x14, 0x10, 0x08, 0x3c, 0x64, 0x40, 0x40, 0x20, 0x10, 0x38, 0x5e, 0x49, 0x47, 0x00, 0x20, 0x10, 0x18, 0x67, 0x40, 0x40, 0x20, 0x10, 0x3c, 0x4a, 0x49, 0x47, 0x20, 0x20, 0x10, 0x00, 0xf3, 0x12, 0x12, 0x12, 0x3f, 0xc0, 0x00, 0x00, 0x3f, 0xd3, 0x12, 0x17, 0x12, 0x12, 0x12, 0xf3, 0x00, 0x00, 0xf3, 0x12, 0x12, 0x12, 0x12, 0xd2, 0x3f, 0x00, 0x00, 0x80, 0x7f, 0x12, 0x12, 0x12, 0x12, 0xf3, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x64, 0x74, 0x5c, 0x4c, 0x00, 0x38, 0x7c, 0x54, 0x5c, 0x58, 0x00, 0x7d, 0x7d, 0x00, 0x38, 0x7c, 0x44, 0xfc, 0xfc, 0x00, 0x04, 0x3f, 0x7f, 0x44, 0x44, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xff, 0x00, 0x00, 0x00, 0x00, 0x00, 0x03, 0x04, 0x08, 0x10, 0x21, 0x22, 0x42, 0x44, 0x44, 0xcf, 0x48, 0x48, 0xcf, 0x44, 0x44, 0x22, 0x21, 0x10, 0x08, 0x04, 0x02, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0xff, 0x00, 0x00, 0xe0, 0x10, 0x08, 0x08, 0x08, 0x08, 0x10, 0x00, 0x00, 0xc0, 0x60, 0x20, 0x20, 0x60, 0xc0, 0x00, 0x00, 0xe0, 0x40, 0x20, 0x20, 0x20, 0xc0, 0x00, 0x60, 0x80, 0x00, 0x00, 0xc0, 0x00, 0x00, 0x80, 0x60, 0x00, 0x40, 0x20, 0x20, 0x20, 0x60, 0xc0, 0x00, 0x20, 0xc0, 0x00, 0x00, 0x00, 0xc0, 0x20, 0x00, 0xc0, 0x20, 0x20, 0x20, 0x40, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x01, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0f, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x0f, 0x00, 0x00, 0x0f, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x0f, 0x00, 0x00, 0x03, 0x04, 0x08, 0x08, 0x08, 0x08, 0x04, 0x00, 0x00, 0x07, 0x0c, 0x08, 0x08, 0x0c, 0x07, 0x00, 0x00, 0x0f, 0x00, 0x00, 0x00, 0x00, 0x0f, 0x00, 0x00, 0x03, 0x0c, 0x03, 0x00, 0x03, 0x0c, 0x03, 0x00, 0x00, 0x06, 0x09, 0x09, 0x09, 0x05, 0x0f, 0x00, 0x40, 0x40, 0x33, 0x0e, 0x03, 0x00, 0x00, 0x00, 0x04, 0x09, 0x09, 0x09, 0x06, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xf8, 0xf8, 0x8c, 0x8c, 0x8c, 0x0c, 0x0c, 0x18, 0x00, 0x00, 0x00, 0xc0, 0xc0, 0x80, 0xc0, 0xc0, 0xc0, 0x80, 0x00, 0x00, 0x00, 0xcc, 0xcc, 0x00, 0x00, 0x80, 0x80, 0xc0, 0xc0, 0xc0, 0xc0, 0x80, 0x00, 0x00, 0xfe, 0xfe, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80, 0xc0, 0xc0, 0xc0, 0xc0, 0xfe, 0xfe, 0x00, 0x00, 0x80, 0x80, 0xc0, 0xc0, 0xc0, 0xc0, 0x80, 0x00, 0x00, 0x80, 0xc0, 0xc0, 0xc0, 0xc0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xfc, 0xfc, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80, 0x80, 0xc0, 0xc0, 0xc0, 0xc0, 0x80, 0x00, 0x00, 0xfe, 0xfe, 0x80, 0xc0, 0xc0, 0xc0, 0x80, 0x00, 0x00, 0x00, 0x80, 0x80, 0xc0, 0xc0, 0xc0, 0xc0, 0x80, 0x00, 0x00, 0xc0, 0xc0, 0xc0, 0xc0, 0xc0, 0xc0, 0x80, 0x00, 0x00, 0x00, 0x80, 0xc0, 0xc0, 0xc0, 0xc0, 0x00, 0x00, 0x00, 0x00, 0x30, 0x60, 0x61, 0x61, 0x61, 0x63, 0x73, 0x3e, 0x1c, 0x00, 0x00, 0xff, 0xff, 0x20, 0x60, 0x60, 0x60, 0x3b, 0x3f, 0x04, 0x00, 0x7f, 0x7f, 0x00, 0x04, 0x1f, 0x3f, 0x62, 0x62, 0x62, 0x62, 0x63, 0x03, 0x00, 0x3f, 0x7f, 0x60, 0x00, 0x00, 0x00, 0x00, 0x04, 0x1f, 0x3b, 0x60, 0x60, 0x60, 0x20, 0x7f, 0x7f, 0x00, 0x04, 0x1f, 0x3f, 0x62, 0x62, 0x62, 0x62, 0x63, 0x03, 0x00, 0x63, 0x67, 0x66, 0x6c, 0x7c, 0x38, 0x00, 0x00, 0x00, 0x00, 0x00, 0x7f, 0x7f, 0x60, 0x60, 0x60, 0x60, 0x60, 0x60, 0x04, 0x1f, 0x3f, 0x62, 0x62, 0x62, 0x62, 0x63, 0x03, 0x00, 0x7f, 0x7f, 0x20, 0x60, 0x60, 0x60, 0x3b, 0x1f, 0x04, 0x04, 0x1f, 0x3f, 0x62, 0x62, 0x62, 0x62, 0x63, 0x03, 0x00, 0x7f, 0x7f, 0x00, 0x00, 0x00, 0x00, 0x7f, 0x7f, 0x00, 0x00, 0x63, 0x67, 0x66, 0x6c, 0x7c, 0x38, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x07, 0x07, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 };

static const __flash uint8_t fau[] = { 0x00, 0x01, 0x02, 0x03, 0x06, 0x09, 0x0e, 0x18, 0x19, 0x1e, 0x27, 0x39, 0x60, 0x61, 0x78, 0x79, 0x80, 0x86, 0x98, 0x9e, 0xe0, 0xe6, 0xe7, 0xee, 0xee, 0xee, 0xee, 0xee, 0x0e, 0xa0, 0x0c, 0xee, 0xee, 0x0c, 0xe0, 0xe0, 0x00, 0xee, 0xae, 0xe0, 0xe0, 0x40, 0xee, 0x8e, 0xf3, 0x93, 0xe9, 0xe9, 0xe9, 0x69, 0x40, 0x10, 0x12, 0xf3, 0xf3, 0x73, 0x12, 0x10, 0xe0, 0xa0, 0xf3, 0xb3, 0xe0, 0xe0, 0x40, 0xf3, 0x93, 0xf6, 0x96, 0xe0, 0xe0, 0xe0, 0x40, 0x10, 0x34, 0x15, 0xf6, 0x36, 0x0a, 0x23, 0x0a, 0xf6, 0x36, 0x15, 0x34, 0x10, 0xe0, 0x20, 0xf6, 0xb6, 0xe0, 0xe0, 0x40, 0xf6, 0x96, 0xef, 0x8f, 0xee, 0xee, 0xee, 0x4e, 0xef, 0x6f, 0x05, 0x01, 0x80, 0x05, 0x0b, 0xef, 0x4f, 0x0e, 0x0c, 0xe0, 0xef, 0xaf, 0xe0, 0xe0, 0x40, 0xef, 0x8f, 0xf3, 0x93, 0xe9, 0xe9, 0xa9, 0xf3, 0xd3, 0x11, 0xf0, 0x30, 0x11, 0xf3, 0x93, 0x12, 0x30, 0x60, 0xf3, 0xb3, 0xe0, 0xe0, 0x40, 0xf3, 0x93, 0xf6, 0x96, 0xe0, 0xe0, 0x40, 0x10, 0x14, 0x15, 0xf6, 0xf6, 0xf6, 0xf6, 0xf6, 0x56, 0x15, 0x14, 0x10, 0x00, 0xf6, 0xb6, 0x14, 0x10, 0xe0, 0xc0, 0x10, 0x14, 0xf6, 0x96, 0xef, 0x8f, 0xe0, 0xe0, 0x00, 0x0c, 0x0e, 0xef, 0x8f, 0x08, 0xe1, 0xe1, 0x81, 0x08, 0xef, 0x8f, 0x0e, 0x0d, 0x08, 0x0b, 0xef, 0xaf, 0xee, 0x8e, 0xef, 0xaf, 0x0b, 0x08, 0xe9, 0x89, 0xe0, 0xc0, 0x07, 0xe9, 0xa9, 0x04, 0xe0, 0xe0, 0xe0, 0x00, 0x04, 0xe9, 0xa9, 0x07, 0x40, 0x22, 0x24, 0x26, 0xe9, 0xe9, 0xc9, 0x26, 0x24, 0x22, 0x80 } ; 

static const __flash char startfield[][15] = {
	" Zufallskarte ",
	" im Startbild ",
	" auf FAU-Logo "
};

static uint8_t field[10][130] = { { 0 } };
static uint16_t r[2];	// Rules (from world)
static int8_t w = 2;
static int8_t s = 0;

static uint8_t getPressedButtons() {
	static uint8_t previousState[2] = {UNKNOWN, UNKNOWN};
	uint8_t currentState[2] = { sb_button_getState(BUTTON0), sb_button_getState(BUTTON1) };
	uint8_t r = 0;
	for (uint8_t i=0;i<2;i++){
		r |= currentState[i] == PRESSED && previousState[i] == RELEASED ? (1 << i) : 0;
		previousState[i] = currentState[i];
	}
	return r;
}

static void menu(){
	sb_display_fillScreen(NULL);
	sb_display_showStringFromFlash(0, 0, PSTR("Spiel des Lebens"));
	sb_display_showStringSmallFromFlash(1, 36, PSTR("mit Regelwerk"));
	sb_display_showStringFromFlash(2, 8, world[w].name);
	sb_display_showStringSmallFromFlash(3, 8, PSTR("Nachbarn"));
	sb_display_showStringSmallFromFlash(4, 16, PSTR("Geburt"));
	sb_display_showStringSmallFromFlash(5, 28, PSTR("Tod"));
	sb_display_showStringFromFlash(6, 8, startfield[s]);
	sb_display_showStringFromFlash(7, 36, PSTR("starten"));
	char neighbours[] = { ' ', '0', ' ', '1', ' ', '2', ' ', '3', ' ', '4', ' ', '5', ' ', '6', ' ', '7', ' ', '8', ' ', '\0' };
	char rulePage[] = { ' ', ' ', ' ', ' ', ' ', ' ', ' ', ' ', ' ', ' ', ' ', ' ', ' ', ' ', ' ', ' ', ' ', ' ', ' ', '\0' };
	int8_t prev = -1;
	while (true){
		int8_t item = 11 - sb_adc_read(POTI)*12/1024;
		uint8_t button = getPressedButtons();
		if (item != prev || button != 0){
			// World
			if (item < 10 || prev < 10){
				sb_display_showString(2, 0, item == 0 ? "\21" : " ");
				sb_display_showString(2, 120, item == 0 ? "\20" : " ");
				if (item < 10 && button != 0){
					if (item == 0){
						if (BTN0(button))
							w = (w + 1) % COUNT(world);
						else if (BTN1(button) && --w < 0)
							w = COUNT(world) - 1;
						sb_display_showStringFromFlash(2, 8, world[w].name);
						r[0] = world[w].rule[0];
						r[1] = world[w].rule[1];
					} else if (BTN1(button))
						r[0] ^= 1UL << (item -1);
					else
						r[1] ^= 1UL << (item -1);
				}
				for (uint8_t i = 0; i <= 9; i++)
					neighbours[i*2] = i == item - 1 && i != 9 ? '<' : (i == item && i != 0 ? '>': ' ');
					sb_display_showStringSmall(3, 44, neighbours);
				for (uint8_t j = 0; j < 2 ; j++){
					for (uint8_t i = 0; i <= 8; i++)
						rulePage[i*2] = BIT(i,r[j]) != j ? 'X' : ' ';
					sb_display_showStringSmall(4 + j, 48, rulePage);
				}
			}
			if (item == 10 || prev == 10){
				sb_display_showString(6, 0, item == 10 ? "\20" : " ");
				sb_display_showString(6, 120, item == 10 ? "\21" : " ");
				if (button != 0){
					if (item == 10 && BTN0(button))
						s = (s + 1) % 3;
					else if (BTN1(button) && --s < 0)
						s = 2;
					sb_display_showStringFromFlash(6, 8, startfield[s]);
				}
			}
			if (item == 11 || prev == 11){
				sb_display_showString(7, 28, item == 11 ? "\20" : " ");
				sb_display_showString(7, 92, item == 11 ? "\21" : " ");
				if (item == 11 &&  button != 0)
					return;
			}
			prev = item;
		}
	}
}


static void simulate(){
	while (true){
		// Create helpers
		for (uint8_t p = 0 ; p < 8; p++){
			field[p][0] = field[p][128];
			field[p][129] = field[p][1];
		}
		for (uint8_t i = 0 ;i<130; i++){
			uint8_t above = 0;
			uint8_t below = 0;
			for (uint8_t p = 0 ; p < 8; p++){
				above |= BIT(7, field[p][i]) << ((p + 1) % 8);
				below |= BIT(0, field[p][i]) << ((p + 7) % 8);
			}
			ABOVE[i]=above;
			BELOW[i]=below;
		}
		// Calculate next generation
		uint16_t cells = 0;
		for (uint8_t p = 0 ; p < 8; p++){
			uint8_t prev = field[p][0];
			for (uint8_t i = 1 ;i<=128; i++){
				uint8_t current = field[p][i];
				uint8_t next = field[p][i+1];
				uint8_t newPage = 0;
				for (uint8_t b = 0; b<8; b++){
					uint8_t neighbours = BIT(b,prev) + BIT(b,next);
					if (b == 0)
						neighbours += BIT(p, ABOVE[i-1]) + BIT(p, ABOVE[i]) + BIT(p, ABOVE[i+1]);
					else
						neighbours += BIT(b-1, prev) + BIT(b-1, current) + BIT(b-1, next);
					if (b == 7)
						neighbours += BIT(p, BELOW[i-1]) + BIT(p, BELOW[i]) + BIT(p, BELOW[i+1]);
					else
						neighbours += BIT(b+1, prev) + BIT(b+1, current) + BIT(b+1, next);
					bool life = BIT(neighbours, r[BIT(b, current)]);
					cells += life;
					newPage |= life << b;
				}
				field[p][i] = newPage;
				prev = current;
			}
		}

		// Draw
		for (uint8_t p = 0 ; p < 8; p++)
			sb_display_drawBitmap(p, 0, 1, 128, &field[p][1]);

		// Indicate liveliness with leds
		sb_led_setMask((1L << (cells / 1024L)) - 1);

		// Interaction
		uint8_t button = getPressedButtons();
		if (BTN1(button))
			do {
				button = getPressedButtons();
			} while (!BTN1(button));
		else if (BTN0(button))
			return;
	}
}

void main(void){
	// init
	r[0] = world[w].rule[0];
	r[1] = world[w].rule[1];
	if (sb_display_enable() == 0){
		sb_display_fillScreenFromFlash(splash);
		while(getPressedButtons() == 0);

		while (true){
			menu();
			// Field init
			switch (s){
				case 1:
					for (uint8_t p = 0 ; p < 8; p++)
						for (uint8_t i = 1 ;i<=128; i++)
							field[p][i] = splash[p*128 + i -1];
					break;
				case 2:
				{
					// Compressed with a dictionary run length encoding
					uint16_t x=0;
					for (uint8_t d = 23; d < COUNT(fau); d++)
						for (uint8_t n = 0; n <= fau[d] >> 5; n++){
							field[x >> 7][(x & 0x7f) + 1] = fau[fau[d] & 0x1f];
							x++;
						}
					break;
				}
				default:
					for (uint8_t p = 0 ; p < 8; p++)
						for (uint8_t i = 1 ;i<=128; i++)
							field[p][i] = rand()%256;
			}
			// Draw
			for (uint8_t p = 0 ; p < 8; p++)
				sb_display_drawBitmap(p, 0, 1, 128, &field[p][1]);

			// Game
			simulate();
		}
	}
	sb_led_on(RED0);
	sb_led_on(RED1);
	while(true);
}
