#include <string.h>
#include <stdlib.h>
#include <stdint.h>
#include <led.h>
#include <7seg.h>
#include <timer.h>
#include <adc.h>
#include <display.h>
#include <avr/sleep.h>
#include <avr/io.h>
#include <avr/interrupt.h>
#include <avr/pgmspace.h>
#include <avr/eeprom.h>
#include <util/delay.h>

#define VERSION_STRING "1.0"

#define TICKS_PER_SECOND 18
#define TICKS_PER_WORLD_MOVE 3
// must be at least the biggest game object spawning (better some more space between)
#define TICKS_SPAWN_BLOCKED 20
#define TICKS_SPAWN_BLOCKED_PIPE 35
// must be at least wall size
#define TICKS_WALL_SPAWN_BLOCKED 8
#define TICKS_SMASHED_MUSHROOM 8
#define SPAWN_SIZE 32
#define MAX_GAME_OBJECTS 16
#define SPEED 1
#define DISPLAY_WIDTH 128
#define DISPLAY_HEIGHT 8
#define POINTS_DISPLAY_WIDTH 16
#define MIN_TICKS_JUMPING (10/SPEED)
#define MAX_TICKS_JUMPING (20/SPEED)
#define TICKS_RAGING (230/SPEED)
#define TICKS_INVINCIBLE 20
#define FLOWER_TICKS 30
#define JUMPING_HEIGHT 2
#define SHIFTING_WINDOW 32
#define MAX_DIFFICULTY 4
#define MAX_LEVEL_DIFFICULTY 32
#define MIN_LEVEL_DIFFICULTY 10
#define MAX_WALLS 7
#define MAX_HOLES 1
#define MAX_PIPES 2
#define MAX_ENEMY_POOL_SIZE 3
#define DEBOUNCING_TIME 100

#define SAVE_SREG (SREG & (1 << SREG_I))
#define RESTORE_SREG(a) (SREG |= (a))

//#define DEBUG_BUILD
#ifdef DEBUG_BUILD
#define DBG(a) {a}
#else
#define DBG(a) {if(0) {a;}}
#endif

//#define SLEEP_ENABLE

#include <bitmaps/mushroom.c>
#include <bitmaps/evil_mushroom.c>
#include <bitmaps/mario.c>
#include <bitmaps/wall.c>
#include <bitmaps/ground.c>
#include <bitmaps/coin.c>
#include <bitmaps/pipe.c>
#include <bitmaps/flower.c>
#include <bitmaps/star.c>
#include <bitmaps/flag.c>
#include <bitmaps/splashscreen.c>
#include <bitmaps/i4logo.c>
#include <strings.c>

// === Datastructure Definitions === //

typedef enum drawable_type {
	EMPTY           = 0,
	MARIO           = 1,
	MUSHROOM        = 2,
	EVIL_MUSHROOM   = 3,
	WALL            = 4,
	SPECIAL_WALL    = 5,
	PIPE            = 6,
	HOLE            = 7,
	COIN            = 8,
	FLAG            = 9
} drawable_type;

typedef enum state {
	OBJECT_MOVING           = (1 << 0),
	OBJECT_DIRECTION_RIGHT  = (1 << 1),
	OBJECT_JUMPING          = (1 << 2),
	OBJECT_SMALL            = (1 << 3),
	OBJECT_FALLING          = (1 << 4),
	OBJECT_RAGING           = (1 << 5),
	HAS_FLOWER              = (1 << 6),
	SPECIAL_WALL_USED       = (1 << 7)
} state;

#define OBJECT_IS_MOVING(state)          ((((state) & (OBJECT_MOVING)) != 0))
#define OBJECT_DIRECTION_IS_RIGHT(state) ((((state) & (OBJECT_DIRECTION_RIGHT)) != 0))
#define OBJECT_IS_JUMPING(state)         ((((state) & (OBJECT_JUMPING)) != 0))
#define OBJECT_IS_SMALL(state)           ((((state) & (OBJECT_SMALL)) != 0))
#define OBJECT_IS_FALLING(state)         ((((state) & (OBJECT_FALLING)) != 0))
#define OBJECT_IS_RAGING(state)          ((((state) & (OBJECT_RAGING)) != 0))
#define HAS_FLOWER(state)                ((((state) & (HAS_FLOWER)) != 0))
#define SPECIAL_WALL_IS_USED(state)      ((((state) & (SPECIAL_WALL_USED)) != 0))

typedef enum shutdown_errorcode {
	DISPLAY_UNAVAILABLE  = (1 << 0),
	DISPLAY_DISABLED     = (1 << 1),
	END_OF_MAIN          = (1 << 2),
	NO_MARIO             = (1 << 3),
	INVALID_ENEMY        = (1 << 4),
	TOO_MANY_GAMEOBJECTS = (1 << 5),
	UNKNOWN_ERROR        = 0xff
} shutdown_errorcode;

typedef enum game_state {
	MAIN_MENU,
	LEVEL_ACTIVE,
	LEVEL_PAUSED,
	LEVEL_ABORT,
	FLAG_ADDED,
	FLAG_FINAL,
	MARIO_DEAD,
	MARIO_WON
} game_state;

typedef struct drawable drawable;
struct drawable {
	drawable_type type;
	uint8_t page;
	uint8_t column;
	uint8_t width;
	uint8_t height;
	uint8_t collision_margin;
	void (*draw)(drawable *d);
	uint8_t state; //moving state
	uint8_t extra;
	uint8_t difficulty;
};

typedef enum task {
	NEW_GAME,
	HIGHSCORE,
	TUTORIAL,
	ABOUT
} task;

typedef void (*task_ptr)(void);
typedef struct mainmenu_entry {
	task task;
	task_ptr task_ptr;
	char *name;
} mainmenu_entry;

// === Main Menu Declarations === //

static void show_splashscreen(void);
static void write_highscore_to_eeprom(uint16_t value);
static uint16_t read_highscore_from_eeprom(void);
static mainmenu_entry *mainmenu(void);
static void new_game(void);
static void print_highscore(void);
static void do_tutorial(void);
static void print_about(void);
static void print_text(const char *text, uint8_t page, uint8_t column, uint8_t wait);
static void print_text_wrapping(const __flash char *text, uint8_t page, uint8_t column);

static mainmenu_entry mainmenu_entries[] = {
	{NEW_GAME,      &new_game,            "New Game"},
	{HIGHSCORE,     &print_highscore,     "Highscore"},
	{TUTORIAL,      &do_tutorial,         "Tutorial"},
	{ABOUT,         &print_about,         "About"},
};

// === Game Function Declarations === //

static void init(void);
static void init_level(void);
static game_state play_level(void);
static void start_timer(void);
static void stop_timer(void);
static void start_game(void);
static void pause_game(uint8_t);
static void game_tick(void);
static void shutdown(shutdown_errorcode error);
static uint8_t collide(drawable *a, drawable *b);
static uint8_t collide2(drawable *a, drawable *b);
static uint8_t collide3(drawable *a, uint8_t b_page, uint8_t b_height,
                        uint8_t b_column, uint8_t b_width);
static uint8_t collide4(drawable *a, uint8_t b_page, uint8_t b_height,
                        uint8_t b_column, uint8_t b_width);
static uint8_t _collide(uint8_t a_page, uint8_t a_height, uint8_t a_column,
                        uint8_t a_width, uint8_t a_collision_margin,
                        uint8_t b_page, uint8_t b_height, uint8_t b_column,
                        uint8_t b_width, uint8_t b_collision_margin);
static void mario_reduce_live(void);
static void mario_die(uint8_t animated);
static void mario_shrink(void);
static void mario_grow(drawable *d);
static void mario_rage(drawable *d);
static void draw_rectangle(uint8_t page, uint8_t column,
                           uint8_t height, uint8_t width);
static void fill_screen_from_mid(void);
static void draw_drawable(drawable *d, const uint8_t *bitmap);
static void balance_difficulty(void);

static void copy_drawable(drawable *src, drawable *dest);
static int8_t add_game_object(drawable *, uint8_t force);
static drawable *find_game_object(drawable_type type);
static void remove_game_object(drawable *);
static void draw_game_objects(void);
static void draw_ground(void);

static void add_mario(void);
static void draw_mario(drawable *d);
static void add_evil_mushroom(void);
static void draw_evil_mushroom(drawable *d);
static void add_wall(uint8_t column, uint8_t special);
static void draw_wall(drawable *d);
static void add_pipe(void);
static void draw_pipe(drawable *d);
static void add_hole(uint8_t random_start);
static void draw_hole(drawable *d);
static void add_coin(uint8_t column);
static void draw_coin(drawable *d);
static void add_flag(void);
static void draw_flag(drawable *d);
static void drop_flag(drawable *d);

// === Global Variables === //

static uint16_t EEMEM highscore = 0;

static drawable game_objects[MAX_GAME_OBJECTS];
static uint8_t number_game_objects = 0;
static drawable *mario = 0;
static uint8_t ticks_left_jumping = 0;
static uint8_t ticks_left_raging = 0;

static const uint8_t content_empty = 0;
static const uint8_t content_full = 0xFF;

static ALARM *alarm = NULL;
static volatile uint8_t tick = 0;
static volatile uint8_t button0 = 0;
static volatile uint8_t button1 = 0;
static game_state gamestate = MAIN_MENU;
static game_state old_gamestate = MAIN_MENU;
static uint8_t world_has_moved = 0;
static uint8_t current_difficulty = 0;
static uint8_t level_difficulty = 0;
static uint8_t level_ticks_jumping = 0;
static uint8_t current_level = 0;
static int16_t difficulties_passed = 0;
static uint8_t number_walls = 0;
static uint8_t number_holes = 0;
static uint8_t number_pipes = 0;
static uint8_t mario_ticking = 0;
static uint8_t smashed_mushroom = 0;
static uint8_t spawn_blocked = 0;
static uint8_t last_enemy_spawned = 0;
static drawable_type last_enemy = UNKNOWN_ERROR;
static uint8_t last_wall_spawned = 0;
static uint8_t ticks_left_invincible = 0;
static uint8_t points = 0;

// === Implementation === //

void main(void) {

	init();
	show_splashscreen();

	while(1) {
		mainmenu_entry *entry = mainmenu();
		entry->task_ptr();
	}

	shutdown(END_OF_MAIN);
	while(1);
}

static void show_splashscreen(void) {
	sb_display_fillScreen(NULL);
	sb_display_fillScreenFromFlash(splashscreen);

	uint8_t content[DISPLAY_WIDTH];
	for(uint8_t i = 0; i < DISPLAY_WIDTH; i++) {
		content[i] = splashscreen[7*128+i] | 0xf0;
	}

	button0 = 0;
	for(uint8_t i = 0; i < 100; i++) {
		if(button0) {
			break;
		}
		sb_display_drawBitmap(7, 0, 1, (DISPLAY_WIDTH*i)/100, content);
		sb_timer_delay(50);
	}
}

static void init(void) {

	DDRD &= ~(1 << PD2 | 1 << PD3); // configure buttons as input
	PORTD |= (1 << PD2 | 1 << PD3); // activate pull-up

	// activate button0 and button1 on falling edge
	EICRA &= ~(1 << ISC00 | 1 << ISC10);
	EICRA |= (1 << ISC01 | 1 << ISC11);

	EIMSK |= (1 << INT0 | 1 << INT1); // enable button0/button1 interrupts

	if(sb_display_available() != 0) {
		sb_led_on(RED0);
		shutdown(DISPLAY_UNAVAILABLE);
	}

	if(sb_display_enable() != 0) {
		sb_led_on(RED0);
		shutdown(DISPLAY_DISABLED);
	}

	sb_display_fillScreen(NULL);

	for(uint8_t i = 0; i < MAX_GAME_OBJECTS; i++) {
		game_objects[i].type = EMPTY;
	}

	sei();
}

static mainmenu_entry *mainmenu(void) {
	sb_display_fillScreen(NULL);
	sb_display_showStringWide(0, 4, "-- MAIN MENU --");
	sb_display_drawBitmapFromFlash(6, 0, 2, 16, mario_right);
	sb_display_drawBitmapFromFlash(4, DISPLAY_WIDTH-16, 2, 16, mario_left_jump);
	sb_display_drawBitmap(3, DISPLAY_WIDTH-12, 1, 8, wall_empty);
	sb_display_drawBitmap(2, DISPLAY_WIDTH-12, 1, 8, coin);
	sb_display_draw(7, 62, 0xff);
	sb_display_draw(7, 61, 0xff);
	sb_display_draw(7, 60, 0xff);
	sb_display_draw(7, DISPLAY_WIDTH-15, 0xff);
	sb_display_draw(7, DISPLAY_WIDTH-14, 0xff);
	sb_display_draw(7, DISPLAY_WIDTH-13, 0xff);

	drawable evil;
	evil.page = 6;
	evil.column = DISPLAY_WIDTH-40;
	evil.width = 16;
	evil.height = 2;

	start_timer();
	button0 = 0;
	uint8_t act_entry = 0;
	int8_t mushroom_direction = -1;

	while(1) {
		uint8_t num_entries = sizeof(mainmenu_entries)/sizeof(mainmenu_entries[0]);
		act_entry = (uint8_t) (sb_adc_read(POTI) * num_entries / 1024);

		if(button0) {
			button0 = 0;
			break;
		}

		if(tick) {
			tick = 0;
			evil.column += mushroom_direction;

			if(evil.column == 64
			   || evil.column == DISPLAY_WIDTH-32) {

				mushroom_direction *= -1;
			}

			draw_drawable(&evil, evil_mushroom);
		}

		for(uint8_t i = 0; i < num_entries; i++) {
			if(i == act_entry) {
				sb_display_drawBitmap(2+i, 20, 1, 8, mushroom);
			} else {
				sb_display_drawBitmap(2+i, 20, 1, 8, NULL);
			}

			sb_display_showString(2+i, 32, mainmenu_entries[i].name);
		}
	}

	fill_screen_from_mid();
	stop_timer();
	return mainmenu_entries+act_entry;
}

static void print_highscore(void) {
	sb_display_fillScreen(NULL);

	char buffer[6];
	draw_rectangle(2, 0, 4, 128);
	itoa(read_highscore_from_eeprom(), buffer, 10);
	sb_display_showStringWide(3, 8, "Highscore: ");
	sb_display_showStringWide(3, 8+11*8, buffer);
	sb_timer_delay(2000);
}

static void do_tutorial(void) {
	for(uint8_t i = 0; i < 4; i++) {
		sb_display_fillScreen(NULL);
		if(i == 2) {
			sb_display_drawBitmapFromFlash(5, 0, 2, 16, mario_right_jump);
		} else {
			sb_display_drawBitmapFromFlash(6, 0, 2, 16, mario_right);
		}
		print_text_wrapping(st[i], 0, 30);
		sb_timer_delay(3000);
	}

	sb_display_fillScreen(NULL);
	sb_display_drawBitmap(6, DISPLAY_WIDTH-32, 2, 16, evil_mushroom);
	print_text_wrapping(st[4], 0, 0);
	sb_display_drawBitmapFromFlash(4, DISPLAY_WIDTH-32, 2, 16, mario_left_jump);
	sb_timer_delay(300);
	sb_display_drawBitmap(4, DISPLAY_WIDTH-32, 2, 16, NULL);
	uint8_t c[sizeof(mario_left)];
	memcpy_P(c, mario_left, sizeof(mario_left));
	for(uint8_t i = 1; i < 16-1; i++) {
		uint8_t idx = 16*1+i;
		c[idx] = c[idx] | 0xE0;
	}
	sb_display_drawBitmap(6, DISPLAY_WIDTH-32, 2, 16, c);
	sb_timer_delay(300);
	sb_display_drawBitmapFromFlash(6, DISPLAY_WIDTH-32, 2, 16, mario_left_jump);
	sb_timer_delay(2400);

	sb_display_fillScreen(NULL);
	sb_display_drawBitmap(7, 0, 1, 8, pipe_body);
	sb_display_drawBitmap(6, 0, 1, 8, pipe_head);
	sb_display_drawBitmap(5, 0, 1, 8, flower);
	print_text_wrapping(st[5], 0, 30);
	sb_timer_delay(3000);

	for(uint8_t k = 6; k < 8; k++) {
		sb_display_fillScreen(NULL);
		for(uint8_t i = 6; i < 8; i++) {
			for(uint8_t j = 0; j < 1; j++) {
				sb_display_drawBitmap(i, j*8, 1, 8, ground);
			}

			for(uint8_t j = 3; j < 8; j++) {
				sb_display_drawBitmap(i, j*8, 1, 8, ground);
			}
		}
		print_text_wrapping(st[k], 0, 30);
		sb_timer_delay(3000);
	}

	for(uint8_t i = 8; i < 11; i++) {
		sb_display_fillScreen(NULL);
		if(i == 9) {
			sb_display_drawBitmapFromFlash(6, DISPLAY_WIDTH-32, 2, 16, small_mario_left);
		} else {
			sb_display_drawBitmapFromFlash(6, DISPLAY_WIDTH-32, 2, 16, mario_left);
		}
		sb_display_drawBitmap(5, DISPLAY_WIDTH-28, 1, 8, wall_special);
		print_text_wrapping(st[i], 0, 0);
		sb_timer_delay(300);
		sb_display_drawBitmapFromFlash(6, DISPLAY_WIDTH-32, 2, 16, mario_left_jump);
		sb_display_drawBitmap(5, DISPLAY_WIDTH-28, 1, 8, wall_empty);
		const uint8_t *content;
		switch(i) {
		case 8:  content = coin;     break;
		case 9:  content = mushroom; break;
		case 10: content = star;     break;
		}
		sb_display_drawBitmap(4, DISPLAY_WIDTH-28, 1, 8, content);
		sb_timer_delay(2700);
	}

	for(uint8_t i = 11; i < 13; i++) {
		sb_display_fillScreen(NULL);
		uint8_t content[4*3];
		for(uint8_t i = 0; i < sizeof(content); i++) {
			content[i] = 0xff;
		}
		sb_display_drawBitmap(2, 0, 4, 3, content);
		sb_display_drawBitmap(2, 3, 1, 8, flag);
		print_text_wrapping(st[i], 0, 30);
		if(i == 11) {
			sb_timer_delay(3000);
		}
	}

	sb_timer_delay(300);
	for(uint8_t i = 1; i < 4; i++) {
		sb_display_drawBitmap(2+i-1, 3, 1, 8, NULL);
		sb_display_drawBitmap(2+i, 3, 1, 8, flag);
		sb_timer_delay(300);
	}
	sb_timer_delay(1800);

	for(uint8_t i = 13; i < 15; i++) {
		sb_display_fillScreen(NULL);
		sb_display_drawBitmapFromFlash(6, 0, 2, 16, mario_right);
		sb_display_drawBitmap(6, DISPLAY_WIDTH/2, 2, 16, evil_mushroom);
		sb_display_drawBitmapFromFlash(3, DISPLAY_WIDTH-24-16, 3, 24, i4logo);
		print_text_wrapping(st[i], 0, 0);
		sb_timer_delay(3000);
	}

	fill_screen_from_mid();
}

static void print_about(void) {
	for(uint8_t i = 0; i < 3; i++) {
		sb_display_fillScreen(NULL);
		sb_display_drawBitmapFromFlash(6, DISPLAY_WIDTH-32, 2, 16, mario_left);
		print_text_wrapping(sa[i], 0, 0);
		sb_timer_delay(3000);
	}

	sb_display_fillScreen(NULL);
	sb_display_drawBitmapFromFlash(6, DISPLAY_WIDTH-32, 2, 16, mario_left);
	sb_display_drawBitmapFromFlash(5, 16, 3, 24, i4logo);
	print_text_wrapping(sa[3], 0, 0);
	sb_timer_delay(3000);

	sb_display_fillScreen(NULL);
	sb_display_drawBitmapFromFlash(6, DISPLAY_WIDTH-32, 2, 16, mario_left);
	print_text_wrapping(sa[4], 0, 0);
	sb_timer_delay(3000);

	sb_timer_delay(3000);

	fill_screen_from_mid();
}

static void print_text_wrapping(const __flash char *text, uint8_t page, uint8_t column) {
	const char *start;
	const char *end;
	const char *last_space;
	char buf[17];
	char s[strlen_P(text)+1];
	strcpy_P(s, text);
	uint8_t wait = 1;

	start = s;
	end = start;
	button0 = 0;
	while(*end != '\0') {

		last_space = 0;
		while(end-start < (DISPLAY_WIDTH-column)/8
		      && *end != '\n'
		      && *end != '\0') {

			end++;
			if(*end == ' ') {
				last_space = end;
			}
		}

		if(last_space != 0
		   && *end != '\0'
		   && *end != '\n') {
			end = last_space;
		}

		if(button0) {
			wait = 0;
		}

		strncpy(buf, start, end-start);
		buf[end-start] = '\0';
		print_text(buf, page, column, wait);
		page++;

		if(*end != '\0'
		   && (*end == '\n'
		       || last_space != 0)) {

			end++;
		}
		start = end;

		if(page >= DISPLAY_HEIGHT) {
			return;
		}
	}
}

static void print_text(const char *text, uint8_t page, uint8_t column, uint8_t wait) {

	char buf[2];
	buf[1] = '\0';
	button0 = 0;

	while(*text != '\0'
	      && column < DISPLAY_WIDTH) {

		buf[0] = *text;
		sb_display_showStringWide(page, column, buf);
		text++;
		column += 8;

		if(button0) {
			wait = 0;
		}

		if(wait) {
			sb_timer_delay(75);
		}
	}
}

static void write_highscore_to_eeprom(uint16_t value) {
	eeprom_write_word(&highscore, value);
}

static uint16_t read_highscore_from_eeprom(void) {
	uint16_t value = eeprom_read_word(&highscore);

	if(value == 0xffff) {
		write_highscore_to_eeprom(0);
		value = 0;
	}

	if((PIND & (1 << PD3)) == 0) {
		write_highscore_to_eeprom(0);
		value = 0;
	}

	return value;
}

static void new_game(void) {

	current_level = 1;
	uint16_t total_points = 0;
	game_state state;

	do {
		sb_display_fillScreen(NULL);
		draw_rectangle(2, 0, 4, 128);
		char buffer[6];
		itoa(current_level, buffer, 10);
		sb_display_showStringWide(3, 35, "Level ");
		sb_display_showStringWide(3, 35+6*8, buffer);
		sb_timer_delay(2000);

		init_level();
		state = play_level();

		if(state == LEVEL_ABORT) {
			break;
		}

		total_points += points * level_difficulty;

		sb_display_fillScreen(NULL);
		draw_rectangle(1, 0, 6, 128);
		if(state == MARIO_WON) {
			sb_display_showString(2, 8, "Level Won");
		} else if (state == MARIO_DEAD) {
			sb_display_showString(2, 8, "Level Lost");
		}

		sb_display_showString(4, 8, "Points: ");
		itoa(points*level_difficulty, buffer, 10);
		sb_display_showString(4, 8+8*8, buffer);

		sb_display_showString(5, 8, "Total: ");
		itoa(total_points, buffer, 10);
		sb_display_showString(5, 8+8*8, buffer);

		button0 = 0;
		while(button0 == 0);

		if(current_level < 99) {
			current_level++;
		}

	} while(state == MARIO_WON);

	uint16_t h = read_highscore_from_eeprom();

	if(h < total_points) {
		sb_display_fillScreen(NULL);

		char buffer[6];
		draw_rectangle(0, 0, 8, 128);
		itoa(total_points, buffer, 10);
		sb_display_showStringWide(1, 4, "New");
		sb_display_showStringWide(2, 4, "Highscore:");
		sb_display_showStringWide(2, 4+11*8, buffer);

		itoa(h, buffer, 10);
		sb_display_showStringWide(5, 4, "Old");
		sb_display_showStringWide(6, 4, "Highscore:");
		sb_display_showStringWide(6, 4+11*8, buffer);

		write_highscore_to_eeprom(total_points);

		sb_timer_delay(5000);
	}
}

static game_state play_level(void) {

	start_game();

	uint8_t cnt = 0;

	sei();
	while(1) {

		#ifdef SLEEP_ENABLE
		sleep_enable();
		cli();
		#endif
		while(tick == 0
		      && button0 == 0
		      && button1 == 0) {

			#ifdef SLEEP_ENABLE
			DBG(sb_led_on(BLUE1);)
			sei();
			sleep_cpu();
			cli();
			DBG(sb_led_off(BLUE1);)
			#endif
		}
		#ifdef SLEEP_ENABLE
		sei();
		sleep_disable();
		#endif

		if(tick) {
			tick = 0;
			DBG(sb_led_toggle(YELLOW0);)

			if(cnt == TICKS_PER_WORLD_MOVE
			   && gamestate != FLAG_FINAL) {

				world_has_moved = 1;
				cnt = 0;
			} else {
				world_has_moved = 0;
			}
			cnt++;

			if(world_has_moved) {
				last_wall_spawned++;
				last_enemy_spawned++;
			}

			spawn_blocked = 0; // gets set in draw_game_objects()
			draw_ground();
			draw_game_objects(); // must be called after draw_ground for pipes

			balance_difficulty();

			char tmp[8];
			DBG(sb_display_showStringSmall(0, 0, itoa(current_difficulty, tmp, 10));)
			DBG(sb_display_showStringSmall(1, 0, itoa(points, tmp, 10));)
			DBG(sb_display_showStringSmall(2, 0, itoa(number_game_objects, tmp, 10));)
			DBG(sb_display_showStringSmall(3, 0, itoa(difficulties_passed, tmp, 10));)

			DBG(sb_display_showStringSmall(4, 0, "  ");)
			DBG(sb_display_showStringSmall(4, 0, itoa(last_enemy_spawned, tmp, 10));)

			DBG(if(spawn_blocked) {sb_display_showStringSmall(0, 16, "x");})
			DBG(if(!spawn_blocked) {sb_display_showStringSmall(0, 16, "o");})

			if(points > 99) {
				points = 99;
			}
			sb_display_showString(0, 0, itoa(points, tmp, 10));

			world_has_moved = 0;
		}

		if(gamestate == MARIO_DEAD) {
			fill_screen_from_mid();
			return MARIO_DEAD;
		}

		if(gamestate == MARIO_WON) {
			fill_screen_from_mid();
			return MARIO_WON;
		}

		if(button0) {
			button0 = 0;
			DBG(sb_led_toggle(GREEN0);)

			if(!OBJECT_IS_JUMPING(mario->state)) {
				mario->state |= OBJECT_JUMPING;
				ticks_left_jumping = level_ticks_jumping;
			}

			if(gamestate == LEVEL_PAUSED) {
				return LEVEL_ABORT;
			}
		}

		if(button1) {
			button1 = 0;
			DBG(sb_led_toggle(BLUE0);)

			if(gamestate == LEVEL_ACTIVE
			   || gamestate == FLAG_ADDED
			   || gamestate == FLAG_FINAL) {
				pause_game(1);
			} else if(gamestate == LEVEL_PAUSED) {
				start_game();
			}
		}
	}
}

static void init_level(void) {

	cli();

	for(uint8_t i = 0; i < MAX_GAME_OBJECTS; i++) {
		game_objects[i].type = EMPTY;
	}

	number_game_objects = 0;
	mario = 0;

	ticks_left_jumping = 0;
	ticks_left_raging = 0;
	if(alarm != NULL) {
		sb_timer_cancelAlarm(alarm);
		alarm = NULL;
		sei();
		for(volatile uint8_t i = 0; i < 8; i++) {} // work off already triggered alarms
		cli();
	}

	tick = 0;
	button0 = 0;
	button1 = 0;
	gamestate = LEVEL_ACTIVE;
	old_gamestate = LEVEL_ACTIVE;
	world_has_moved = 0;
	current_difficulty = 0;
	level_difficulty = MIN_LEVEL_DIFFICULTY + current_level*2;
	if(level_difficulty > MAX_LEVEL_DIFFICULTY) {
		level_difficulty = MAX_LEVEL_DIFFICULTY;
	}
	difficulties_passed = 0;
	number_walls = 0;
	number_holes = 0;
	spawn_blocked = 0;
	smashed_mushroom = 0;
	mario_ticking = 0;
	if(MIN_TICKS_JUMPING + (current_level-1)*2 <= MAX_TICKS_JUMPING) {
		level_ticks_jumping = MAX_TICKS_JUMPING-(current_level-1)*2;
	} else {
		level_ticks_jumping = MIN_TICKS_JUMPING;
	}
	last_wall_spawned = 0;
	last_enemy_spawned = TICKS_SPAWN_BLOCKED / 2 ;
	last_enemy = UNKNOWN_ERROR;
	points = 0;
	ticks_left_invincible = 0;

	add_mario();

	// randomly add walls
	for(uint8_t i = 0; i < DISPLAY_WIDTH; i += 8) {
		if(rand() % 2 == 0) {
			add_wall(i, 0);
		}
	}
}

static void start_game(void) {

	sb_display_fillScreen(NULL);

	cli();
	start_timer();
	gamestate = old_gamestate;
	sei();

	// simple debouncing
	sb_timer_delay(DEBOUNCING_TIME);
	button1 = 0;
}

static void start_timer(void) {
	uint8_t sreg = SAVE_SREG;

	cli();
	if(alarm != NULL) {
		sb_timer_cancelAlarm(alarm);
		alarm = NULL;
	}

	alarm = sb_timer_setAlarm(game_tick, 1000/TICKS_PER_SECOND, 1000/TICKS_PER_SECOND);

	RESTORE_SREG(sreg);
}

static void stop_timer(void) {
	uint8_t sreg = SAVE_SREG;

	cli();
	if(alarm != NULL) {
		sb_timer_cancelAlarm(alarm);
		alarm = NULL;
		tick = 0;
	}

	RESTORE_SREG(sreg);
}

static void pause_game(uint8_t show_msg) {

	cli();
	stop_timer();
	old_gamestate = gamestate;
	gamestate = LEVEL_PAUSED;

	sei();

	if(show_msg) {
		sb_display_fillScreen(NULL); // Remove in final version
		draw_rectangle(0, 0, 8, 128);
		sb_display_showStringWide(2, 20, "Game Paused");
		sb_display_showStringWide(4, 7, "Btn1 -> Resume");
		sb_display_showStringWide(5, 7, "Btn0 -> Menu");
	}

	// simple debouncing
	sb_timer_delay(DEBOUNCING_TIME);
	button1 = 0;
}

static void balance_difficulty(void) {

	if(gamestate != LEVEL_ACTIVE) {
		return;
	}

	if(last_enemy_spawned >= TICKS_SPAWN_BLOCKED
	   && !spawn_blocked
	   && current_difficulty < MAX_DIFFICULTY) {

		if(difficulties_passed >= level_difficulty) {
			add_flag();
			return;
		}

		if(rand() % (2+current_difficulty) == 0) {

			drawable_type enemy_pool[MAX_ENEMY_POOL_SIZE];
			uint8_t size = 0;

			enemy_pool[size++] = EVIL_MUSHROOM;

			if(((last_enemy != PIPE
			     && last_enemy != HOLE)
			    || last_enemy_spawned > TICKS_SPAWN_BLOCKED_PIPE)
			   && number_pipes < MAX_PIPES) {

				enemy_pool[size++] = PIPE;
			}

			if(number_holes < MAX_HOLES) {
				enemy_pool[size++] = HOLE;
			}

			if(size > MAX_ENEMY_POOL_SIZE) {
				shutdown(UNKNOWN_ERROR);
			}

			if(size > 0) {
				last_enemy_spawned = 0;
				last_enemy = enemy_pool[rand() % size];

				uint8_t random_hole = 0;
				switch(last_enemy) {
				case EVIL_MUSHROOM:
					add_evil_mushroom();
					break;
				case PIPE:
					add_pipe();
					break;
				case HOLE:
					if(current_level >= 3) {
						random_hole = rand() % 2;
					}
					add_hole(random_hole);
					break;
				default:
					shutdown(INVALID_ENEMY);
				}
			}
		}
	}

	if(world_has_moved
	   && number_walls < MAX_WALLS
	   && last_wall_spawned >= TICKS_WALL_SPAWN_BLOCKED) {

		if(rand() % (1+number_walls/2) == 0) {
			last_wall_spawned = 0;
			if(rand() % 5 == 0) {
				add_wall(DISPLAY_WIDTH, 1);
			} else {
				add_wall(DISPLAY_WIDTH, 0);
			}
		}
	}
}

static void shutdown(shutdown_errorcode error) {

	sei();

	// indicate non-recoverable error
	for(uint8_t i = 0; i < 3; i++) {
		sb_led_setMask(0xff);
		_delay_ms(500);
		sb_led_setMask(0x0);
		_delay_ms(500);
	}
	sb_led_setMask(error);

	while(1);
}

static void game_tick(void) {
	tick = 1;
}

ISR(INT0_vect) {
	button0 = 1;
}

ISR(INT1_vect) {
	button1 = 1;
}

static void copy_drawable(drawable *src, drawable *dest) {
	memcpy(dest, src, sizeof(drawable));
}

// return value: number of available slots for game_objects
// deletes game_object at game_objects[MAX_GAME_OBJECTS-1]
// if no free entry available and forced to do
// return value: -1 on error
static int8_t add_game_object(drawable *d, uint8_t force) {

	if(number_game_objects >= MAX_GAME_OBJECTS) {
		if(force) {
			remove_game_object(game_objects + (MAX_GAME_OBJECTS - 1));
		} else {
			return -1;
		}
	}

	uint8_t i = 0;
	while(game_objects[i].type != EMPTY) {
		i++;
	}

	copy_drawable(d, game_objects+i);
	number_game_objects++;

	if(d->type == HOLE) {
		number_holes++;
	} else if (d->type == WALL
	           || d->type == SPECIAL_WALL) {
		number_walls++;
	} else if (d->type == PIPE) {
		number_pipes++;
	}

	current_difficulty += d->difficulty;

	return MAX_GAME_OBJECTS-number_game_objects;
}

// returns first occurence of a type of game object
// returns NULL if no object is found
static drawable *find_game_object(drawable_type type) {

	for(uint8_t i = 0; i < MAX_GAME_OBJECTS; i++) {
		if(game_objects[i].type == type) {
			return &(game_objects[i]);
		}
	}

	return NULL;
}

static void remove_game_object(drawable *d) {

	if(d < game_objects
	   || d >= game_objects+MAX_GAME_OBJECTS) {

		return;
	}

	if(d->type == HOLE) {
		number_holes--;
	} else if (d->type == WALL
	           || d->type == SPECIAL_WALL) {
		number_walls--;
	} else if (d->type == PIPE) {
		number_pipes--;
	}

	d->type = EMPTY;

	number_game_objects--;
	current_difficulty -= d->difficulty;
	difficulties_passed += d->difficulty;
}

static void draw_rectangle(uint8_t page, uint8_t column, uint8_t height, uint8_t width) {

	if(page+height > 8
	   || column+width > 128) {

		sb_led_setMask(0xff);
		while(1);
	}

	uint8_t content[width*height];
	memset(content, 0, sizeof(content));

	for(uint8_t i = 0; i < width; i++) {
		content[i] |= 0x03;
		content[(height-1)*width+i] |= 0xC0;
	}

	for(uint8_t i = 0; i < height; i++) {
		content[i*width] |= 0xff;
		content[i*width+1] |= 0xff;
		content[i*width+width-1] |= 0xff;
		content[i*width+width-2] |= 0xff;
	}

	sb_display_drawBitmap(page, column, height, width, content);
}

static void fill_screen_from_mid(void) {

	uint8_t content[DISPLAY_WIDTH*DISPLAY_HEIGHT];
	memset(content, 0xff, sizeof(content));
	for(uint8_t i = 1; i < 5; i++) {
		sb_display_drawBitmap(4-i, 64-i*16, i*2, i*32, content);
		sb_timer_delay(75);
	}
}

static void draw_ground(void) {
	uint16_t ground_drawn = 0;

	for(uint8_t i = 0; i < MAX_GAME_OBJECTS; i++) {

		uint8_t fill_ground = 0;

		if(game_objects[i].type == PIPE) {
			sb_display_drawBitmap(game_objects[i].page,
				game_objects[i].column,
				1,
				game_objects[i].width,
				pipe_head);
			sb_display_drawBitmap(game_objects[i].page+1,
				game_objects[i].column,
				1,
				game_objects[i].width,
				pipe_body);
			sb_display_drawBitmap(game_objects[i].page+2,
				game_objects[i].column,
				1,
				game_objects[i].width,
				pipe_body);
			if(HAS_FLOWER(game_objects[i].state)) {
				sb_display_drawBitmap(game_objects[i].page-2,
					game_objects[i].column,
					2,
					game_objects[i].width,
					flower);
			}

			fill_ground = 1;
		}

		if(game_objects[i].type == HOLE) {
			draw_drawable(game_objects+i, NULL);
			fill_ground = 1;
		}

		if(fill_ground) {

			uint8_t draw_ground_left = game_objects[i].column % 8;
			uint8_t draw_ground_right = (game_objects[i].column+game_objects[i].width) % 8;
			sb_display_drawBitmap(6, game_objects[i].column-draw_ground_left, 1, draw_ground_left, ground);
			sb_display_drawBitmap(7, game_objects[i].column-draw_ground_left, 1, draw_ground_left, ground);
			sb_display_drawBitmap(6, game_objects[i].column+game_objects[i].width, 1, 8-draw_ground_right, ground+draw_ground_right);
			sb_display_drawBitmap(7, game_objects[i].column+game_objects[i].width, 1, 8-draw_ground_right, ground+draw_ground_right);

			uint8_t field = game_objects[i].column/8;
			uint8_t field_end = (game_objects[i].column+game_objects[i].width)/8;

			for(uint8_t i = field; i <= field_end; i++) {
				ground_drawn |= (1 << i);
			}
		}
	}

	for(uint8_t i = 0; i < 128; i += 8) {
		if(!(ground_drawn & (1 << (i/8)))) {
			sb_display_drawBitmap(6, i, 1, 8, ground);
			sb_display_drawBitmap(7, i, 1, 8, ground);
		}
	}
}

static void draw_game_objects(void) {

	for(uint8_t i = 0; i < MAX_GAME_OBJECTS; i++) {
		if(game_objects[i].type != EMPTY) {
			game_objects[i].draw(game_objects+i);
		}
	}
}

static void add_mario(void) {
	drawable m;

	if(mario != 0) {
		return;
	}

	m.type = MARIO;
	m.page = 4;
	m.column = 32;
	m.width = 16;
	m.height = 2;
	m.collision_margin = 1;
	m.draw = draw_mario;
	m.state = OBJECT_DIRECTION_RIGHT;
	m.extra = 0;
	m.difficulty = 0;

	add_game_object(&m, 0);

	mario = find_game_object(MARIO);
	if(mario == NULL) {
		shutdown(NO_MARIO);
	}
}

static void add_evil_mushroom(void) {
	drawable evil;

	evil.type = EVIL_MUSHROOM;
	evil.page = 4;
	evil.column = DISPLAY_WIDTH-16;
	evil.width = 16;
	evil.height = 2;
	evil.collision_margin = 1;
	evil.draw = draw_evil_mushroom;
	evil.state = OBJECT_MOVING;
	evil.extra = 0;
	evil.difficulty = 1;

	add_game_object(&evil, 0);
}

static void add_wall(uint8_t column, uint8_t special) {
	drawable wall;

	if(special) {
		wall.type = SPECIAL_WALL;
		wall.page = 1;
	} else {
		wall.type = WALL;
		if(rand() % 2 == 0) {
			wall.page = 1;
		} else {
			wall.page = 0;
		}
	}

	wall.column = column;
	wall.width = 8;
	wall.height = 1;
	wall.collision_margin = 0;
	wall.draw = draw_wall;
	wall.state = 0;
	wall.extra = 0;
	wall.difficulty = 0;

	add_game_object(&wall, 0);
}

static void add_pipe(void) {
	drawable pipe;

	pipe.type = PIPE;
	pipe.page = DISPLAY_HEIGHT-3;
	pipe.column = DISPLAY_WIDTH-8;
	pipe.width = 10;
	pipe.height = 3;
	pipe.collision_margin = 1;
	pipe.draw = draw_pipe;
	pipe.state = HAS_FLOWER;
	pipe.extra = rand() % FLOWER_TICKS;
	if(pipe.extra == 0) {
		pipe.extra = 1;
	}
	pipe.difficulty = 1;

	add_game_object(&pipe, 0);
}

static void add_hole(uint8_t random_start) {
	drawable hole;

	hole.type = HOLE;
	hole.page = 6;
	if(random_start) {
		hole.column = DISPLAY_WIDTH/2 + (rand() % (DISPLAY_WIDTH/2));

		for(uint8_t i = 0; i < MAX_GAME_OBJECTS; i++) {
			if(game_objects[i].type == PIPE
			   && collide4(game_objects+i, 6, 2, hole.column, 18)) {

				hole.column = DISPLAY_WIDTH;
				break;
			}
		}
	} else {
		hole.column = DISPLAY_WIDTH;
	}

	hole.width = 0;
	hole.height = 2;
	hole.collision_margin = 0;
	hole.draw = draw_hole;
	hole.state = 0;
	hole.extra = 0;
	hole.difficulty = 1;

	add_game_object(&hole, 0);
}

static void add_coin(uint8_t column) {
	drawable coin;

	coin.type = COIN;
	coin.page = 1;
	coin.column = column;
	coin.width = 8;
	coin.height = 1;
	coin.collision_margin = 0;
	coin.draw = draw_coin;
	coin.state = 0;
	coin.extra = 4;
	coin.difficulty = 0;

	add_game_object(&coin, 0);
}

static void add_flag(void) {
	drawable flag;

	flag.type = FLAG;
	flag.page = 2;
	flag.column = DISPLAY_WIDTH-8-4;
	flag.width = 8+3;
	flag.height = 4;
	flag.collision_margin = 0;
	flag.draw = draw_flag;
	flag.state = 0;
	flag.extra = 0;
	flag.difficulty = 0;

	gamestate = FLAG_ADDED;
	add_game_object(&flag, 1);
}

static void draw_mario(drawable *d) {

	mario->state &= ~(OBJECT_FALLING);
	if(world_has_moved) {
		mario_ticking++;
	}

	if(OBJECT_IS_JUMPING(d->state)) {
		if(ticks_left_jumping == level_ticks_jumping) {
			draw_drawable(d, NULL);
			d->page = 4-JUMPING_HEIGHT;

			for(uint8_t i = 0; i < MAX_GAME_OBJECTS; i++) {
				if(game_objects[i].type == SPECIAL_WALL
				   && !SPECIAL_WALL_IS_USED(game_objects[i].state)) {

					if(d->column+d->width >= game_objects[i].column
					   && d->column <= game_objects[i].column+game_objects[i].width) {
						game_objects[i].state |= SPECIAL_WALL_USED;

						if(rand() % 20 == 0) {
							mario_rage(game_objects+i);
						} else if(OBJECT_IS_SMALL(d->state)
						          && rand() % 5 == 0) {

							mario_grow(game_objects+i);
						} else {
							points++;
							add_coin(game_objects[i].column);
						}
						break;
					}
				}
			}
		}

		ticks_left_jumping--;

		if(ticks_left_jumping == 0) {
			d->state &= ~(OBJECT_JUMPING);
			d->state |= (OBJECT_FALLING);
			draw_drawable(d, NULL);
			uint8_t standing_on_pipe = 0;
			for(uint8_t i = 0; i < MAX_GAME_OBJECTS; i++) {
				if(game_objects[i].type == PIPE) {
					if(d->column+d->width >= game_objects[i].column
					   && d->column <= game_objects[i].column+game_objects[i].width) {

						d->page = 3;
						standing_on_pipe = 1;
						break;
					}
				}
			}

			if(!standing_on_pipe) {
				d->page = 4;
			}
		}
	} else { // not jumping
		if(mario->page != 4) {
			uint8_t standing_on_pipe = 0;
			for(uint8_t i = 0; i < MAX_GAME_OBJECTS; i++) {
				if(game_objects[i].type == PIPE) {
					if(d->column+d->width >= game_objects[i].column
					   && d->column <= game_objects[i].column+game_objects[i].width) {

						standing_on_pipe = 1;
						break;
					}
				}
			}

			if(!standing_on_pipe) {
				draw_drawable(d, NULL);
				d->page = 4;
			}
		}
	}

	if(smashed_mushroom) {
		smashed_mushroom--;
	}

	if(OBJECT_IS_FALLING(mario->state)) {
		for(uint8_t i = 0; i < MAX_GAME_OBJECTS; i++) {
			if(game_objects[i].type == EVIL_MUSHROOM) {
				if(collide(mario, game_objects+i)) {
					smashed_mushroom = TICKS_SMASHED_MUSHROOM;
					draw_drawable(game_objects+i, NULL);
					remove_game_object(game_objects+i);
					points++;
				}
			}
		}
	}

	if(ticks_left_raging > 0) {
		ticks_left_raging--;
	}

	if(OBJECT_IS_RAGING(mario->state)
		&& ticks_left_raging == 0) {

		mario->state = mario->state & ~(OBJECT_RAGING);
	}

	if(ticks_left_invincible > 0) {
		ticks_left_invincible--;
	}

	uint16_t poti = sb_adc_read(POTI);
	uint8_t direction = poti/(1023/3);

	if(direction == 0) {
		d->state |= OBJECT_DIRECTION_RIGHT;
		d->state |= OBJECT_MOVING;
	} else if(direction == 1) {
		d->state &= ~(OBJECT_MOVING);
	} else {
		d->state &= ~(OBJECT_DIRECTION_RIGHT);
		d->state |= OBJECT_MOVING;
	}

	const uint8_t *content;

	if(OBJECT_IS_MOVING(d->state)) {
		if(OBJECT_DIRECTION_IS_RIGHT(d->state)) {
			if(DISPLAY_WIDTH-d->column <= SHIFTING_WINDOW) {
			} else {
				d->column = (d->column + SPEED) % DISPLAY_WIDTH;
			}
		} else {
			if(d->column >= SPEED) {
				d->column = (d->column - SPEED) % DISPLAY_WIDTH;
			}
		}
	}

	switch((d->state) & (OBJECT_JUMPING
	                     | OBJECT_DIRECTION_RIGHT
	                     | OBJECT_SMALL
	                     | OBJECT_MOVING)) {

	case OBJECT_JUMPING | OBJECT_DIRECTION_RIGHT | OBJECT_SMALL:
	case OBJECT_JUMPING | OBJECT_DIRECTION_RIGHT | OBJECT_SMALL | OBJECT_MOVING:
		content = small_mario_right_jump;
		break;
	case OBJECT_JUMPING | OBJECT_DIRECTION_RIGHT:
	case OBJECT_JUMPING | OBJECT_DIRECTION_RIGHT | OBJECT_MOVING:
		content = mario_right_jump;
		break;
	case OBJECT_DIRECTION_RIGHT | OBJECT_SMALL:
	case OBJECT_DIRECTION_RIGHT | OBJECT_SMALL | OBJECT_MOVING:
		if((mario_ticking % 2) == 0) {
			content = small_mario_right;
		} else {
			content = small_mario_right_tick;
		}
		break;
	case OBJECT_DIRECTION_RIGHT:
	case OBJECT_DIRECTION_RIGHT | OBJECT_MOVING:
		if((mario_ticking % 2) == 0) {
			content = mario_right;
		} else {
			content = mario_right_tick;
		}
		break;
	case OBJECT_JUMPING | OBJECT_SMALL:
	case OBJECT_JUMPING | OBJECT_SMALL | OBJECT_MOVING:
		content = small_mario_left_jump;
		break;
	case OBJECT_JUMPING:
	case OBJECT_JUMPING | OBJECT_MOVING:
		content = mario_left_jump;
		break;
	case OBJECT_SMALL:
	case OBJECT_SMALL | OBJECT_MOVING:
		if((mario_ticking % 2) == 0) {
			content = small_mario_left;
		} else {
			content = small_mario_left_tick;
		}
		break;
	case OBJECT_MOVING:
		if((mario_ticking % 2) == 0) {
			content = mario_left;
		} else {
			content = mario_left_tick;
		}
		break;
	default:
		content = mario_left;
		break;
	};

	if((OBJECT_IS_RAGING(mario->state))
		&& ((ticks_left_raging % 5) == 0)) {

		content = NULL;
	}

	uint8_t c[sizeof(mario_left)];
	if(smashed_mushroom) {
		memcpy_P(c, content, sizeof(mario_left));
		for(uint8_t i = 1; i < mario->width-1; i++) {
			uint8_t idx = mario->width*1+i;

			if(content != NULL) {
				c[idx] = c[idx] | 0xE0;
			} else {
				c[idx] = 0xE0;
			}
		}
		content = c;
	}

	if(smashed_mushroom) {
		sb_display_drawBitmap(d->page, d->column, 2, 16, content);
	} else {
		sb_display_drawBitmapFromFlash(d->page, d->column, 2, 16, content);
	}
}

static uint8_t collide(drawable *a, drawable *b) {
	return _collide(a->page, a->height, a->column, a->width, a->collision_margin,
	                b->page, b->height, b->column, b->width, b->collision_margin);
}

static uint8_t collide2(drawable *a, drawable *b) {
	return _collide(a->page, a->height, a->column, a->width, 0,
	                b->page, b->height, b->column, b->width, 0);
}

static uint8_t collide3(drawable *a, uint8_t b_page, uint8_t b_height,
                        uint8_t b_column, uint8_t b_width) {
	return _collide(a->page, a->height, a->column, a->width, a->collision_margin,
	                b_page, b_height, b_column, b_width, 0);
}

static uint8_t collide4(drawable *a, uint8_t b_page, uint8_t b_height,
                        uint8_t b_column, uint8_t b_width) {

	return _collide(a->page, a->height, a->column, a->width, 0,
	                b_page, b_height, b_column, b_width, 0);
}

static uint8_t _collide(uint8_t a_page, uint8_t a_height, uint8_t a_column, uint8_t a_width, uint8_t a_collision_margin,
                        uint8_t b_page, uint8_t b_height, uint8_t b_column, uint8_t b_width, uint8_t b_collision_margin) {

//              ----------------
//              |              |
//     height_a |              |
//              |          ----|---------------
//              ----------------              |
//               width_a   |                  |
//                         |                  | height_b
//                         |                  |
//                         |                  |
//                         --------------------
//                               width_b

	// overlap in height
	if(a_page+a_height >= b_page && a_page < b_page+b_height) { // obere kante von a über unterer kante von b und untere kante von a unter oberer kante von b
		if(a_column+a_width-a_collision_margin >= b_column+b_collision_margin && a_column+a_collision_margin <= b_column+b_width-b_collision_margin) {
			return 1;
		}
	}

	return 0;
}

static void draw_evil_mushroom(drawable *d) {

	// kollisionsabfrage
	if(collide(d, mario)) {
		mario_reduce_live();
	}

	// evil mushroom leaves on the left side
	if(d->column == 0) {
		draw_drawable(d, NULL);
		remove_game_object(d);
		return;
	}

	if(OBJECT_IS_FALLING(d->state)) {
		if(world_has_moved) {
			sb_display_drawBitmap(d->page, d->column, 1, d->width, NULL);
			d->page++;

			if(d->page > DISPLAY_HEIGHT) {
				draw_drawable(d, NULL);
				remove_game_object(d);
				return;
			}

			if(d->column == 0) {
				draw_drawable(d, NULL);
				remove_game_object(d);
				return;
			}

			sb_display_drawBitmap(d->page, d->column, 2, 16, evil_mushroom);
			return;
		}
	}

	// evil mushroom leaves on the right side
	if(d->column > DISPLAY_WIDTH-d->width) {
		d->column = DISPLAY_WIDTH-d->width;
		draw_drawable(d, NULL);
		remove_game_object(d);
		return;
	}

	// test for pipes and holes
	for(uint8_t i = 0; i < MAX_GAME_OBJECTS; i++) {
		if(game_objects[i].type == HOLE) {
			if(d->column >= game_objects[i].column
			   && d->column+d->width <= game_objects[i].column+game_objects[i].width) {

				d->state |= OBJECT_FALLING;
			}

		}

		if(game_objects[i].type == PIPE
		   && collide2(d, game_objects + i)) {

			// Toggle funktioniert hier nicht, weil je nach Reihenfolge
			// erst die Pipe zieht und dann Evil Mushroom und Pipe2 Pixel
			// ineinandern sein können und dann dauerhaft getoggelt wird
			if(d->column - game_objects[i].column <= 2*SPEED) {
				d->state &= ~(OBJECT_DIRECTION_RIGHT);
			} else {
				d->state |= (OBJECT_DIRECTION_RIGHT);
			}
		}
	}

	if(OBJECT_DIRECTION_IS_RIGHT(d->state)) {
		if(d->column + SPEED >= DISPLAY_WIDTH) {
			d->column = DISPLAY_WIDTH;
		} else {
			d->column = d->column + SPEED;
		}
	} else {
		if(d->column < SPEED) {
			d->column = 0;
		} else {
			d->column = d->column - SPEED;
		}
	}

	// update spawn blocking state
	if(d->column > DISPLAY_WIDTH-SPAWN_SIZE) {
		spawn_blocked = 1;
	}

	sb_display_drawBitmap(d->page, d->column, 2, 16, evil_mushroom);
}

static void draw_wall(drawable *d) {
	if(world_has_moved) {
		d->column -= SPEED;
	}

	if(d->column == 0) {
		if(d->page != 0) {
			sb_display_drawBitmap(d->page, d->column, 2, 16, NULL);
		}
		remove_game_object(d);
		return;
	}

	// don't draw last rectangle because of points
	if(d->page == 0 && d->column < POINTS_DISPLAY_WIDTH) {
		if(d->column == POINTS_DISPLAY_WIDTH-1) {
			sb_display_drawBitmap(d->page, d->column+SPEED, 1, 8, NULL);
			draw_drawable(d, NULL);
		}
		return;
	}

	sb_display_drawBitmap(d->page, d->column+SPEED, 1, 8, NULL);
	if(d->type == WALL) {
		sb_display_drawBitmap(d->page, d->column, 1, 8, wall);
	} else {
		if(SPECIAL_WALL_IS_USED(d->state)) {
			sb_display_drawBitmap(d->page, d->column, 1, 8, wall_empty);
		} else {
			sb_display_drawBitmap(d->page, d->column, 1, 8, wall_special);
		}
	}
}

static void draw_pipe(drawable *d) {
	if(world_has_moved) {
		d->column -= SPEED;
	}

	if(d->column == 0) {
		draw_drawable(d, NULL);
		if(HAS_FLOWER(d->state)) {
			sb_display_drawBitmap(d->page-2, d->column, 2, d->width, NULL); //clear flower
		}
		remove_game_object(d);
		return;
	}

	if(HAS_FLOWER(d->state)
	   && collide3(mario, d->page-2, 2, d->column+2, 4)) {

		mario_reduce_live();
	}

	// ticks of flower presence/absence
	d->extra -= 1;

	if(d->extra == 0) {
		if(HAS_FLOWER(d->state)) {
			sb_display_drawBitmap(d->page-2, d->column, 2, d->width, NULL); //clear flower
			d->extra = FLOWER_TICKS;
		} else {
			d->extra = (FLOWER_TICKS / 2 + (current_level-1)*2);
			if(d->extra >= FLOWER_TICKS) {
				d->extra = FLOWER_TICKS;
			}
		}

		d->state ^= HAS_FLOWER;
	}

	if(collide2(d, mario)) {
		if(OBJECT_IS_MOVING(mario->state)) {
			if(OBJECT_DIRECTION_IS_RIGHT(mario->state)) {
				mario->column -= 2;
			} else {
				mario->column += 1;
			}
		} else {
			mario->column -= 1;
		}

		if(mario->column == 0) {
			mario_die(1);
		}
	}
}

static void draw_hole(drawable *d) {
	if(world_has_moved) {

		if(d->width < 18) {
			d->width += SPEED;
		}

		d->column -= SPEED;
	}

	if(d->column == 0) {
		remove_game_object(d);
		return;
	}

	if(!OBJECT_IS_JUMPING(mario->state)
	   && mario->column+mario->collision_margin > d->column
	   && mario->column+mario->width-mario->collision_margin < d->column+d->width) {

		mario_die(0);
		return;
	}
}

static void draw_coin(drawable *d) {
	d->extra--;

	if(world_has_moved) {
		sb_display_draw(d->page, d->column+d->width-1, 0x00);
		d->column -= SPEED;
	}

	if(d->extra == 0) {
		draw_drawable(d, NULL);

		if(d->page == 0) {
			remove_game_object(d);
			return;
		}

		d->extra = 4;
		d->page--;
	}

	sb_display_drawBitmap(d->page, d->column+d->width+1, d->height, 1, NULL);
	draw_drawable(d, coin);
}

static void draw_flag(drawable *d) {

	if(world_has_moved) {
		sb_display_drawBitmap(2, d->column+2, d->height, 1, NULL);
		sb_display_drawBitmap(d->page, d->column+3, 1, 8, NULL);

		d->column -= SPEED;
	}

	uint8_t content[d->height*3];
	for(uint8_t i = 0; i < sizeof(content); i++) {
		content[i] = 0xff;
	}
	sb_display_drawBitmap(2, d->column, d->height, 3, content);
	sb_display_drawBitmap(d->page, d->column+3, 1, 8, flag);

	if(d->column <= DISPLAY_WIDTH - SPAWN_SIZE/2 - 10) {
		gamestate = FLAG_FINAL;
	}

	if(collide(mario, d)) {
		drop_flag(d);
	}
}

static void drop_flag(drawable *d) {
	while(d->page < 5) {
		sb_display_drawBitmap(d->page, d->column+3, 1, 8, NULL);
		d->page++;
		sb_display_drawBitmap(d->page, d->column+3, 1, 8, flag);
		sb_timer_delay(500);
	}

	pause_game(0);
	gamestate = MARIO_WON;
}

static void draw_drawable(drawable *d, const uint8_t *bitmap) {
	sb_display_drawBitmap(d->page, d->column, d->height, d->width, bitmap);
}

static void mario_reduce_live(void) {

	DBG(return;)

	if(ticks_left_invincible > 0) {
		return;
	}

	if(OBJECT_IS_RAGING(mario->state)) {
		return;
	}

	if(OBJECT_IS_SMALL(mario->state)) {
		mario_die(1);
	} else {
		mario_shrink();
		ticks_left_invincible = TICKS_INVINCIBLE;
	}
}

static void mario_die(uint8_t animated) {
	pause_game(0);
	sb_display_drawBitmap(mario->page, mario->column, mario->height, mario->width, NULL);

	const uint8_t *content;
	if(OBJECT_IS_SMALL(mario->state)) {
		if(OBJECT_DIRECTION_IS_RIGHT(mario->state)) {
			content = small_mario_right_jump;
		} else {
			content = small_mario_left_jump;
		}
	} else {
		if(OBJECT_DIRECTION_IS_RIGHT(mario->state)) {
			content = mario_right_jump;
		} else {
			content = mario_left_jump;
		}
	}

	if(animated) {
		while(mario->page >= 2) {
			mario->page--;
			sb_display_drawBitmapFromFlash(mario->page, mario->column, mario->height, mario->width, content);
			_delay_ms(200);
			sb_display_drawBitmapFromFlash(mario->page, mario->column, mario->height, mario->width, NULL);
			draw_ground();
		}
	}

	while(mario->page != DISPLAY_HEIGHT) {
		mario->page++;
		sb_display_drawBitmapFromFlash(mario->page, mario->column, mario->height, mario->width, content);
		_delay_ms(200);
		sb_display_drawBitmapFromFlash(mario->page, mario->column, mario->height, mario->width, NULL);
		draw_ground();
	}

	mario = 0;
	gamestate = MARIO_DEAD;
}

static void mario_shrink(void) {
	pause_game(0);

	for(uint8_t i = 0; i < 4; i++) {
		if(OBJECT_DIRECTION_IS_RIGHT(mario->state)) {
			sb_display_drawBitmapFromFlash(mario->page, mario->column, mario->height, mario->width, mario_right);
			_delay_ms(200);
			sb_display_drawBitmapFromFlash(mario->page, mario->column, mario->height, mario->width, small_mario_right);
			_delay_ms(200);
		} else {
			sb_display_drawBitmapFromFlash(mario->page, mario->column, mario->height, mario->width, mario_left);
			_delay_ms(200);
			sb_display_drawBitmapFromFlash(mario->page, mario->column, mario->height, mario->width, small_mario_left);
			_delay_ms(200);
		}
	}

	mario->state |= OBJECT_SMALL;

	start_game();
}

static void mario_grow(drawable *d) {
	pause_game(0);

	const uint8_t *content;
	const uint8_t *content_small;
	if(OBJECT_DIRECTION_IS_RIGHT(mario->state)) {
		content = mario_right_jump;
		content_small = small_mario_right_jump;
	} else {
		content = mario_left_jump;
		content_small = small_mario_left_jump;
	}

	uint8_t c[sizeof(mushroom)];
	for(uint8_t i = 0; i < sizeof(mushroom); i++) {
		c[i] = mushroom[i];
	}

	for(uint8_t i = 0; i < 4; i++) {
		sb_display_drawBitmapFromFlash(mario->page, mario->column, mario->height, mario->width, content);
		sb_display_drawBitmap(0, d->column, d->height, d->width, c);

		for(uint8_t j = 0; j < sizeof(mushroom); j++) {
			c[j] = c[j] >> 1;
		}

		_delay_ms(200);

		sb_display_drawBitmapFromFlash(mario->page, mario->column, mario->height, mario->width, content_small);
		sb_display_drawBitmap(0, d->column, d->height, d->width, c);

		for(uint8_t j = 0; j < sizeof(mushroom); j++) {
			c[j] = c[j] >> 1;
		}
		_delay_ms(200);
	}
	sb_display_drawBitmapFromFlash(mario->page, mario->column, mario->height, mario->width, content);

	mario->state &= ~(OBJECT_SMALL);

	points++;
	start_game();
}

static void mario_rage(drawable *d) {
	pause_game(0);

	const uint8_t *content;
	if(OBJECT_DIRECTION_IS_RIGHT(mario->state)) {
		if(OBJECT_IS_SMALL(mario->state)) {
			content = small_mario_right_jump;
		} else {
			content = mario_right_jump;
		}
	} else {
		if(OBJECT_IS_SMALL(mario->state)) {
			content = small_mario_left_jump;
		} else {
			content = mario_left_jump;
		}
	}

	uint8_t c[sizeof(star)];
	for(uint8_t i = 0; i < sizeof(star); i++) {
		c[i] = star[i];
	}

	for(uint8_t i = 0; i < 4; i++) {
		draw_drawable(mario, NULL);
		sb_display_drawBitmap(0, d->column, d->height, d->width, c);

		for(uint8_t j = 0; j < sizeof(star); j++) {
			c[j] = c[j] >> 1;
		}

		_delay_ms(200);

		sb_display_drawBitmapFromFlash(mario->page, mario->column, 2, 16, content);
		sb_display_drawBitmap(0, d->column, d->height, d->width, c);

		for(uint8_t j = 0; j < sizeof(mushroom); j++) {
			c[j] = c[j] >> 1;
		}
		_delay_ms(200);
	}
	sb_display_drawBitmapFromFlash(mario->page, mario->column, 2, 16, content);

	mario->state = mario->state | (OBJECT_RAGING);
	ticks_left_raging = TICKS_RAGING;

	points++;
	start_game();
}

