#!/usr/bin/python3

import argparse
import os

from PIL import Image

DEFAULT_THRESHOLD = 10

def check_threshold(string):
	value = int(string)
	if value < 0 or value > 255:
		raise argparse.ArgumentTypeError("threshold must be between 0 and 255")

	return value

def main():

	#argument parsing
	parser = argparse.ArgumentParser()
	parser.add_argument("image", help="image to be converted")
	parser.add_argument("-t", "--threshold", help="threshold when a pixel is considered (0-255)", type=check_threshold, default=DEFAULT_THRESHOLD)
	parser.add_argument("-s", "--silent", help="show no metadata and picture", action="store_true")
	parser.add_argument("-l", "--light-terminal", help="assume light background color for terminal", action="store_true")
	parser.add_argument("-n", "--name", help="name for resulting array", default="")
	parser.add_argument("-a", "--flash", help="use flash memory instead of sram", default="", action="store_const", const=" __flash")
	args = parser.parse_args()

	im = Image.open(args.image, mode='r')

	if not args.silent:
		print(im)
		print(im.info)

	white = "\033[107m  "
	black = "\033[40m  "
	default = "\033[49m"

	# use file name (without filetype extension) as array name
	if args.name == "":
		args.name = os.path.basename(args.image.strip().lower()).split(".")[0]

	if not args.silent:
		print(im.size)
		for y in range(0, im.size[1]):
			for x in range(0, im.width):
				value = im.getpixel((x,y))
				gray_value = 0
				if isinstance(value, tuple):
					gray_value = (value[0]+value[1]+value[2])/3
				else:
					gray_value = value
				if(gray_value > args.threshold):
					if args.light_terminal:
						print(black, end='')
					else:
						print(white, end='')
				else:
					if args.light_terminal:
						print(white, end='')
					else:
						print(black, end='')
			print(default)

	number_pages = (int) (im.height/8)
	if im.height % 8 != 0:
		number_pages += 1
	print("static%s const uint8_t %s[%d] = {" % (args.flash, args.name, number_pages * im.width))

	for i in range(0, number_pages):
		print("\t", end='')
		for j in range(0, im.width):
			current_byte = 0
			for k in range(0, 8):
				if i*8+k < im.height: 
					value = im.getpixel((j,i*8+k))
					gray_value = 0
					if isinstance(value, tuple):
						gray_value = (value[0]+value[1]+value[2])/3
					else:
						gray_value = value

					if gray_value <= args.threshold:
						current_byte = current_byte | (1 << k)
			print("0x%02x" % current_byte, end='')
			if j != im.width-1 or i != number_pages-1:
				print(", ", end='')
			else:
				print(" };", end='')
		print("")

if __name__ == "__main__":
	main()

