#include <stddef.h>
#include <stdbool.h>
#include <avr/io.h>
#include <avr/interrupt.h>
#include <avr/pgmspace.h>
#include <util/delay.h>

#include "led.h"
#include "7seg.h"
#include "console.h"
#include "display.h"

#ifndef THRESHOLD
// Decrease if you are a pro, increase if you are a beginner
#define THRESHOLD 60
#endif

_Static_assert(THRESHOLD > 0, "Threshold cannot be below 1");
_Static_assert(THRESHOLD < UINT16_MAX / 4, "Threshold is too big");

// morse tree as binary heap
static const __flash char morsetree[] = {
	'\0',
	'e', 't',
	'i', 'a', 'n', 'm',
	's', 'u', 'r', 'w', 'd', 'k', 'g', 'o',
	'h', 'v', 'f', '\x81', 'l', '\x84', 'p', 'j', 'b', 'x', 'c', 'y', 'z', 'q', '\x94', '\xb0',
	'5', '4', '\xFB', '3', '\x82', '\0', 'd', '2', '&', '\x8A', '+', '\0', '\xb1', '\x85', '\x8C', '1', '6', '=', '/', '\0', '\x80', '\x13', '(', '\0', '7', '\0', 'g', '\xA4', '8', '\0', '9', '0'
#ifdef MORSE_PUNCTUATION
	, '\0', '\0', '\0', '\0', '\0', '\xB3', '\0', '\0', '\0', '\0', '\0', '\0', '?', '_', '\0', '\0', '\0', '\0', '"', '\0', '\0', '.', '\0', '\0', '\0', '\0', '@', '\0', '\0', '\0', '\'', '\0', '\0', '-', '\0', '\0', '\0', '\0', '\0', '\0', '\0', '\0', ';', '!', '\0', ')', '\0', '\0', '\0', '\0', '\0', ',', '\0', '\0', '\0', '\0', ':', '\0', '\0', '\0', '\0', '\0', '\0', '\0'
#endif
};


static const uint8_t morsetreelen =
#ifdef MORSE_PUNCTUATION
	(1 << 7)
#else
	(1 << 6)
#endif
;

static bool display = false;

inline static bool hasSignal(){
	const uint8_t samples = 10;
	uint8_t active = 0;
	for (uint8_t i = 0; i < samples ; i++){
		active += (PIND & (1 << PD2)) ? 0 : 1;
		_delay_ms(1);
	}
	return active >= samples / 2;
}

static char buffer[6][17] = {
	"                ",
	" \xC9\xCD\xCD\xCD\xCD\xCD\xCD\xCD\xCD\xCD\xCD\xCD\xCD\xBB ",
	" \xBA Morse-Code \xBA ",
	" \xBA bersetzer \xBA ",
	" \xC8\xCD\xCD\xCD\xCD\xCD\xCD\xCD\xCD\xCD\xCD\xCD\xCD\xBC ",
	"                ",
};

static inline void writeOut(const char *str){
	static uint8_t line = 5;
	static uint8_t position = 0;

	for (uint8_t i = 0; str[i] != '\0'; i++){
		buffer[line][position] = str[i];
		if (position < 15)
			sb_display_showString(5, 8*position++, &str[i]);
		else { // new line
			position = 0;
			line = (line + 1) % 6;
			// Clear line buffer (with spaces so it overwrites the display)
			for (uint8_t j = 0; j < 16; j++)
				buffer[line][j] = ' ';
			// Scroll & display
			for (uint8_t j = 0; j < 6; j++)
				sb_display_showString(j, 0, buffer[(line + j + 1) % 6]);
		}
	}

}

static bool nextCharacter(char m){
	// OLED Display
	char str[3] = { m, '\0', '\0' };
	// 7 Seg
	char segSym[3] = { m, ' ', '\0' };

	// Special characters cannot be displayed on 7seg
	switch (m){
		case '\x80':
			segSym[0] = 'c';
			break;
		case '\x84':
		case '\x85':
			segSym[0] = 'a';
			break;
		case '\x82':
		case '\x8A':
			segSym[0] = 'e';
			break;
		case '\x8C':
			segSym[0] = 'i';
			break;
		case '\x94':
			segSym[0] = 'o';
			break;
		case '\x81':
			segSym[0] = 'u';
			break;
		case '\xA4':
			segSym[0] = 'n';
			break;
		case '\xB0':
			str[0] = segSym[0] = 'c';
			str[1] = segSym[1] = 'h';
			break;
		// Thorn
		case '\xB1':
			str[0] = segSym[0] = 't';
			str[1] = segSym[1] = 'h';
			break;
		// Understood
		case '\xFB':
			segSym[0] = 'v';
			segSym[1] = 'e';
			break;
		// End of contact
		case '\xB3':
			segSym[0] = 's';
			segSym[1] = 'k';
			break;
		// Attention
		case '\x13':
			segSym[0] = 'c';
			segSym[1] = 't';
			break;

		// Symbols cannot be displayed on 7 seg display
		case '&':
			segSym[0] = 'a';
			segSym[1] = 's';
			break;
		case '+':
			segSym[0] = 'a';
			segSym[1] = 'r';
			break;
		case '=':
			segSym[0] = 'b';
			segSym[1] = 't';
			break;
		case '/':
			segSym[0] = 'd';
			segSym[1] = 'n';
			break;
		case '(':
			segSym[0] = 'k';
			segSym[1] = 'n';
			break;

#ifdef MORSE_PUNCTUATION
		// Extendend symbols
		case '?':
			segSym[0] = 'i';
			segSym[1] = 'm';
			break;
		case '"':
			segSym[0] = 'a';
			segSym[1] = 'f';
			break;
		case '.':
			segSym[0] = 'a';
			segSym[1] = 'a';
			break;
		case '@':
			segSym[0] = 'a';
			segSym[1] = 'c';
			break;
		case '\'':
			segSym[0] = 'j';
			segSym[1] = 'n';
			break;
		case ';':
			segSym[0] = 'n';
			segSym[1] = 'n';
			break;
		case '!':
			segSym[0] = 'k';
			segSym[1] = 'w';
			break;
		case ')':
			segSym[0] = 'k';
			segSym[1] = 'k';
			break;
		case ',':
			segSym[0] = 'm';
			segSym[1] = 'i';
			break;
		case ':':
			segSym[0] = 'o';
			segSym[1] = 's';
			break;
#endif

		// Non-code
		case '\0':
			return false;
	}
	// 7 seg output
	sb_7seg_showString(segSym);

	// OLED display output
	writeOut(str);

	// Serial output
	sb_console_putString(str);

	return true;
}

void main(){
	DDRD &= ~(1 << PD2);
	PORTD |= (1 << PD2);
	if (sb_display_enable() == 0){
		display = true;
		for (uint8_t i = 0; i < 6; i++){
			sb_display_showString(i, 0, buffer[i]);
		}
		sb_display_showStringFromFlash(6, 0, PSTR("\xC4\xC4\xC4\xC4\xC4\xC4\xC4\xC4\xC4\xC4\xC4\xC4\xC4\xC4\xC4\xC4"));
		sb_display_showString(7, 0, buffer[0]);
	}
	sei();
	uint16_t durationPress = 0;
	uint16_t durationPause = 0;
	uint8_t symbol = 0;
	int8_t pos = 0;
	while (1){
		if (hasSignal()){
			// Clear line
			if (display && pos == 0){
				sb_display_showString(7, 16, "              ");
			}
			if (durationPause > 0){
				sb_led_off(YELLOW0);
				sb_led_off(GREEN0);
				durationPause = 0;
			}
			// signal press
			if (durationPress == 0){
				sb_led_on(BLUE1);
			}
			// DAH?
			if (durationPress < 0xffff && ++durationPress == THRESHOLD){
				sb_led_on(GREEN1);
				sb_led_on(YELLOW1);
			}
		} else {
			if (durationPress > 0){
				sb_led_off(BLUE1);
				sb_led_off(GREEN1);
				sb_led_off(YELLOW1);

				if (symbol > morsetreelen || symbol > 127)
					sb_led_on(RED1);
				// send DIT / DAH
				else
					symbol = 2 * symbol + (durationPress < THRESHOLD ? 1 : 2);
				if (display)
					sb_display_showString(7, 16 + (pos++)*8, durationPress < THRESHOLD ? "\xFA" : "\x2D");

				durationPress = 0;
			}

			if (durationPause < 0xffff){
				// next letter
				if (++durationPause == THRESHOLD){
					sb_led_on(YELLOW0);
					if (symbol > morsetreelen || !nextCharacter(morsetree[symbol])){
						// Invalid character
						sb_led_on(RED1);
						// no  space on invalid symbols
						durationPause = 0xffff;
					}
					else
						sb_led_off(RED1);
					pos = 0;
					symbol = 0;
				}
				// Space
				else if (durationPause == 5 * THRESHOLD){
					sb_led_off(YELLOW0);
					sb_led_on(GREEN0);
					nextCharacter(' ');
				}
			}
		}
	}
}
