#include <stdlib.h>
#include <stddef.h>
#include <avr/io.h>
#include <avr/interrupt.h>
#include <avr/sleep.h>

#include <display.h>
#include <7seg.h>
#include <timer.h>
#include <button.h>
#include <led.h>
#include <adc.h>

#define BINARY(X) __extension__ 0b ## X
#include "board.h"
#include "monsters.h"
#include "splash.h"

typedef void(* alarmcallback_t )(void);

//needs to be volatile since it is set in multiple ISRs
static volatile uint8_t event;
static volatile STATE state;

uint16_t pacman_speed;
uint16_t ghost_speed;
uint8_t ghost_vulnerable;

position player_pos;
position ghost_pos;

uint8_t points;
uint8_t dots_left;

// global variable to count the number of overflows
uint16_t ghost_of;
uint16_t pacmn_of;

uint16_t dots[BOARD_HEIGHT];

// since button1 is not debounced in hw the simples solution is to use the spiclib  functions
BUTTONSTATE button1_last_state;
BUTTONSTATE button0_last_state;

static void render_tile(position *, uint8_t *);
static void render_ghost();
static void render_pacman();
static void calc_bitmap(const __flash uint8_t[8], uint8_t *, uint8_t);
static void set_speed();
static void calc_next_ghost_dir();
static void timer0_init();
static void wait_time(uint16_t);

static void be_vulnerable(){
	if(ghost_pos.dead) return;
	sb_led_setMask(ghost_vulnerable);
	if(ghost_vulnerable){
		ghost_vulnerable = ghost_vulnerable >> 1;
		sb_timer_setAlarm((alarmcallback_t) be_vulnerable, 800 , 0);
	}
}


// TIMER0 overflow interrupt service routine
// called whenever TCNT0 overflows
ISR(TIMER0_OVF_vect) {
	BUTTONSTATE bs = sb_button_getState(BUTTON1);
	if(button1_last_state == PRESSED){
		button1_last_state = bs;
	} else if(button1_last_state != PRESSED && bs == PRESSED){
		button1_last_state = PRESSED;
		event = EVENT_CHANGE_DIR;
		if(state == START){
			state = HALT;
			return;
		}
		player_pos.next_dir -= 1;
		if(player_pos.next_dir < 0) player_pos.next_dir += 4;
		return;
	}
	bs = sb_button_getState(BUTTON0);
	if(button0_last_state == PRESSED){
		button0_last_state = bs;
	} else if(button0_last_state != PRESSED && bs == PRESSED){
		button0_last_state = PRESSED;
		event = EVENT_CHANGE_DIR;
		if(state == START){
			state = HALT;
			return;
		}
		player_pos.next_dir += 1;
		player_pos.next_dir %= 4;
		return;
	}

	if(state == START) return;

	// keep a track of number of overflows
	++pacmn_of;
	++ghost_of;

	if (pacmn_of >= pacman_speed){
		TCNT0 = 0;            // reset counter
		pacmn_of = 0;     // reset overflow counter
		event = EVENT_ACT_P_POS;
		return;
	}

	if(ghost_pos.dead) return;
	if (ghost_of >= ghost_speed){
		TCNT0 = 0;            // reset counter
		ghost_of = 0;     // reset overflow counter
		event = EVENT_ACT_G_POS;
	}

}

static void deaktivate_timers(){
	TIMSK0 = 0;
}

static void timer0_init(){
	//TIMER0
	// set up timer with prescaler = 256
	TCCR0B |= (1 << CS01);

	// initialize counter
	TCNT0 = 0;

	// enable overflow interrupt
	TIMSK0 |= (1 << TOIE0);
}

static inline uint8_t erase_dot(uint8_t px, uint8_t py){
	uint8_t ret = (dots[(py / TILE_SIZE) % BOARD_HEIGHT] >> ((px / TILE_SIZE)) % BOARD_WIDTH) & 1;
	dots[py / TILE_SIZE] &= ~(1 << (px / TILE_SIZE));
	return ret;
}

static void init(){
	button1_last_state = UNKNOWN;
	button0_last_state = UNKNOWN;
	dots_left = 0;
	ghost_of = 0;
	pacmn_of = 0;

	event = 0;
	ghost_vulnerable = 0;
	state = START;
	dots_left = 0;
	points = 0;
	sb_7seg_showNumber(points);
	set_speed();
	sb_display_fillScreen(NULL); // Clear display
	for(uint8_t y = 0; y < BOARD_HEIGHT; ++y){
		for(uint8_t x = 0; x < BOARD_WIDTH; ++x){
			dots[y] &= ~(1 << x);
			if(board[y] & (1 << x)){
				sb_display_drawBitmapFromFlash(y, x * TILE_SIZE, 1, TILE_SIZE, wall);
			} else {
				if(special_dots[y] & (1 << x) ) {
					sb_display_drawBitmapFromFlash(y, x * TILE_SIZE, 1, TILE_SIZE, dot_filled);
				} else {
					sb_display_drawBitmapFromFlash(y, x * TILE_SIZE, 1, TILE_SIZE, dot);
					dots[y] |= 1 << x;
				}
				++dots_left;
			}
		}
	}
	// Set Player
	player_pos.px = POS_PACMAN_X * TILE_SIZE;
	player_pos.py = POS_PACMAN_Y * TILE_SIZE;
	player_pos.next_dir = 0;
	player_pos.dir = RIGHT;
	player_pos.dead = false;
	render_pacman();
	erase_dot(player_pos.px, player_pos.py);
	--dots_left;
	// Set Ghost
	ghost_pos.px = POS_GHOST_X * TILE_SIZE;
	ghost_pos.py = POS_GHOST_Y * TILE_SIZE;
	ghost_pos.next_dir = 0;
	ghost_pos.dir = LEFT;
	ghost_pos.dead = false;
	render_ghost();
	calc_next_ghost_dir();
}

static void set_speed(){
	int16_t poti = sb_adc_read(POTI);
	float bn = 1.0 - (float)(poti)/(float)(MAX_DEV_VAL);

	pacman_speed = MAX_SPEED + (MIN_SPEED - MAX_SPEED) * bn;
	ghost_speed = pacman_speed * 1.2;
}

ISR(ADC_vect){
	event = 72;
	set_speed();
}

// BOARD ACCESS: HELPER FUNCTIONS

inline int8_t next_pos(int8_t steps, position *pos){
	switch(pos->dir){
		case RIGHT:
			pos->px += steps;
			break;
		case DOWN:
			pos->py += steps;
			break;
		case LEFT:
			pos->px -= steps;
			break;
		case UP:
			pos->py -= steps;
			break;
		default: return -1;
	}
	return 0;
}

static uint8_t next_pos_in_board(position *pos){
	switch(pos->dir){
		case RIGHT:
			if(pos->px + TILE_SIZE >= 127) return false;
			break;
		case DOWN:
			if(pos->py + TILE_SIZE >= 63) return false;
			break;
		case LEFT:
			if(pos->px < TILE_SIZE ) return false;
			break;
		case UP:
			if(pos->py < TILE_SIZE) return false;
			break;
		default:
			sb_7seg_showString("di");
			while(1);
	}
	return true;
}

static uint8_t pos_free(position *pos){
	return (board[pos->py / TILE_SIZE] & ( 1 << (pos->px / TILE_SIZE))) == 0;
}

static uint8_t next_pos_free(position *pos){
	if(!next_pos_in_board(pos)) return false;

	if(next_pos(8, pos)){
		sb_7seg_showString("cf");
		while(1);
	}
	uint8_t ret = pos_free(pos);

	if(next_pos(-8, pos)){
		sb_7seg_showString("CF");
		while(1);
	}
	return ret;
}

static inline bool is_special_dot(uint8_t px, uint8_t py){
	return (special_dots[py / TILE_SIZE] & (1 << (px / TILE_SIZE))) != 0;
}


// RENDER FUNCTIONS

static void render_pacman(){
	uint8_t tile[TILE_SIZE];

	uint8_t pos_lag = (player_pos.px % 8) + (player_pos.py % 8);
	uint8_t bias;

	switch(player_pos.dir){
		case DOWN: bias = ROTATE_1; break;
		//case LEFT: bias = ROTATE_2; break;
		case LEFT: bias = MIRROR_V; break;
		case UP:   bias = ROTATE_3; break;
		default:   bias = NORMAL;
	}

	switch(pos_lag){
		case 1:
		case 6: calc_bitmap(pacman_bm[1], tile, bias); break;
		case 2:
		case 5: calc_bitmap(pacman_bm[2], tile, bias); break;
		case 3:
		case 4: calc_bitmap(pacman_bm[3], tile, bias); break;
		default:  calc_bitmap(pacman_bm[0], tile, bias);

	}
	render_tile(&player_pos, tile);
}

static void render_ghost(){
	uint8_t tile[TILE_SIZE];
	switch(ghost_pos.dir){
		case LEFT: calc_bitmap(ghost_bm[0], tile, MIRROR_V); break;
		case UP: calc_bitmap(ghost_bm[1], tile, NORMAL); break;
		case DOWN: calc_bitmap(ghost_bm[2], tile, NORMAL); break;
		default: calc_bitmap(ghost_bm[0], tile, NORMAL);
	}
	render_tile(&ghost_pos, tile);
}

static inline void clear_pacman_pos(){
	render_tile(&player_pos, 0);
}

static inline void clear_ghost_pos(){
	render_tile(&ghost_pos, 0);
}

static void render_tile(position *pos, uint8_t *tile){

	uint8_t page_start = (pos->py) % 8;
	if(page_start){
		if(tile){
			uint8_t page_cont[TILE_SIZE];
			for(uint8_t i = 0; i < TILE_SIZE; ++i){
				page_cont[i] = 0x0 | (tile[i] << (page_start));
			}
			sb_display_drawBitmap(pos->py/TILE_SIZE,pos->px,1, TILE_SIZE, page_cont);

			for(uint8_t i = 0; i < TILE_SIZE; ++i){
				page_cont[i] = 0x0 | (tile[i] >> (TILE_SIZE - page_start));
			}
			sb_display_drawBitmap(pos->py/TILE_SIZE+1,pos->px,1, TILE_SIZE, page_cont);
		} else {
			sb_display_drawBitmap(pos->py/TILE_SIZE,pos->px,1, TILE_SIZE, 0);
			sb_display_drawBitmap((pos->py/TILE_SIZE)+1,pos->px,1, TILE_SIZE, 0);
		}
	} else {
		sb_display_drawBitmap((pos->py/TILE_SIZE),pos->px,1, TILE_SIZE, tile);
	}
}

// ACTUALIZE GHOST DIR

static inline uint16_t euclidian_distance_sqr(uint8_t px1, uint8_t py1, uint8_t px2, uint8_t py2){
	uint16_t dx = px1 - px2;
	uint16_t dy = py1 - py2;
	return  dx * dx + dy * dy;
}

static inline uint16_t distance_ghost_pacman(){
	return euclidian_distance_sqr(ghost_pos.px, ghost_pos.py, player_pos.px, player_pos.py);
}

static void calc_next_ghost_dir(){
	uint16_t distance = -1;
	if(ghost_vulnerable) distance = 0;
	uint8_t best_dir = 0;
	for(int8_t dir = 0; dir < 4; ++dir){
		ghost_pos.dir = dir;
		if(!next_pos_in_board(&ghost_pos)) continue;
		next_pos(TILE_SIZE, &ghost_pos);
		if(pos_free(&ghost_pos)){
			uint16_t tmp = distance_ghost_pacman();
			if(!ghost_vulnerable && tmp < distance){
				distance = tmp;
				best_dir = dir;
			} else if(ghost_vulnerable && tmp > distance){
				distance = tmp;
				best_dir = dir;
			}
		}
		next_pos(-TILE_SIZE, &ghost_pos);
	}
	ghost_pos.dir = best_dir;
}

static void update_board(void){
	set_speed();

	switch(event){

		case EVENT_CHANGE_DIR:
			if(state != HALT){
				break;
			}

		case EVENT_ACT_P_POS:
			if(state == PLAY){
				clear_pacman_pos();
				if(next_pos(1, &player_pos)){
					sb_7seg_showString("PA");
					while(1);
				}
				render_pacman();
			}

			state = PLAY;
			//new tile reached
			if( !(player_pos.px % 8) && !(player_pos.py % 8)){
				//eat dot
				uint8_t dot= erase_dot(player_pos.px, player_pos.py);
				if(dot){
					points += dot;
					dots_left -= dot;
					if( is_special_dot(player_pos.px, player_pos.py) ){
						ghost_vulnerable = 0xFF;
						be_vulnerable();
					}

				}
				sb_7seg_showNumber(points);
				//set next dir
				player_pos.dir = player_pos.next_dir;

				if(next_pos_free(&player_pos)){
				} else {
					//sb_led_on(RED0);
					state = HALT;
					render_pacman();
				}
		}
		break;

		case EVENT_ACT_G_POS:
			clear_ghost_pos();
			if(next_pos(1, &ghost_pos)){
				sb_7seg_showString("GO");
				while(1);
			}
			render_ghost();

			if( !(ghost_pos.px % 8) && !(ghost_pos.py % 8)){
				uint8_t dot = erase_dot(ghost_pos.px, ghost_pos.py);
				dots_left -= dot;
				calc_next_ghost_dir();
			}

			break;

		default:;

	};

}
// active-low: low-Pegel (logisch 0; GND am Pin) =>  LED leuchtet

static void calc_bitmap(const __flash uint8_t bm[8], uint8_t *display_bm, uint8_t orient){
	// Clear
	for(uint8_t i = 0; i < TILE_SIZE; ++i) display_bm[i] = 0;
	// Draw
	for(uint8_t x = 0; x < TILE_SIZE; ++x){
		for(uint8_t y = 0; y < TILE_SIZE; ++y){
			uint8_t pixel = 0;
			switch(orient){
				case ROTATE_1:
					pixel = (bm[TILE_SIZE - 1 - x] >> (TILE_SIZE - 1 - y)) & 1;
					break;
				case ROTATE_2:
					pixel = (bm[TILE_SIZE - 1 - y] >> x) & 1;
					break;
				case ROTATE_3:
					pixel = (bm[x] >> y) & 1;
					break;
				case MIRROR_V:
					pixel = (bm[y] >> x) & 1;
					break;
				case MIRROR_H:
					pixel = (bm[TILE_SIZE - 1 - y] >> (TILE_SIZE - 1 - x)) & 1;
					break;
				default:
					pixel = (bm[y] >> (TILE_SIZE - 1 - x)) & 1;
			}
			display_bm[x] |= (pixel << y);
		}
	}
}

//ON WIN / ON LOOSE

static void wait(){
	event = EVENT_WAIT;
}

static void wait_time(uint16_t time){
	sb_timer_setAlarm((alarmcallback_t) wait, time, 0);
	cli();
	while( !event ) {
		sleep_enable();
		sei();
		sleep_cpu();
		cli();
		sleep_disable();
	}
	sei();
	event = 0;
}

static void show_win(void){
	uint8_t set_leds = 0xFF;
	uint16_t time = 100;
	for(int i = 0; i <= 8; ++i){
		sb_led_setMask(set_leds);
		set_leds = set_leds << 1;
		wait_time(time);
	}
	uint8_t cursor = 7;
	for(uint8_t i = 0; i < 15; ++i){
		sb_led_setMask(set_leds ^ (1 << cursor));
		if(i < 7) --cursor;
		else ++cursor;
		wait_time(time);
	}
	uint8_t leds[] = {0x00, 0x81, 0xC3, 0xE7, 0xFF};
	for(uint8_t i = 0; i < sizeof(leds); ++i){
		sb_led_setMask(leds[i]); //1000 0001
		wait_time(time);
	}
	for(int8_t i = (int8_t)sizeof(leds) - 2; i >= 0; --i){
		sb_led_setMask(leds[i]); //1000 0001
		wait_time(time);
	}
}

static void show_loose(void){
	uint8_t set_leds = 0;
	uint16_t time = 500;
	sb_led_setMask(set_leds);
	const char loser[] = "LOSER";
	sb_display_fillScreen(NULL); // Clear display
	for(uint8_t i = 0; i < 20; ++i){
		sb_led_toggle(RED0);
		sb_led_toggle(RED1);
		sb_led_toggle(YELLOW0);
		sb_led_toggle(YELLOW1);
		if(!(i % 4)  || !((i+1)%4)) sb_display_fillScreen(NULL); // Clear display
		else sb_display_showStringWide(3,128/2-20,loser);
		wait_time(time);
	}
}

void main(void){
	sei();

	if(sb_display_available() != 0){
		sb_7seg_showString("--");
		while(1);
	}

	if(sb_display_enable() != 0){
		sb_7seg_showString("++");
		while(1);
	}

	set_sleep_mode(SLEEP_MODE_IDLE); /* Idle-Modus verwenden */

	sb_display_fillScreenFromFlash(splash);
	wait_time(4000);

	init();

	while(1){
		timer0_init();
		cli();
		while( !event ) {
			sleep_enable();
			sei();
			sleep_cpu();
			cli();
			sleep_disable();
		}
		sei();
		update_board();
		if(!ghost_pos.dead && distance_ghost_pacman() < DEADLY_DISTANCE){
			if(!ghost_vulnerable){
				deaktivate_timers();
				show_loose();
				init();
			} else {
				ghost_pos.dead = true;
				clear_ghost_pos();
				render_pacman();
				points += 20;
				sb_led_setMask(0);
			}
		} else if(!dots_left ){
			deaktivate_timers();
			show_win();
			init();
		}
		event = 0;
	}

	while(1); // Stop loop
}
