#include <avr/interrupt.h>
#include <util/delay.h>
#include <stdint.h>
#include <stdbool.h>
#include <stdlib.h>

#include "7seg.h"
#include "adc.h"
#include "button.h"
#include "com.h"
#include "com_ext.h"
#include "led.h"
#include "display.h"

static const __flash uint8_t splash_pong[128*8] = { 0xfc, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x84, 0x84, 0x84, 0x7c, 0x40, 0x40, 0x7c, 0x84, 0x84, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0xfc, 0x00, 0x00, 0xb8, 0xb8, 0xb8, 0x00, 0x00, 0x00, 0x80, 0xe0, 0xf0, 0xf8, 0xf8, 0xf8, 0x00, 0x00, 0x00, 0x80, 0xc0, 0xc0, 0xc0, 0x60, 0x60, 0xe0, 0xc0, 0x00, 0xc0, 0xc0, 0xe0, 0xe0, 0x60, 0x60, 0x60, 0x60, 0xc0, 0xc0, 0x00, 0x60, 0x60, 0x60, 0x00, 0x00, 0x00, 0x00, 0x80, 0x80, 0xc0, 0xe0, 0x60, 0xe0, 0xe0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x40, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xc0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xff, 0x00, 0x40, 0x60, 0x50, 0xd0, 0x08, 0x08, 0x04, 0x02, 0x1e, 0xe0, 0x01, 0x06, 0x04, 0xfc, 0x04, 0x04, 0xfc, 0x08, 0x08, 0x10, 0x21, 0x42, 0x82, 0x04, 0x18, 0x60, 0x80, 0x00, 0x00, 0x00, 0x00, 0xff, 0x00, 0x00, 0xff, 0xff, 0xff, 0x00, 0x3c, 0x3f, 0x3f, 0x3f, 0x39, 0xff, 0xff, 0xff, 0x38, 0x38, 0x00, 0xc3, 0xc7, 0xce, 0xcc, 0xcc, 0xfc, 0x78, 0x30, 0x00, 0x00, 0xff, 0xff, 0x1d, 0x0c, 0x0e, 0x06, 0x07, 0x03, 0x01, 0x18, 0x7e, 0xfe, 0xe0, 0x60, 0x70, 0x78, 0xfe, 0xc7, 0xc3, 0xc1, 0xc0, 0x61, 0x71, 0x31, 0x10, 0x88, 0x8e, 0x72, 0x20, 0x20, 0x10, 0xfc, 0x1c, 0x4c, 0x62, 0x22, 0x3a, 0x14, 0x10, 0x08, 0x3c, 0x64, 0x40, 0x40, 0x20, 0x10, 0x38, 0x5e, 0x49, 0x47, 0x00, 0x20, 0x10, 0x18, 0x67, 0x40, 0x40, 0x20, 0x10, 0x3c, 0x4a, 0x49, 0x47, 0x20, 0x20, 0x10, 0x00, 0xf3, 0x12, 0x12, 0x12, 0x3f, 0xc0, 0x00, 0x00, 0x3f, 0xd3, 0x12, 0x17, 0x12, 0x12, 0x12, 0xf3, 0x00, 0x00, 0xf3, 0x12, 0x12, 0x12, 0x12, 0xd2, 0x3f, 0x00, 0x00, 0x80, 0x7f, 0x12, 0x12, 0x12, 0x12, 0xf3, 0x00, 0xf0, 0xf0, 0xf0, 0xf0, 0xf0, 0xf0, 0xf0, 0xf0, 0xf0, 0xf0, 0xf0, 0xf0, 0xf0, 0xf0, 0xf0, 0xe0, 0xe0, 0xe0, 0xc0, 0xc0, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x64, 0x74, 0x5c, 0x4c, 0x00, 0x38, 0x7c, 0x54, 0x5c, 0x58, 0x00, 0x7d, 0x7d, 0x00, 0x38, 0x7c, 0x44, 0xfc, 0xfc, 0x00, 0x04, 0x3f, 0x7f, 0x44, 0x44, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xff, 0x00, 0x00, 0x00, 0x00, 0x00, 0x03, 0x04, 0x08, 0x10, 0x21, 0x22, 0x42, 0x44, 0x44, 0xcf, 0x48, 0x48, 0xcf, 0x44, 0x44, 0x22, 0x21, 0x10, 0x08, 0x04, 0x02, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0xff, 0x00, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x0f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfc, 0x00, 0x00, 0xc0, 0xe0, 0xf0, 0xf8, 0xf8, 0xfc, 0xfc, 0xfc, 0xfc, 0xfc, 0xfc, 0xfc, 0xfc, 0xfc, 0xf8, 0xf8, 0xf0, 0xf0, 0xe0, 0x80, 0x00, 0x00, 0xf8, 0xfc, 0xfc, 0xfc, 0xfc, 0xfc, 0xfc, 0xfc, 0xfc, 0xfc, 0xfc, 0xf8, 0xf8, 0xfc, 0xfc, 0xfc, 0xfc, 0xfc, 0xfc, 0xf8, 0xf8, 0xf0, 0x80, 0x00, 0x00, 0xe0, 0xf0, 0xf8, 0xfc, 0xfc, 0xfd, 0xfd, 0xfd, 0xfc, 0xfc, 0xfc, 0xfc, 0xfc, 0xfc, 0xfc, 0xfc, 0xfc, 0xfc, 0xfc, 0xf8, 0xf0, 0x00, 0x0f, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x0f, 0x00, 0x00, 0x0f, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x0f, 0x00, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf8, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x3f, 0x00, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x03, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfe, 0x00, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x01, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x00, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x01, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x07, 0x07, 0x07, 0x07, 0x07, 0x07, 0x03, 0x03, 0x01, 0x00, 0x00, 0x00, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x00, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x00, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x00, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x00, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x80, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x1f, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfc, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7f, 0x1f, 0x03, 0x00, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x00, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x00, 0xc1, 0xc7, 0xcf, 0xdf, 0xdf, 0xdf, 0xdf, 0xdf, 0xdf, 0xdf, 0xdf, 0x9f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x01, 0x01, 0x03, 0x03, 0x03, 0x03, 0x03, 0x03, 0x03, 0x03, 0x03, 0x01, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x00, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x00, 0x01, 0x07, 0x0f, 0x1f, 0x1f, 0x3f, 0x3f, 0x3f, 0x3f, 0x3f, 0x3f, 0x3f, 0x3f, 0x3f, 0x3f, 0x3f, 0x3f, 0x1f, 0x1f, 0x0f, 0x07, 0x03, 0x00 };

typedef struct {
	int16_t x;
	int16_t y;
} pos;

typedef struct {
	int16_t paddleHeight;
	uint8_t maxSpeed;
	uint8_t startSpeed;
} settings;

settings gameSettings = {112L, 31, 10};
const pos ballInit[2] = { { 447L , 512L } , { 1600L , 512L }};
pos ball = { 447L , 512L };
pos ballDir = { 10L, 0L };
pos paddle[2] = { { 47L, 512L}, { 2000L, 512L} };
bool singlePlayer = true;
uint8_t speed = 2;
uint8_t difficulty = 1;

#define BTN0(B) (((B) & (1)) != 0)
#define BTN1(B) (((B) & (2)) != 0)

static uint8_t getPressedButtons() {
	static uint8_t previousState[2] = {UNKNOWN, UNKNOWN};
	uint8_t currentState[2] = { sb_button_getState(BUTTON0), sb_button_getState(BUTTON1) };
	uint8_t r = 0;
	for (uint8_t i=0;i<2;i++){
		r |= currentState[i] == PRESSED && previousState[i] == RELEASED ? (1 << i) : 0;
		previousState[i] = currentState[i];
	}
	return r;
}

inline static void displayDrawPaddles(){
	uint8_t pages[8];
	for (uint8_t p=0;p<2;p++){
		for (int16_t i = 0; i<8 ; i ++){
			pages[i] = 0;
			for (int16_t j = 0; j<8 ; j++)
			if (i*8+j >= paddle[p].y/16L - gameSettings.paddleHeight/32L && i*8+j <= paddle[p].y/16L + gameSettings.paddleHeight/32L)
			pages[i] |= (1<<j);
		}
		sb_display_drawBitmap(0, paddle[p].x / 16, 8, 1, pages);
	}
}

static void menu(){
	// Menu - ugly but simple.
	sb_display_fillScreen(NULL);
	sb_display_showStringSmall(0, 46, "PONG als");
	sb_display_showStringSmall(2, 56, "mit");
	uint8_t prevItem = 5;
	const char * speedName[] = {"zeitlupe", "langsam ", "normal  ", "schnell ","zzzuuppp"};
	const char * difficultyName[] = {"leicht", "normal", "schwer", "ultra "};
	while (true){
		uint8_t item = 4-(sb_adc_read(POTI)*5/1024);
		uint8_t button = getPressedButtons();
		if (item != prevItem || button != 0){
			if (prevItem == 5 || prevItem == 0 || item == 0){
				if (item == 0 && button != 0)
					singlePlayer = !singlePlayer;
				if (singlePlayer)
					sb_display_showString(1, 4, item == 0 ? "\20Einzelspieler\21" : " Einzelspieler ");
				else {
					sb_display_showString(1, 4, item == 0 ? "\20 Mehrspieler \21" : "  Mehrspieler  ");
					sb_display_drawBitmap(5, 8, 1, 110, NULL); // No difficulty
				}
			}
			if (prevItem == 5 || prevItem == 1 || item == 1){
				if (item == 1 && BTN0(button) && gameSettings.paddleHeight < 300)
					gameSettings.paddleHeight += 16;
				else if (item == 1 && BTN1(button) && gameSettings.paddleHeight > 50)
					gameSettings.paddleHeight -= 16;
				sb_display_showStringSmall(3, 8, item == 1 ? ">Paddelgroesse:" : " Paddelgroesse:");
				uint8_t pages[22] = {0};
				for (uint8_t i=0;i<gameSettings.paddleHeight/16;i++)
					pages[i] = 0x18;
				sb_display_drawBitmap(3, 72, 1, 22, pages);
			}
			if (prevItem == 5 || prevItem == 2 || item == 2){
				sb_display_showStringSmall(4, 8, item == 2 ? ">Geschwindigkeit:" : " Geschwindigkeit:");
				if (item == 2 && BTN0(button) && speed < 4)
					speed++;
				else if (item == 2 && BTN1(button) && speed > 0)
					speed--;
				sb_display_showStringSmall(4, 80, speedName[speed]);
			}
			if ((prevItem == 5 || prevItem == 3 || item == 3 || prevItem == 0 || item == 0) && singlePlayer){
				sb_display_showStringSmall(5, 8, item == 3 ? ">Schwierigkeit:" : " Schwierigkeit:");
				if (item == 3 && BTN0(button) && difficulty < 3)
					difficulty++;
				else if (item == 3 && BTN1(button) && difficulty > 0)
					difficulty--;
				sb_display_showStringSmall(5, 72, difficultyName[difficulty]);
			}

			if (prevItem >= 4 || item == 4){
				sb_display_showString(7, 28, item == 4 ? "\20starten\21" : " starten ");
				if (item == 4 && button != 0)
					break;
			}
		}
	}

	// set speed
	switch (speed){
		case 0:
			gameSettings.startSpeed = 5;
			break;
		case 1:
			gameSettings.maxSpeed = 15;
			gameSettings.startSpeed = 8;
			break;
		case 3:
			gameSettings.startSpeed = 15;
			break;
		case 4:
			gameSettings.startSpeed = 30;
			break;
		default:
			gameSettings.maxSpeed = 25;
	}
}

static void game_ai(uint8_t player){
	static int16_t damping[2] = {0};
	int16_t expectedY = (512L + ball.y + paddle[player - 1].y)/3L;
	if ((player == 1 && ballDir.x > 0) || (player == 0 && ballDir.x < 0))
		expectedY = ball.y + ballDir.y * ((paddle[player].x - ball.x)/ballDir.x);
	for (uint8_t i = 0; i <= difficulty * difficulty; i++){
		if (expectedY > 1023L)
			expectedY = 2046L - expectedY;
		else if (expectedY < 0)
			expectedY = -expectedY;
	}
	int16_t distanceX = paddle[player].x < ball.x ? (ball.x - paddle[player].x) : (paddle[player].x - ball.x);
	int16_t distanceY = paddle[player].y < expectedY ? (expectedY - paddle[player].y) : (paddle[player].y - expectedY);

	switch (difficulty){
		case 3:
			paddle[player].y = ball.y;
			break;
		case 2:
			if (distanceY * 3 > gameSettings.paddleHeight){
				int8_t movement = (paddle[player].y < expectedY ? 4 : (-4));
				if ((movement > 0 && damping[player] >= 0) || (movement < 0 && damping[player] <= 0))
				paddle[player].y += movement;
				damping[player] = movement;
			}
			else
				damping[player] = 0;
			break;
		case 1:
			if (distanceX < 1300){
				if (distanceY > (ballDir.x > 0 ? ballDir.x : -ballDir.x) + distanceX/50){
					int8_t movement = (paddle[player].y < expectedY ? 1 : (-1)) * ((player == 1 && ballDir.x > 0) || (player == 0 && ballDir.x < 0) ? (7 - distanceX/300) : 1);
					if ((movement > 0 && damping[player] >= 0) || (movement < 0 && damping[player] <= 0))
						paddle[player].y += movement;
					damping[player] = movement;
				}
				else
					damping[player] = 0;
				break;
			}
		case 0:
			if ((paddle[player].y - ball.y) * (paddle[player].y - ball.y) > gameSettings.paddleHeight)
				paddle[player].y += damping[player] = (paddle[player].y < ball.y ? 7 : (-7));
			break;
	}
}

static bool game(uint8_t player){
	sb_display_fillScreen(NULL);
	uint8_t points[2] = {0};

	ballDir.x = gameSettings.startSpeed;
	ballDir.y = 0L;

	sb_7seg_showNumber(0);

	sb_display_showString(2, 12, "Fertig machen,");
	if (singlePlayer)
		sb_display_showString(4, 20,  player == 0 ? "\21 Spieler 1" : "Spieler 2 \20");
	else
		sb_display_showString(4, 8,  player == 0 ? "\21 lok. Spieler" : "lok. Spieler \20");
	_delay_ms(1000);
	sb_display_fillScreen(NULL);

	while(1){ // Main game loop
		paddle[player].y = 1023 - sb_adc_read(POTI);
		if (singlePlayer) // AI
			game_ai(1 - player);
		else {
			// For performance reasons we just want to receive one byte.
			// And therefore we will scale down the POTI resolution of the local player as well :)
			paddle[player].y -= paddle[player].y%4;
			if (sb_com_hasReceivedItem()){
				uint8_t r;
				if (sb_com_receiveByte(&r) == 0){
					paddle[1 - player].y = r * 4;
				}
			}
		}

		pos old  = ball;
		ball.x += ballDir.x;
		ball.y += ballDir.y;
		if (ball.y < 0) {
			ballDir.y = -ballDir.y;
			ball.y = -ball.y;
		}
		if (ball.y > 1023) {
			ballDir.y = -ballDir.y;
			ball.y = 2046 - ball.y;
		}

		//check for collision with the paddle
		for (int i = 0; i < 2; i++) {
			if (ball.x <= paddle[i].x + 15L && ball.x >= paddle[i].x - 15L && ball.y <= paddle[i].y + gameSettings.paddleHeight/2L && ball.y >= paddle[i].y - gameSettings.paddleHeight/2L){

				//change ball direction
				ballDir.x = -ballDir.x;
				if (ballDir.x > 0){
					if (ball.x < paddle[i].x)
						ball.x = paddle[i].x;
					if (speed > 0 && ballDir.x < gameSettings.maxSpeed)
						ballDir.x++;
				} else {
					if (ball.x > paddle[i].x)
						ball.x = paddle[i].x;
					if (speed > 0 && ballDir.x > -gameSettings.maxSpeed)
						ballDir.x--;
				}

				//change ball angle based on where on the paddle it hit
				ballDir.y = (ball.y - paddle[i].y + ballDir.y) * 30L / gameSettings.paddleHeight;
			}
		}

		if (ball.x <= 0) {
			// player 2 scores
			ball = ballInit[0];
			points[1]++;
			ballDir = (pos){ gameSettings.startSpeed , rand()%5 -3 };
			sb_7seg_showNumber(points[0]  * 10 + points[1]);
			_delay_ms(500);
		}
		else if (ball.x >= 2047) {
			// player 1 scores
			ball = ballInit[1];
			points[0]++;
			ballDir = (pos){ -gameSettings.startSpeed , rand()%5 -3 };
			sb_7seg_showNumber(points[0]  * 10 + points[1]);
			_delay_ms(500);
		}
		for (int i = 0; i < 2; i++)
			if (points[i] > 9){
				sb_7seg_disable();
				if (!singlePlayer)
					sb_display_showStringSmall(3, 26, "lokaler Spieler hat");
				sb_display_showString(4, 28, player == i ? "Gewonnen!" : "Verloren!");
				sb_display_showStringSmall(7, 2, "Druecke Taste 1 fuer neues Spiel");
				uint8_t buttons;
				while((buttons = getPressedButtons()) == 0);
				return BTN0(buttons);
			}

		uint8_t buttons = getPressedButtons();
		// Pause
		if (BTN1(buttons)){
			sb_display_showString(2, 34, "[Pause]");
			sb_display_showStringSmall(4, 38, "Zwischenstand");
			char pointsStr[4] = { '0'+points[0], ':', '0'+points[1], '\0' };
				sb_display_showString(5, 50, pointsStr);
			while(getPressedButtons() == 0);
			sb_display_fillScreen(NULL);
		}
		// Stop game
		else if (BTN0(buttons)){
			sb_display_showString(2, 24, "[Beenden?]");
			sb_display_showStringSmall(7, 0, "Nochmal druecken fuer Abbruch...");
			while((buttons = getPressedButtons()) == 0);
			if (BTN0(buttons))
				return false;
			else
				sb_display_fillScreen(NULL);
		}

		// Draw ballds
		sb_display_draw(ball.y/128, ball.x / 16, 1 << ((ball.y/16)%8));
		if (ball.x/16 != old.x / 16 || ball.y/128 != old.y/128)
			sb_display_draw(old.y/128, old.x / 16, 0);
		displayDrawPaddles();

		_delay_ms(3);
	}
}


int main(void){
	if (sb_display_enable() == 0){
		// Splash screen
		sb_display_fillScreenFromFlash(splash_pong);
		while(getPressedButtons() == 0);
		sei();
		while (true){
			menu();
			int8_t player = 1;
			while (game(player = 1 - player));
		}
	}
	sb_led_on(RED0);
	sb_led_on(RED1);
	while (1);
}
