/* -*- c++ -*- ***************************************************************/
/* Echtzeitsysteme                                                           */
/*---------------------------------------------------------------------------*/
/*                                                                           */
/*                          T E S T _ I N T E R R U P T                      */
/*                                                                           */
/*---------------------------------------------------------------------------*/
/* Triggering and handling interrupts within testcases.                      */
/*****************************************************************************/

#ifndef __test_interrupt_h__
#define __test_interrupt_h__

#include "interrupt/gate.h"

/**
 * \class Test_Interrupt
 * \brief Triggering and handling interrupts within testcases
 *
 * This class is realised as template class as we want to be able to use
 * different peripheral devices for generating the interrupts. The 
 * implementation of device is passed as a template parameter, the particular
 * class has just provide a trigger and an acknowledge method.
 **/
template < class BASE > class Test_Interrupt : public BASE {

  /**
   * this variable is used to implement the synchronous semantics of the
   * method trigger(). This method sets this flag to false and loops until
   * this flag has been set by the acknowledge method()
   *
   * \see Test_Interrupt::trigger()
   * \see Test_Interrupt::acknowledge()
   **/
  volatile bool interrupt_happened;

public:

  /**
   * \brief Constructor
   **/
  Test_Interrupt() : BASE(),interrupt_happened(false) {}

  /**
   * \brief Trigger an interrupt synchronously
   * 
   * This method triggers an interrupt in a synchronous manner, it returns not
   * until the appropriate interrupt service routine has been called and the 
   * interrupt has been acknowledged.
   **/
  void trigger() {
    interrupt_happened = false;
    BASE::trigger();

    while(!interrupt_happened);
  }

  /**
   * \brief Trigger an interrupt asynchronously
   *
   * This method is useful for triggering an interrupt when no interrupts are 
   * allowed. When we would use the synchronous method in that case we would
   * wait for the occurence of the interrupt endlessly.
   * Here we wait until an interrupt request is generated, but we do not wait
   * until the sevice routine gets serviced.
   **/
  void trigger_async() {
    BASE::trigger();

    while(!BASE::is_triggered());
  }

  /**
   * \brief Acknowlegde the triggered interrupt
   **/
  void acknowledge() {
    BASE::acknowledge();
    interrupt_happened = true;
  }
};

#endif /* __test_interrupt_h__ */
