/* -*- c++ -*- ***************************************************************/
/* Echtzeitsysteme                                                           */
/*---------------------------------------------------------------------------*/
/*                                                                           */
/*             A B S T R A C T _ S C H E D U L E _ T A B L E                 */
/*                                                                           */
/*---------------------------------------------------------------------------*/
/* An abstract interface for the schedule table class, this is needed as the */
/* schedule table class is a template parametrized with the actual size of   */
/* the schedule table, but the schedule table scheduler should be able to    */
/* deal with schedule tables of different size, of course.                   */
/*---------------------------------------------------------------------------*/

#ifndef __abstract_schedule_table_h__
#define __abstract_schedule_table_h__

#include "infra/types.h"
#include "thread/thread.h"

class Scheduler_Implementation;

/**
 * \class Abstract_Schedule_Table
 * \brief The abstract interface of the Schedule_Table-class
 *
 * We need this abstract interface for the Schedule_Table-class, because the 
 * ScheduleTable-scheduler should be able to deal with schedule tables of 
 * different sizes.
 *
 * A schedule table can be regarded as a list of tuples. The first part of the
 * tuple is the point in the time when a thread should be started (counted from
 * the beginning of the scheduler round). The second part of the tuple is a
 * pointer to the thread that should be started. It can be assumed no two tuples
 * having the same offset exist and that these tuples are ordered in a proper 
 * order to the schedule table, t.m. if tuple a is added before tuple b, then 
 * then the following assumption also holds: b.offset > a.offset. Beside these
 * restrictions a single thread can be member of several tuples, that are all
 * part of the same schedule table.
 */
class Abstract_Schedule_Table {

protected:

  friend class Schedule_Table_Scheduler;

  /**
   * \brief one schedule table entry
   *
   * One entry within a schedule table
   */
  struct Schedule_Table_Entry {
  };

private:

  /**
   * \brief Retrieve one entry from the schedule table
   * \param slot The index of the entry that should be retrived from the schedule
   *             table
   * \return The entry of the schedule table belonging to the given index. If an
   *         invalid index is given, NULL will be returned.
   */
  virtual Schedule_Table_Entry* get_entry(ezstubs_uint8 slot) = 0;

  /**
   * \brief Initialize the schedule table.
   * \param real_timer_period For each schedule table there is one period that would
   *        fit best for the given schedule table. But in some cases the timer is
   *        not capable of producing interrupts with the wished period. So we have to
   *        initialize the schedule table dependent on the period the timer offers us.
   **/
  virtual void init(us_type real_timer_period) = 0;

  /**
   * \brief Proceed one tick.
   * \retval true Reached a scheduling point within the schedule table - 
   *              scheduling is necessary
   * \retval false Scheduling is not necessary
   *
   * This method counts the interrupts generated by the timer and determines if
   * a scheduling point is reached, in other words: do we have to start the next
   * thread, yes or no. This method is called by the Schedule-Table-Scheduler within 
   * the timer interrupt handler - according to the return-value set_need_reschedule 
   * is called or is not called.
   **/
  bool tick() {
    return false;
  }

  /**
   * \brief Return the next thread to be scheduled
   * \return The next thread to be schduled or NULL if there is no thread to be
   *         scheduled at the current time
   * \note This method also performs the wrap around within the schedule table if
   *       the end of the schedule table has been reached.
   **/
  Thread* next_thread();

public:

  /**
   * \brief Constructor
   * \param t A pointer to the stack that should be used with this schedule table
   * \param round A schedule table is executed cyclically. This parameter determines
   *              the length of one period, also referred to as scheduler round.
   */
  Abstract_Schedule_Table(void* t,us_type round) {}

  /**
   * \brief Virtual Destructor
   * Prevent compiler warnings due to virtual methods
   **/
  virtual ~Abstract_Schedule_Table() {}

  /**
   * \brief How many entries does the schedule table have.
   * \return The count of the entries of this schedule table.
   */
  virtual ezstubs_uint8 size() = 0;
};

#endif /* __abstract_schedule_table_h__ */
