/* -*- c++ -*- ***************************************************************/
/* Echtzeitsysteme                                                           */
/*---------------------------------------------------------------------------*/
/*                                                                           */
/*          M U L T I _ L E V E L _ Q U E U E _ S C H E D U L E R            */
/*                                                                           */
/*---------------------------------------------------------------------------*/
/* A multi level queue scheduler                                             */
/*****************************************************************************/

#ifndef __multi_level_queue_scheduler_h__
#define __multi_level_queue_scheduler_h__

#include "infra/types.h"

#include "interrupt/guard.h"

#include "thread/dispatch.h"
#include "thread/scheduler_base.h"
#include "thread/thread.h"
#include "thread/thread_queue.h"

/**
 * \class Multi_Level_Queue_Scheduler
 * \brief A Multi-Level-Queue-Scheduler
 **/
class Multi_Level_Queue_Scheduler
  : public Scheduler_Base, 
    public Dispatcher, 
    public Thread_Queue< 8 >
{

protected:
  
  /**
   * \Brief The current idle thread
   **/
  Thread* current_idle_thread;

  /**
   * \brief Set the idle thread
   * \param thread The idle thread to be set
   **/
  void set_idle_thread(Thread* thread) {}

  /**
   * \brief Get the next thread to be executed
   * \return The next thread to be executed
   *
   * Determine the thread with the highest thread within the ready queue.
   **/ 
  Thread* schedule();

public:

  /**
   * \brief Constructor
   **/  
  Multi_Level_Queue_Scheduler() {}

  /**
   * \brief Start the scheduler
   *
   * Start the first thread if there is a thread ready fpr execution otherwise
   * wait for the activation of a thread
   **/
  void start();

  /**
   * \brief Start the next thread
   *
   * Query the next thread ready for execution, that is assigned to the cpu.
   * If there is no thread to be executed wait for the activation of a thread.
   **/
  void reschedule();

  /**
   * \brief Get the current thread
   * \return The thread that is currently executed.
   **/
  Thread* current() {
    return 0;
  }

  /**
   * \brief Activate a thread
   * \param thread The thread to be activated
   *
   * Add a thread to the queue of threads, that are ready for execution. The
   * thread will be added as last thread to the queue belonging to the threads
   * priority.
   * A Thread can be activated several times, the order of activations will not
   * be preserved. If a thread has been activated several times and finally
   * finishes execution, the thread will just be enqueued again as often as it
   * has been activated.
   **/ 
  void add(Thread* thread);

  /**
   * \brief Release the processor.
   *
   * A thread releasing the processor will be enqueued as last thread in the
   * queue belonging to its priority.
   **/
  void yield();

  /**
   * \brief Finish execution of the currently running thread.
   *
   * A thread that finishes execution will be started from its beginning, when
   * it is dispatched the cpu the next time.
   * A thread, that has been activated multiple times, will be activated again,
   * when finishing execution. This means it will be enqueued as last thread in
   * the queue belonging to its priority. When the thread is assigned the cpu
   * again it starts from its beginning.
   **/
  void exit();

  /**
   * \brief Kill a thread.
   * \param thread The thread to be killed
   *
   * A thread, that has been killed will be executed from its beginning, when
   * it is assigned the cpu the next time.
   * A thread, that has been activated multiple times, will be activated again,
   * when being killed. This means it will be enqueued as last thread in
   * the queue belonging to its priority. When the thread is assigned the cpu
   * again it starts from its beginning.
   **/
  void kill(Thread* thread);
};

#endif /* __multi_level_queue_scheduler_h__ */
