/* -*- c++ -*- ***************************************************************/
/* Echtzeitsysteme                                                           */
/*---------------------------------------------------------------------------*/
/*                                                                           */
/*          M U L T I _ L E V E L _ Q U E U E _ S C H E D U L E R _          */
/*                  D E F E R R A B L E _ S E R V E R                        */
/*                                                                           */
/*---------------------------------------------------------------------------*/
/* A multi level queue scheduler that is also capable of dealing with        */
/* deferrable servers.                                                       */
/*****************************************************************************/

#ifndef __multi_level_queue_scheduler_deferrable_server_h__
#define __multi_level_queue_scheduler_deferrable_server_h__

#include "devices/scheduler_implementation_timer.h"
#include "infra/types.h"
#include "infra/utils.h"
#include "thread/deferrable_server.h"
#include "thread/idle_thread.h"
#include "thread/multi_level_queue_scheduler.h"

/**
 * \brief An extension of the multi level queue scheduler that is also capable
 *        of dealing with deferrable servers.
 *
 * To support deferrable servers, two things are necessary:
 *
 * - replenish the execution budgets at certain points in time
 * - monitor the execution budget of the deferrable servers and suspend a
 *   deferrable server, when its execution budget is exhausted
 *
 * The first part is implemented by computing the gcd among all replenishment
 * periods of all deferrable servers. The timer is then programmed to issue an
 * interrupt after any gcd time units.
 * To implement also the second requirement with the same timer, the scheduler
 * keeps track of the time that is left until the next replenishment interrupt.
 * Everytime a deferrable server is started time measurement is stopped and the
 * measured time is subtracted from a variable that keeps track of the time that
 * is left until the next replenishment interrupt. Then the timer is programmed
 * to issue an interrupt when the execution budget of the deferrable server is 
 * exhausted, then time measurement is started again. When a deferrable server
 * exits, yields or gets killed, time measurement is stopped and the time till
 * the next replenishment interrupt and the servers execution budget are updated
 * accordingly.
 **/
template < ezstubs_uint8 MAX_PRIO >
class Multi_Level_Queue_Scheduler_Deferrable_Server 
  : public Multi_Level_Queue_Scheduler, 
    public Scheduler_Implementation_Timer
{

  /**
   * \brief Choose the next thread to run,
   * \return The next thread to run.
   **/
  Thread* schedule();

public:

  /**
   * \brief Constructor
   **/
  Multi_Level_Queue_Scheduler_Deferrable_Server() {}

  /**
   * \brief Start the scheduler
   **/
  void start();
  void reschedule();

  /**
   * \brief Register a deferrable server with the scheduler
   * \param server The server to be registered with the scheduler
   **/
  void enlist(Deferrable_Server* server);

  /**
   * \brief Activate a deferrable server
   * \param server The server to be activated
   *
   * A deferrable server only is ready for execution when it has been updated
   * and the deferrable server still has execution budget.
   **/
  void add(Deferrable_Server* server);

  /**
   * \brief Finish the execution.
   *
   * The deferrable server finishes execution but its execution budget is not 
   * lost. The deferrable server can be reactivated and will then be executed
   * again. Normal threads are treated as usual.
   **/
  void exit();

  /**
   * \brief Release the processor
   *
   * When a deferrable server releases the remaining execution budget is lost.
   * Normal threads are treated as usual.
   **/
  void yield();

  /**
   * \brief Kill the deferrable server
   * \param server The server to be killed.
   * \see exit()
   *
   * When a deferrable server is killed it preserves is execution budget.
   **/
  void kill(Deferrable_Server* server);

  bool isr();
  void dsr();
};


template < ezstubs_uint8 MAX_PRIO >
Thread* Multi_Level_Queue_Scheduler_Deferrable_Server< MAX_PRIO >::schedule() {
  return 0;
}

template < ezstubs_uint8 MAX_PRIO >
void Multi_Level_Queue_Scheduler_Deferrable_Server< MAX_PRIO >::start() {
}

template < ezstubs_uint8 MAX_PRIO >
void Multi_Level_Queue_Scheduler_Deferrable_Server< MAX_PRIO >::reschedule() {
}

template < ezstubs_uint8 MAX_PRIO >
void Multi_Level_Queue_Scheduler_Deferrable_Server< MAX_PRIO >::enlist(Deferrable_Server* server) {
}

template < ezstubs_uint8 MAX_PRIO >
void Multi_Level_Queue_Scheduler_Deferrable_Server< MAX_PRIO >::add(Deferrable_Server* server) {
}

template < ezstubs_uint8 MAX_PRIO >
void Multi_Level_Queue_Scheduler_Deferrable_Server< MAX_PRIO >::exit() {
}

template < ezstubs_uint8 MAX_PRIO >
void Multi_Level_Queue_Scheduler_Deferrable_Server< MAX_PRIO >::yield() {
}

template < ezstubs_uint8 MAX_PRIO >
void Multi_Level_Queue_Scheduler_Deferrable_Server< MAX_PRIO >::kill(Deferrable_Server* server) {
}

template < ezstubs_uint8 MAX_PRIO >
bool Multi_Level_Queue_Scheduler_Deferrable_Server< MAX_PRIO >::isr() {
}

template < ezstubs_uint8 MAX_PRIO >
void Multi_Level_Queue_Scheduler_Deferrable_Server< MAX_PRIO >::dsr() {
}

#endif /* __multi_level_queue_schedulerdeferrable_server_h__ */
