/* -*- c++ -*- ***************************************************************/
/* Echtzeitsysteme                                                           */
/*---------------------------------------------------------------------------*/
/*                                                                           */
/*                     C P U _ I M P L E M E N T A  T I O N                  */
/*                                                                           */
/*---------------------------------------------------------------------------*/
/* Die Abstraktion fuer einen 'konkreten' Prozessor                          */
/*---------------------------------------------------------------------------*/

#ifndef __cpu_implementation_h__
#define __cpu_implementation_h__

/**
 * \class CPU_Implementation
 * \brief Abstraction of a processor architecture
 **/
class CPU_Implementation {

public:

  /**
   * \brief Enable interrupts
   **/
  void enable_int() { 
    __asm__ volatile("stmfd r13!,{r0,r1}\n"
		     "mrs   r0,cpsr\n"
		     "mov   r1,#192\n"		   
		     "mvn   r1,r1\n"
		     "and   r0,r0,r1\n"
		     "msr   CPSR_cxsf,r0\n"
		     "ldmfd r13!,{r0,r1}\n"); 
  }

  /**
   * \brief Disable interrupts
   **/
  void disable_int() { 
    __asm__ volatile("stmfd r13!,{r0,r1}\n"
		     "mrs   r0,cpsr\n"
		     "mov   r1,#192\n"
		     "orr   r0,r0,r1\n"
		     "msr   CPSR_cxsf,r0\n"
		     "ldmfd r13!,{r0,r1}\n"); 
  }

  /**
   * \brief Enable IRQs
   **/
  void enable_irq() { 
    __asm__ volatile("stmfd r13!,{r0,r1}\n"
		     "mrs   r0,cpsr\n"
		     "mov   r1,#128\n"		   
		     "mvn   r1,r1\n"
		     "and   r0,r0,r1\n"
		     "msr   CPSR_cxsf,r0\n"
		     "ldmfd r13!,{r0,r1}\n"); 
  }

  /**
   * \brief Disable IRQs
   **/
  void disable_irq() { 
    __asm__ volatile("stmfd r13!,{r0,r1}\n"
		     "mrs   r0,cpsr\n"
		     "mov   r1,#128\n"
		     "orr   r0,r0,r1\n"
		     "msr   CPSR_cxsf,r0\n"
		     "ldmfd r13!,{r0,r1}\n"); 
  }

  /**
   * \brief Enable FIQs
   **/
  void enable_fiq() { 
    __asm__ volatile("stmfd r13!,{r0,r1}\n"
		     "mrs   r0,cpsr\n"
		     "mov   r1,#64\n"		   
		     "mvn   r1,r1\n"
		     "and   r0,r0,r1\n"
		     "msr   CPSR_cxsf,r0\n"
		     "ldmfd r13!,{r0,r1}\n"); 
  }

  /**
   * \brief Disable FIQs
   **/
  void disable_fiq() { 
    __asm__ volatile("stmfd r13!,{r0,r1}\n"
		     "mrs   r0,cpsr\n"
		     "mov   r1,#64\n"
		     "orr   r0,r1\n"
		     "msr   CPSR_cxsf,r0\n"
		     "ldmfd r13!,{r0,r1}\n"); 
  }

  /**
   * \brief Stop the processor
   **/
  void halt() { for(;;); }

  /**
   * \brief put the processor into sleep mode
   **/
  void sleep() { for(;;); }

  /**
   * \brief Set the stack pointer
   * \attention This method has to be inlined!
   **/
  void set_sp(void* new_sp) { 
    __asm__ volatile("mov r13,%0"
		     : /* no output */
		     : "r" (new_sp)); 
  }

  /**
   * \brief Switch from system mode stack to interrupt stack
   *
   * We use the supervisor mode to execute the dsrs, this is
   * necessary as we would destroy the link register when we
   * execute them on the interrupt stack and get interrupted
   * while executing them. This is not perfect, as we cannot
   * cause traps inside a dsr for now, so maybe we better use
   * abort mode or the undefined mode
   **/
  void to_dsr_stack() {
    __asm__ volatile("mrs   r0,cpsr\n"
		     "mov   r1,#0x1F\n"
		     "mvn   r1,r1\n"
		     "and   r0,r0,r1\n"
		     "mov   r1,#0x13\n"
		     "orr   r0,r0,r1\n"
		     "msr   cpsr,r0\n");
  }

  /**
   * \brief Switch from interrupt stack to system mode stack
   **/
  void from_dsr_stack() {
    __asm__ volatile("mrs   r0,cpsr\n"
		     "orr   r0,r0,#0x1f\n"
		     "msr   cpsr,r0\n");
  }
};

#endif /* __cpu_implementation_h__ */
