/* -*- c++ -*- ***************************************************************/
/* Echtzeitsysteme                                                           */
/*---------------------------------------------------------------------------*/
/*                                                                           */
/*                                    T M                                    */
/*                                                                           */
/*---------------------------------------------------------------------------*/
/* The 16-bit timers of the GBA                                              */
/*---------------------------------------------------------------------------*/

#ifndef __devices_tm_h__
#define __devices_tm_h__

#include "devices/derivate.h"
#include "devices/interrupt_controller.h"
#include "devices/interrupt_timer.h"
#include "infra/memory_mapped.h"
#include "infra/types.h"
#include "interrupt/plugbox.h"

/**
 * \class TM
 * \brief The baseclass for the 16-bit timers of the GBA
 **/
template < ezstubs_uint32 OFFSET, ezstubs_uint32 VECTOR > class TM : public Interrupt_Timer {

  Memory_Mapped< ezstubs_uint16,OFFSET + 0x0 > TMCNT_L;
  Memory_Mapped< ezstubs_uint16,OFFSET + 0x2 > TMCNT_H;

  /**
   * \brief The prescalers available for the GBA timers
   **/
  static const ezstubs_uint32 prescalers[];

  /**
   * \brief Does the timer currently periodically generate interrupts or not
   **/
  bool periodic;

  /**
   * \brief Need to correct time measurement results.
   *
   * The GBA Timers support the periodic generation of interrupts only via a 
   * reload value, that is written into the counter register when the counter
   * overflows. This is problematic when an interrupt occurs during time 
   * measurement, especially when time measurement is startet in a isr/dsr and
   * stopped in a subsequent activation of the isr/dsr.
   * At occurrence of the interrupt the reload value is written into the counter
   * register, so the recorded start and stop values of the measurement will not
   * differ a lot, except the reload value changed in the meanwhile. All in all
   * the measurement is useless. In order to facilitate time measurements also
   * across interrupts, we maintain this variable to record the time that has
   * vanished so far. This variable is updated by acknowledge() (as this method
   * is called on every interrupt) and taken into account by get_time().
   *
   * \see reload
   * \see start_time()
   * \see get_time()
   * \see acknowledge()
   **/
  tick_type correction_ticks;

  /**
   * \brief The current reload value of the counter
   *
   * In order to update the variable correction_ticks appropriately we have
   * to know the current reload value of the timer.
   *
   * \see correction_ticks
   * \see start_time()
   * \see get_time()
   * \see acknowledge()
   **/
  tick_type reload;

protected:

  /**
   * \brief Constructor
   **/
  TM() : Interrupt_Timer(VECTOR),periodic(false),correction_ticks(0) {}

public:

  tick_type max_value() { 
    return 0xFFFF; 
  }

  tick_type value() { 
    return TMCNT_L; 
  }

  void start() {
    TMCNT_H |= 0x80;
  }

  void stop() {
    TMCNT_H &= ~0x80;
  }

  ns_type tick(ns_type ns);

  us_type period() { return Interrupt_Timer::period(); }
  us_type period(us_type period);

  ns_type min_period();
  us_type max_period();

  /**
   * \brief Start time measurement
   *
   * Every time we start a time measurement, we can reset correction_ticks.
   * start_time() marks the beginning of time measurement, so no interrupt has
   * been occured during this time measurement so far.
   *
   * \see reload
   * \see correction_ticks
   * \see get_time()
   * \see acknowledge()
   **/
  void start_time() {
    Timer::start_time();
    correction_ticks = 0;
  }

  us_type get_time();

  /**
   * \brief Trigger the generation of an interrupt request
   *
   * This method is needed to implement the functionality of
   * a test interrupt
   **/
  void trigger() {
    period(10);
    trigger_single();
  }

  /**
   * \brief Check whether an interrupt request has been generated
   *
   * This method is needed to implement the functionality of
   * a test interrupt
   **/
  bool is_triggered() {
    return ic.vector_raised(VECTOR);
  }

  void trigger_single() {
    periodic = false;
    ic.unmask_vector(VECTOR);
    TMCNT_H |= 0x40;
    TM::start();
  }

  void trigger_periodical() {
    periodic = true;
    ic.unmask_vector(VECTOR);
    TMCNT_H |= 0x40;
    TM::start();
  }

  void cancel() {
    ic.mask_vector(VECTOR);
    TMCNT_H &= ~0x40;    
    TM::stop();    
  }

  /**
   * \brief Reset the interrupt request
   *
   * As the GBA Timers do not support single shot operation in hardware, we
   * emulate this by a flag. If the timer is supposed to generate a single
   * interrupt, we cancel interrupt generation after the first interrupt.
   *
   * For correct time measurement we have to update correction_ticks here.
   * We accumulate the ticks that elapsed since the call to start_time or
   * the last interrupt. We set start_value to reload as counting operation
   * begins from the reload value after an overflow (i.e. after an interrupt)
   *
   * \see reload
   * \see correction_ticks
   * \see start_time()
   * \see get_time()
   * \see acknowledge()
   **/
  void acknowledge() {
    if(!periodic) {
      cancel();
    }
    
    ic.acknowledge_vector(VECTOR);

    correction_ticks += max_value() - start_value;
    start_value = reload;
  }
};

template < ezstubs_uint32 OFFSET, ezstubs_uint32 VECTOR > 
const ezstubs_uint32 TM< OFFSET,VECTOR >::prescalers[] = {1,64,256,1024};

template < ezstubs_uint32 OFFSET, ezstubs_uint32 VECTOR > 
ns_type TM< OFFSET,VECTOR >::tick(ns_type t) {
  ns_type mod_tick = cpu.module_tick();
  ezstubs_uint8 i = 0;

  for(;i < 4;i++) {
    if(t <= mod_tick * prescalers[i]) {
      break;
    }
  }
  
  tick_duration = mod_tick * prescalers[i];
  TMCNT_H = (TMCNT_H & 0xFFFC) | i;

  return tick_duration;
}

template < ezstubs_uint32 OFFSET, ezstubs_uint32 VECTOR > 
us_type TM< OFFSET,VECTOR >::period(us_type p) {
  ns_type mod_tick = cpu.module_tick();
  ezstubs_uint16 ticks = 0;
  ezstubs_uint8 i = 0;

  /**
   * computing the prescaler
   **/
  for(;i < 4;i++) {
    us_type max_period = (us_type)(((ezstubs_uint32)max_value() * (ezstubs_uint32)mod_tick * prescalers[i]) / 1000);
    if(p <= max_period) {
      break;
    }
  }
  
  /**
   * computing the reload value and the final period and tick duration
   **/
  ticks = (p * 1000) / (mod_tick * prescalers[i]);
  reload = max_value() - ticks;
  period_duration = (us_type)(((ticks * mod_tick) * prescalers[i]) / 1000);
  tick_duration = mod_tick * prescalers[i];

  /**
   * initialize the I/O registers
   **/
  TMCNT_L = reload;
  TMCNT_H = (TMCNT_H & 0xFFFC) | i;

  return period_duration;
}

template < ezstubs_uint32 OFFSET, ezstubs_uint32 VECTOR > 
ns_type TM< OFFSET,VECTOR >::min_period() {
  return cpu.module_tick() * prescalers[0];
}

template < ezstubs_uint32 OFFSET, ezstubs_uint32 VECTOR > 
us_type TM< OFFSET,VECTOR >::max_period() {
  ns_type mod_tick = cpu.module_tick();
  us_type max_period = (max_value() / 1000) * mod_tick * prescalers[3];

  return max_period;
}

template < ezstubs_uint32 OFFSET, ezstubs_uint32 VECTOR > 
ns_type TM< OFFSET,VECTOR >::get_time() {
  /**
   * we do not really have to take care about overflows within get_time() 
   * anymore, as this already taken care of by the maintainance of 
   * correction_ticks
   **/
  tick_type ticks = stop_value - start_value + correction_ticks;
  us_type result = (ticks * Timer::tick()) / 1000;

  return result;
}

#endif /* __devices_tm_h__ */
