/* -*- c++ -*- ***************************************************************/
/* Echtzeitsysteme                                                           */
/*---------------------------------------------------------------------------*/
/*                                                                           */
/*                                  G A T E                                  */
/*                                                                           */
/*---------------------------------------------------------------------------*/
/* Klasse von Objekten, die in der Lage sind, Unterbrechungen zu behandeln.  */
/*****************************************************************************/

#ifndef __Gate_include__
#define __Gate_include__

#include "object/chain.h"

/**
 * \class Gate
 * \brief The interface class for interrupt-handlers.
 *
 * The interrupt-synchronization implemented in EZ-Stubs is the so called
 * isr-/dsr-interrupt-synchronization. Every interrupt-handler provides
 * at least an dsr that is executed on interrupt-level. From within the
 * isr no kernel functions may be called. If the interupt-handler needs
 * to access kernel-functionality, the dsrs can activate a dsr. This
 * dsr is executed
 *  - immediately after the isr, if the isr did not interrupt a
 *    secured section
 *  - when a secured section is left, if the isr did interrupt a secured
 *    section
 * The dsr now may call kernel-functions, except those, that may block.
 */

class Gate : public Chain {

  /**
   * \brief Is there a dsr pending for this interrupt-handler
   */
  bool dsr_queued;

public:

  /**
   * \brief Constructor
   *
   * Initially the dsr is not queued
   **/
  Gate() : dsr_queued(false) {}
      
  /**
   * \brief Destructor
   *
   * Only needed to prevent compiler warnings due to virtual methods
   **/ 
  virtual ~Gate() {}

  /**
   * \brief The isr
   * \retval true Queue an dsr
   * \retval false Do not queue an dsr.
   *
   * This routine is executed immediately after the corresponding interrupt
   * occured. It is executed on interrupt level and any calls to kernel-
   * functions are forbidden.
   */
  virtual bool isr() = 0;

  /**
   * \brief The dsr
   *
   * This routine is executed under the control of the operating system. Calls
   * to the kernel are allowed except for those that may block.
   */
  virtual void dsr() = 0;


  /**
   * \brief Reset the interrupt source.
   *
   * In the most cases resetting the interrupt source means clearing the
   * interrupt request bit of a specific interrupt source within a status
   * and control register of some peripheral hardware.
   */
  virtual void acknowledge() = 0;

  /**
   * \brief Set the pending-status of the dsr.
   */
  void queued(bool q) {
    dsr_queued = q;
  };

  /**
   * \brief Get the pending-status of the dsr.
   */
  bool queued() {
    return dsr_queued;
  };
};

#endif
