/*****************************************************************************/
/* Echtzeitsysteme                                                           */
/*---------------------------------------------------------------------------*/
/*                                                                           */
/*                              C O N T E X T                                */
/*                                                                           */
/*---------------------------------------------------------------------------*/
/* Routines for maintaining a threads context.                               */
/*---------------------------------------------------------------------------*/

#ifndef __context_h__
#define __context_h__

#include "thread/runnable.h"

/**
 * \file context.h
 * \brief Routines for managing a thread's context.
 **/

extern "C" {

  /**
   * \brief Initialize the stack for a flow of control,t.m. initialize its
   *        context.
   * \param sp A pointer to the stack to be used.
   * \param kickoff The address of the function kickoff, any flow of control
   *                start in the function kickoff.
   * \param object A pointer to a object of type runnable representing the
   *               flow of control.
   * \attention Within this function only a minimal part of the stack is set up,
   *            in the normal case these are the address of the kickoff routine,
   *            the parameter for the kickoff routine and the interrupt state of
   *            machine. Non-volatile registers are not set up - their content
   *            is not specified in order to prevent a corruption of the current
   *            runtime stack in some cases. But nevertheless, the stack pointer
   *            has to be modified in such so it can be passed to switch_context.
   * \see switch_context
   **/
  void init_context(void** sp,void (*kickoff)(Runnable*),void* object);

  /**
   * \brief Load the context of a thread from the stack and start the thread.
   * \param sp A pointer to the threads context.
   * \attention This method expects that the context is properly initialized
   *            by init_context().
   * \see init_context
   **/
  void load_context(void* sp);

  /**
   * \brief Switch between to flows of control by exchanging their contexts.
   *        First save the context of the currently running thread to the
   *        stack, then load the context of the next thread.
   * \param sp_now The address of the pointer to the stack of the currently running
   *        thread. After switching threads this pointer points to the context
   *        of the thread that released the cpu. The context of this thread will be
   *        stored on the stack.
   * \param sp_then A pointer to the context that should be loaded.
   * \attention The stack of the thread that should be started, must be
   *            initialized properly, either by init_context or by
   *            switch_context.
   * \see init_context
   **/
  void switch_context(void** sp_now,void* sp_then);
}

#endif /* __context_h__ */
