/* -*- c++ -*- ***************************************************************/
/* Echtzeitsysteme                                                           */
/*---------------------------------------------------------------------------*/
/*                                                                           */
/*           S C H E D U L E R _ I M P L E M E N T A T I O N                 */
/*                                                                           */
/*---------------------------------------------------------------------------*/
/* Interface to the implementation of the scheduler, in this case of a       */
/* scheduletable scheduler.                                                  */
/*****************************************************************************/

#ifndef __schedule_table_scheduler_h__
#define __schedule_table_scheduler_h__

#include "infra/types.h"

#include "devices/scheduler_implementation_timer.h"
#include "thread/abstract_schedule_table.h"
#include "thread/scheduler_base.h"
#include "thread/thread.h"

/**
 * \class Schedule_Table_Scheduler
 * \brief The interface of the scheduletable scheduler.
 *
 * The principle of the schedule table scheduler is very easy - it just executes
 * a given static schedule. A schedule is a list of tuples, each tuple consists
 * of a thread and an offset. The thread is the thread to be started and the 
 * offset tells the scheduler when the thread should be started counted from the
 * beginning of the schedule table.
 *
 * This implementation of the schedule table scheduler does not take care of any
 * thread that is currently running, when a new thread is started. The currently
 * running thread will quasi be killed and the new thread is executed. This has
 * the advantage that all threads of a schedule table can be executed on the same
 * stack, also no context saving overhead is created.
 */
class Schedule_Table_Scheduler : public Scheduler_Implementation_Timer, public Scheduler_Base {

  bool                     start_first_thread;

protected:

  /**
   * \brief The schedule table that is currently executed
   **/
  Abstract_Schedule_Table* current_schedule_table;

  /**
   * \brief The schedule table that will be executed next
   * \see Schedule_Table_Scheduler::switch_schedule_table
   **/
  Abstract_Schedule_Table* next_schedule_table;

  /**
   * \brief Determine the next thread to run.
   * \return The thread that should be resumed next.
   */ 
  Thread*                  schedule();

public:

  /**
   * \brief Constructor
   **/
  Schedule_Table_Scheduler();

  /**
   * \brief Set the schedule should be executed.
   * \param schedule_table A pointer to the schedule that should be executed.
   * \attention Must be called before start()!
   * \attention May only be called before start()!
   * \see Scheduler_Implementation::start()
   */
  void set_schedule_table(Abstract_Schedule_Table* schedule_table);

  /**
   * \brief Get the schedule that is currently executed.
   * \return The currently executed schedule.
   */
  Abstract_Schedule_Table* get_schedule_table();

  /**
   * \brief Execute a different schedule.
   * \param schedule_table The schedule that should be executed next.
   * \return The currently executed schedule.
   * \attention May only be called after start()!
   * \attention The currently executed schedule may not be altered!
   *
   * The schedules are not exchanged immediately after switch_schedule_table() has been
   * called. The current schedule is executed until the current round is finished, then
   * the switch occurs and the new schedule will be executed.
   */
  Abstract_Schedule_Table* switch_schedule_table(Abstract_Schedule_Table* schedule_table);

  /**
   * \brief Start the scheduler.
   * \attention May only be called once!
   * \attention May not be called before set_schedule_table()!
   * \attention This method does not return!
   * \see Scheduler_Implementation::set_schedule_table()
   *
   * Begin to execute the schedule set be set_schedule_table()
   */
  void                     start();

  /**
   * \brief Stop the currently running thread.
   */
  void                     exit();

  /**
   * \brief Determinte the next thread to be executed and switch threads.
   * \see Guard::leave()
   * \see Scheduler_Implementation::get_need_reschedule()
   * This method is called by the system when leaving a secured section and
   * scheduling is necessary. The new thread will be started, regardless whether
   * another thread is currently running. The preempted thread is quasi killed
   * it will not be resumed, when the started thread finishes execution. The
   * next time the killed thread is started according to the schedule table it
   * will start from the beginning of the thread.
   */
  void                     reschedule();

  /**
   * The isr dealing with the timer interrupts.
   */
  bool                     isr();

  /**
   * The dsr dealing with the timer interrupts.
   */
  void                     dsr();
};

#endif /* __schedule_table_scheduler_h__ */
