/* -*- c++ -*- ***************************************************************/
/* Echtzeitsysteme                                                           */
/*---------------------------------------------------------------------------*/
/*                                                                           */
/*                                   M U T E X                               */
/*                                                                           */
/*---------------------------------------------------------------------------*/
/* A mutex implementation based upon the stack based priority ceiling        */
/* protocol.                                                                 */
/*****************************************************************************/

#ifndef __mutex_h__
#define __mutex_h__

#include "infra/types.h"
#include "object/chain.h"

class Mutex_Thread;

/**
 * \class Mutex
 * \brief Mutex implementation based on the stack based priority ceiling 
 *        protocol.
 * \see Mutex_Thread
 *
 * The stack based priority ceiling protocol is implemented as follows:
 * - when a thread tries to lock a mutex (and of course the priority stuff is
 *   ok), the thread will always successfully lock the mutex (this directly
 *   follows from the stack based priority ceiling protocol
 * - the thread inherits the ceiling priority of the mutex
 * - therefore the thread is added as first thread to the queue belonging to 
 *   ceiling priority of the mutex
 * - the mutex is added to a stack of mutexes locked by this thread, this is
 *   necessary to restore the correct priority if the thread holds more than
 *   one mutex
 * - when the thread unlocks the mutex, its foregoing priority is restored (this
 *   may also be the ceiling priority of another mutex!) and the mutex is removed
 *   from the stack of held mutexes.
 * - in order to minimize the overhead, not every kind of thread is capable of
 *   locking mutexes, only threads derived from class Mutex_Thread are
 **/
class Mutex : public Chain {

public:

  /**
   * \brief Get the ceiling priority of the mutex
   * \return The ceiling priority of the mutex
   **/
  ezstubs_uint8 get_priority() {
  }

  /**
   * \brief Lock the mutex
   * \retval true The mutex has successfully been locked
   * \retval false The mutex has not been locked successfully, because either
   *               the thread is not enlisted for this mutex, or another mutex
   *               with a higher ceiling priority has already been taken.
   *
   * By calling this method a critical section is entered. It is ensured that
   * other flow of control can succesfully call this method as long as the
   * mutex has not been unlocked by calling the unlock()-method. The precise
   * semantics of this method however depend on the particular synchronization
   * protocol implemented by the mutex.
   **/
  bool lock();

  /**
   * \brief Unlock the mutex
   * \retval true The mutex has successfully been locked
   * \retval false The mutex has not been locked successfully, because the mutex
   *               was either not locked at all or another mutex has to be 
   *               unlocked before.
   *
   * By calling this method a critical section is locked other flows of control
   * are now able to successfully call the method lock() and enter this critical
   * section on their own. As for the method() lock() the precise semantics of
   * this method are defined by the implemented synchronization protocol.
   **/
  bool unlock();

  /**
   * \brief Register with the mutex
   * \param thread A thread that wants to use the mutex.
   *
   * In order to compute the proper ceiling priority it has be known in
   * advance which thread will try to lock a specific mutex. Therefore
   * all threads that want to use a particular mutex first have to be
   * registered with the mutex.
   **/
  void enlist(Mutex_Thread* thread);
};

#endif /* __mutex_h__ */
