#ifndef LIBEZS_SB16_H
#define LIBEZS_SB16_H

/*!
 * @file ezs_sb16.h
 * @brief a driver for the Sound Blaster 16
 * @author Florian Franzmann
 * \ingroup hw
 */
#include <stdint.h>
#include "ezs_dma.h"
#include "ezs_common.h"

/*
 * DSP commands
 */
const uint8_t DSP_8BIT_DMA_PAUSE;
const uint8_t DSP_8BIT_DMA_CONTINUE;

const uint8_t DSP_16BIT_DMA_PAUSE;
const uint8_t DSP_16BIT_DMA_CONTINUE;

const uint8_t DSP_16BIT_DMA_EXIT;
const uint8_t DSP_8BIT_DMA_EXIT;

const uint8_t DSP_VERSION;

const uint8_t DSP_READY;
const uint8_t DSP_SPEAKER_ON;
const uint8_t DSP_SPEAKER_OFF;

/*
 * mixer commands
 */
const uint8_t MIXER_MASTER_LEFT;
const uint8_t MIXER_MASTER_RIGHT;
const uint8_t MIXER_VOC_LEFT;
const uint8_t MIXER_VOC_RIGHT;
const uint8_t MIXER_MIDI_LEFT;
const uint8_t MIXER_MIDI_RIGHT;
const uint8_t MIXER_CD_LEFT;
const uint8_t MIXER_CD_RIGHT;
const uint8_t MIXER_LINE_LEFT;
const uint8_t MIXER_LINE_RIGHT;
const uint8_t MIXER_MIC_VOLUME;
const uint8_t MIXER_PC_SPEAKER;
const uint8_t MIXER_OUTPUT_SWITCHES;
const uint8_t MIXER_INPUT_LEFT;
const uint8_t MIXER_INPUT_RIGHT;
const uint8_t MIXER_INPUT_GAIN_LEFT;
const uint8_t MIXER_INPUT_GAIN_RIGHT;
const uint8_t MIXER_OUTPUT_GAIN_LEFT;
const uint8_t MIXER_OUTPUT_GAIN_RIGHT;
const uint8_t MIXER_AGC;
const uint8_t MIXER_TREBLE_LEFT;
const uint8_t MIXER_TREBLE_RIGHT;
const uint8_t MIXER_BASS_LEFT;
const uint8_t MIXER_BASS_RIGHT;

#define DMA_SAMPLECOUNT 1024
#define DMA_BUFFERLENGTH (DMA_SAMPLECOUNT * 4);


typedef struct SB16 {
  int16_t card_id;
  int16_t io_addr;
  int16_t interrupt;
  uint8_t DMA_no;
  uint8_t h_DMA_no;

  int16_t DSP_reset;
  int16_t DSP_read;
  int16_t DSP_write;
  int16_t mixer_addr;
  int16_t mixer_data;
  int16_t DSP_int_ack;
  int16_t DSP_status;

  uint8_t major_version;
  uint8_t minor_version;

  EZS_DMA DMA;
  EZS_DMA h_DMA;
} SB16;

typedef struct SB16_Mixer {
  uint8_t master_left;
  uint8_t master_right;
  uint8_t voc_left;
  uint8_t voc_right;
  uint8_t midi_left;
  uint8_t midi_right;
  uint8_t cd_left;
  uint8_t cd_right;
  uint8_t line_left;
  uint8_t line_right;
  uint8_t mic_volume;
  uint8_t pc_speaker;
  uint8_t output_switches;
  uint8_t input_switches_left;
  uint8_t input_switches_right;
  uint8_t input_gain_left;
  uint8_t input_gain_right;
  uint8_t output_gain_right;
  uint8_t output_gain_left;
  uint8_t agc;
  uint8_t treble_left;
  uint8_t treble_right;
  uint8_t bass_left;
  uint8_t bass_right;
} SB16_Mixer;

/*!
 * \brief initialize an SB16 object
 *
 * \param sb pointer to the SB16 object
 * \param io_addr IO port address for communcating with the DSP
 * \param interrupt for DMA
 * \param DMA number of low DMA
 * \param h_DMA number of high DMA
 */
void ezs_sb16_init(SB16 *sb,
                   int16_t io_addr,
                   int16_t interrupt,
                   int16_t DMA,
                   int16_t h_DMA);

/*!
 * \brief reset the SB16. should take no more than 100 microseconds
 */
void ezs_sb16_reset_DSP(SB16 *sb);

/*!
 * \brief acknowledge an interrupt
 */
void ezs_sb16_acknowledge_irq(SB16 *sb);

/*!
 * \brief write a byte to the DSP
 */
void ezs_sb16_write_DSP(SB16 *sb, uint8_t value);

/*!
 * \brief read a byte from the DSP
 */
uint8_t ezs_sb16_read_DSP(SB16 *sb);

/*!
 * \brief write settings to the mixer.
 * use ezs_sb16_get_mixer_settings() to fetch the current settings, change values, then use
 * this function to write the new settings.
 */
void ezs_sb16_set_mixer_settings(SB16 *sb, SB16_Mixer *settings);

/*!
 * \brief read settings from the mixer.
 */
void ezs_sb16_get_mixer_settings(SB16 *sb, SB16_Mixer *settings);

/*!
 * \brief play a sample in direct io mode. Only 8bit mono at an arbitrary rate is supported in this mode.
 * \param sb a pointer to an SB16 object
 * \param sample the sample to send to the DSP
 */
void ezs_sb16_play_sample(SB16 *sb, uint8_t sample);

void ezs_sb16_setup_mode(
        SB16 *sb,
        void *buffer,
        uint16_t block_size);
void ezs_sb16_setup_mode(
        SB16 *sb,
        void *buffer,
        uint16_t block_size);

/*!
 * \brief set the output samping rate when using DMA
 * \param sb a pointer to an SB16 object
 * \param rate the output rate, must be in the range of [5000, 44100] Hz
 */
void ezs_sb16_set_output_rate(SB16 *sb, uint16_t rate);

/*!
 * \brief set up the DSP for DMA
 *
 * \param sb a pointer to an SB16 object
 * \param buffer a buffer for the samples
 * \param block_size the number of samples to be transferred
 *        If an 8KB DMA buffer is used, the DSP block transfer size should be
 *        set to 4K of 8bit samples for 8bit data transfer, 2K of 16bit samples
 *        for 16bit data transfer. At the end of every block transfer the DSP
 *        will generate an interrupt to the application.
 *
 * How to set up a DMA transfer:
 * 1. If using auto-initialize DMA, allocate a DMA buffer in contiguous memory
 *    without straddling a 64KB physical page boundary.
 * 2. Set up the DSP interrupt service routine.
 * 3. Enable the interrupt used.
 * 4. Program the DMA controller for 8bit or 16bit DMA single cycle or auto-initialize mode.
 * 5. call ezs_sb16_set_output_rate()
 * 6. call ezs_sb16_setup_DMA()
 *
 * In single cycle mode the following steps should be taken when receiving an
 * interrupt from the DSP:
 *
 * 1. Program the DMA controller for the next block.
 * 2. Program the DSP for the next block using ezs_sb16_setup_DMA().
 *
 * At the end of data transfer:
 *
 * 1. Disable the interrupt used.
 * 2. Restore the original interrupt service routine.
 *
 *
 * In auto-initialize mode, upon receiving an interrupt from the DSP, the
 * following steps should be performed in the interrupt service routine:
 *
 * 1. Transfer data between the DMA buffer and the storage buffer.
 *
 * If you want to exit auto-initialize DMA,
 * 2a. call ezs_sb16_exit_8bit_autoinitialize_DMA() or
 *     ezs_sb16_exit_16bit_autoinitialize_DMA()
 * 2b. or program the DSP for single cycle DMA mode.
 *
 * At the end of the data transfer:
 * 1. Disable the interrupt used.
 * 2. Restore the original interrupt service routine.
 * 3. Release the allocated DMA buffer.
 *
 */
void ezs_sb16_setup_DMA(
    SB16 *sb,
    void *buffer,
    uint16_t block_size);
/*!
 * \brief exit 8bit auto-initialize DMA mode.
 */
void ezs_sb16_exit_8bit_DMA(SB16 *sb);

/*!
 * \brief exit 16bit auto-initialize DMA mode.
 */
void ezs_sb16_exit_16bit_DMA(SB16 *sb);

/*!
 * \brief stop playing anything.
 */
void ezs_sb16_stop_sound(SB16 *sb);
#endif // LIBEZS_SB16_H
