#ifndef EZS_DMA_H_INCLUDED
#define EZS_DMA_H_INCLUDED
#ifdef __cplusplus
extern "C" {
#endif

/*!
 * @file ezs_dma.h
 * @brief a driver for the i386's DMA unit
 * @author Florian Franzmann
 * \ingroup hw
 */
#include <stdint.h>
#include "ezs_common.h"

extern const uint8_t DMA_DEMAND_MODE;
extern const uint8_t DMA_SINGLE_MODE;
extern const uint8_t DMA_BLOCK_MODE;
extern const uint8_t DMA_CASCADE_MODE;

extern const uint8_t DMA_ADDRESS_INCREMENT;
extern const uint8_t DMA_ADDRESS_DECREMENT;

extern const uint8_t DMA_AUTO_INIT;
extern const uint8_t DMA_SINGLE_CYCLE;

extern const uint8_t DMA_VERIFY_TRANSFER;
extern const uint8_t DMA_WRITE_TRANSFER;
extern const uint8_t DMA_READ_TRANSFER;

typedef struct EZS_DMA {
  uint8_t channel;

  uint8_t mode_byte;
  uint8_t control_byte;

  uint8_t address_port;
  uint8_t port_address;
  uint8_t count_port;
  uint8_t page_port;

  uint8_t mask_register;
  uint8_t clear_register;
  uint8_t mode_register;

  int16_t transfer_length;
  int16_t page;
  uint16_t mask_save;
  uint32_t phys;
  
  uint8_t eight_bit;
} EZS_DMA;

typedef enum EZS_DMA_Initialization {
  EZS_DMA_Single_Cycle,
  EZS_DMA_Auto_Initialization
} EZS_DMA_Initialization;

typedef enum EZS_DMA_Mode {
  EZS_DMA_Demand,
  EZS_DMA_Single,
  EZS_DMA_Block,
  EZS_DMA_Cascade
} EZS_DMA_Mode;

typedef enum EZS_DMA_Direction {
  EZS_DMA_Decrement,
  EZS_DMA_Increment
} EZS_DMA_Direction;

typedef enum EZS_DMA_Transfer_Type {
  EZS_DMA_Verify,
  EZS_DMA_Write,
  EZS_DMA_Read
} EZS_DMA_Transfer_Type;

/*!
 * \brief initialize an EZS_DMA object
 *
 * usage:
 *
 * EZS_DMA dma;
 * ezs_dma_init(&dma);
 *
 * -- disable interrupts --
 *
 * ezs_dma_disable_channel(&dma);
 * ezs_dma_set_mode_byte(&dma, channel, mode_select, address_increment_decrement, auto_initialize_bit, transfer_type_bit);
 * ezs_dma_clear_flip_flop(&dma);
 * ezs_dma_set_buffer_info(&dma);
 *
 * -- enable interrupts --
 *
 * ezs_dma_enable_channel(&dma);
 *
 */
void ezs_dma_init(EZS_DMA *dma);
void ezs_dma_set_mode(EZS_DMA *dma,
                              uint8_t channel,
                              EZS_DMA_Mode mode,
                              EZS_DMA_Direction direction,
                              EZS_DMA_Initialization auto_initialize,
                              EZS_DMA_Transfer_Type transfer_type);
void ezs_dma_enable_channel(EZS_DMA *dma);
void ezs_dma_disable_channel(EZS_DMA *dma);
void ezs_dma_clear_flip_flop(EZS_DMA *dma);
void ezs_dma_set_buffer_info(EZS_DMA *dma, uint8_t page, void *buffer, uint16_t length);


#ifdef __cplusplus
}
#endif
#endif // EZS_DMA_H_INCLUDED
