#include "lwt.h"

#include <pthread.h>
#include <stdlib.h>
#include <stdint.h>
#include <unistd.h>
#include <errno.h>

#include "util/die.h"
#include "sched/scheduler.h"
#include "sched/thread.h"
#include "arch/context.h"

static pthread_t *threads = 0;
static uint32_t nthreads = 0;
static lwt::Thread *kill_threads = 0;

static void lwt_schedule()
{
	lwt::Scheduler::switchToLoop();
	for(uint32_t i = 1; i != nthreads; ++i) {
		pthread_join(threads[i], NULL);
	}

	free(threads);
	free(kill_threads);
}

static void kill_func(void*, void*)
{
	lwt::Scheduler::returnFromLoop();
	lwt::fatal("returned from \"returnFromLoop\" call");
}

static void* thread_start(void *unused)
{
	(void)unused;
	lwt::Scheduler::threadLocalInit();
	lwt::Scheduler::switchToLoop();
	return NULL;
}

static int lwt_argc = 0;
static char **lwt_argv = 0;
static int *lwt_main_ret = 0;
static void(*lwt_start_func)(int, char**, int*) = 0;

void lwt_destroy()
{
	for(uint32_t i = 0; i != nthreads; ++i) {
		kill_threads[i].init(kill_func, 0, 0);
		lwt::Scheduler::readyThread(&kill_threads[i]);
	}
}

static void lwt_first_thread(void *a, void *b)
{
	(void)a;
	(void)b;

	lwt_start_func(lwt_argc, lwt_argv, lwt_main_ret);
	lwt_destroy();
}

int lwt_run(lwt_func func, void *a, void *b)
{
	if(!lwt::Scheduler::readyFunc(func, a, b)) {
		return 1;
	}
	return 0;
}

int lwt_init(void(*start_func)(int, char**, int*), int argc, char *argv[], int *main_ret)
{
	lwt_start_func = start_func;
	lwt_argc = argc;
	lwt_argv = argv;
	lwt_main_ret = main_ret;

	if(!lwt::Scheduler::init()) {
		return -1;
	}

	const long processors = sysconf(_SC_NPROCESSORS_ONLN);
	//const long processors = 1;
	if(processors <= 0) {
		lwt::die("sysconf(_SC_NPROCESSORS_ONLN) = %ld", processors);
		return 1;
	}

	nthreads = processors;

	lwt::Scheduler::threadLocalInit();

	kill_threads = (lwt::Thread*)malloc(sizeof(lwt::Thread) * nthreads);
	if(kill_threads == NULL) {
		lwt::die("malloc");
		return 2;
	}

	threads = (pthread_t*)malloc(sizeof(pthread_t) * nthreads);
	if(threads == NULL) {
		lwt::die("malloc");
		return 2;
	}

	for(uint32_t i = 1; i != nthreads; ++i)
	{
		pthread_attr_t attr;
		errno = pthread_attr_init(&attr);
		if(errno != 0) {
			lwt::die("pthread_attr_init");
			return 3;
		}

		cpu_set_t cpuset;
		CPU_ZERO(&cpuset);
		CPU_SET(i % processors, &cpuset);

		errno = pthread_attr_setaffinity_np(&attr, sizeof(cpu_set_t), &cpuset);
		if(errno != 0) {
			lwt::die("pthread_attr_setaffinity_np");
			return 4;
		}

		errno = pthread_create(&threads[i], &attr, thread_start, NULL);
		if(errno != 0) {
			lwt::die("pthread_create");
			return 5;
		}

		errno = pthread_attr_destroy(&attr);
		if(errno != 0) {
			lwt::die("pthread_attr_destroy");
			return 6;
		}
	}

	lwt_run(lwt_first_thread, 0, 0);
	lwt_schedule();

	return 0;
}

extern "C" void lwt_main(int, char**, int*);
extern "C" int main(int argc, char *argv[]) __attribute__((weak));
int main(int argc, char *argv[])
{
	int ret = 0;
	lwt_init(lwt_main, argc, argv, &ret);
	return ret;
}
