#ifndef   LWT_SCHEDULER_HEADER
#define   LWT_SCHEDULER_HEADER

#include "lwt.h"

#include <semaphore.h>
#include <errno.h>

#include "util/die.h"
#include "mem/alloc.h"
#include "sched/thread.h"
#include "adt/simplequeue.h"
#include "spinlocks/readlock.h"
#include "arch/context.h"

extern "C" void lwt_save_and_switch(void*);

namespace lwt {

class Scheduler
{
	private:
	static sem_t readySem;
	static lwt::SimpleQueue<lwt::Thread> queue;
	static lwt::ReadSpinlock spinlock;
	static __thread lwt::Thread pthreadThread;

	public:
	static __thread lwt::Context *currentContext;
	static __thread lwt::Thread *currentThread;
	static __thread lwt::Thread dummy;
	static __thread void(*postBlockFunc)(void*);
	static __thread void *postBlockArg;
	static __thread lwt::Context *oldContext;

	static void signalReadySem()
	{
		if(sem_post(&readySem) != 0) {
			lwt::fatal("sem_post in signalReadySem");
		}
	}

	static void waitReadySem()
	{
		while(sem_wait(&readySem) != 0) {
			const int errnoVal = errno;
			if(errnoVal != EINTR) {
				lwt::fatal("sem_wait in waitReadySem");
			}
		}
	}

	static bool init()
	{
		if(sem_init(&readySem, 0, 0) != 0) {
			lwt::die("sem_init");
			return false;
		}

		queue.init();
		spinlock.init();

		return true;
	}

	static void threadLocalInit()
	{
		currentContext = 0;
		currentThread = &dummy;
	}

	static void readyThread(lwt::Thread *thread)
	{
		spinlock.lock();
		queue.enqueue(thread);
		spinlock.unlock();

		signalReadySem();
	}

	static bool readyFunc(lwt_func func, void *a, void *b)
	{
		lwt::Thread *thread = lwt::Alloc::thread();

		if(thread == 0) {
			return false;
		}

		thread->init(func, a, b);
		readyThread(thread);
		return true;
	}

	static lwt::Thread* getNext()
	{
		lwt::Thread *out;
		
		spinlock.lock();
		out = queue.dequeue();
		spinlock.unlock();

		return out;
	}

	static void block(void (*func)(void*), void *ptr)
	{
		lwt::Context *newContext = lwt::Context::make();
		if(newContext == 0) {
			lwt::fatal("lwt::Scheduler::block");
		}
	
		oldContext = currentContext;
		currentContext = newContext;
		postBlockFunc = func;
		postBlockArg = ptr;
	
		void *newStack = newContext->getTopOfStack();
	
		lwt_save_and_switch(newStack);
	}
	
	static void saveCurrentThread(void *unused)
	{
		(void)unused;
		if(currentThread->ptr2 == 0) {
			lwt::fatal("pthread stack address is 0");
		}
		pthreadThread.ptr1 = currentThread->ptr1;
		pthreadThread.ptr2 = currentThread->ptr2;
	}
	
	static void switchToLoop()
	{
		block(saveCurrentThread, 0);
	}
	
	static void returnFromLoop()
	{
		lwt_restore_context(pthreadThread.ptr1, pthreadThread.ptr2);
	}

	static void loop()
	{
		for(;;) {
			waitReadySem();
	
			currentThread = getNext();
			currentThread->func(currentThread->ptr1, currentThread->ptr2);
			lwt::Alloc::freeThread(currentThread);
			currentThread = 0;
		}
	}
};


} // namespace llwt

#endif /* LWT_SCHEDULER_HEADER */
