#ifndef _7SEG_H
#define _7SEG_H

#include <stdint.h>

#include "check.h"

/**
 * \addtogroup SEG 7seg (Seven Segment Display)
 *
 * \brief Controls the two 7-segment displays on the board
 *
 * The two 7-segment displays of the SPiCboard share one common
 * port of the MCU. The two displays can be connected and
 * disconnected from the port using two transistors. By quickly
 * and periodically connecting and disconnecting the displays
 * an observer will not be able to notice when a display is
 * disabled and both displays can be used apparently simultaneously.
 *
 * The module uses the 8-bit <b>Timer 2</b> of the ATmega328PB
 * to multiplex the two 7-segment displays
 *
 * \note As the timer is used, interrupts must be enabled for the display to work
 *       (if one of the 7 segment displays seems to be not working
 *       it is quite likely that interrupts are not enabled!)
 *
 * \sa timer.h
 *
 * @{
 * \file 7seg.h
 * \version \$Rev: 9414 $
 */

/**
 * \brief Prints a number in the range [-9; 99] on the 7-segment display
 *
 * \param nmbr  the number to print
 *
 * \retval  0   success
 * \retval -1   nmbr is smaller than -9
 * \retval -2   nmbr is greater than 99
 */
int8_t sb_7seg_showNumber(int8_t nmbr);

/**
 * \brief Prints the hexadecimal representation of an 8-bit unsigned integer on the 7-segment display
 *
 * \param nmbr  the number to print
 *
 * \retval 0    on success
 * \retval !0   on error
 */
int8_t sb_7seg_showHexNumber(uint8_t nmbr);

/**
 * \brief Prints a 2 character string on the 7-segment display
 *
 * Supported characters are in the group [-_ 0-9A-Za-z] (contains space).
 * Read <a href="http://en.wikipedia.org/wiki/Seven-segment_display_character_representations">this</a>
 * article for possible representations of these characters. Two
 * characters of the set should never have the same representation.
 * No differentiation is made between upper- and lowercase characters.
 *
 * \param str  the 0-terminated string
 *
 * \retval  0  success
 * \retval -1  character at position 0 not printable
 * \retval -2  character at position 1 not printable
 * \retval -3  both characters not printable
 * \retval -4  str is an empty string
 */
int8_t sb_7seg_showString(const char *str);

/**
 * \brief Set the LEDs of the 7-segment display manually
 *
 * The bitfield contains one bit for each of the 7 segment LEDs of a block.
 * A set bit enables and a cleared bit disables the corresponding LED.
 * The most significant bit determines the block:
 * If set, the first block (tens' place) will be used, if cleared the second block (ones' place)
 * \image html 7seg.png
 * For example a value of 0x86 (decimal 134, binary representation: 1000 0110) will enlight the LEDs 1 and 2 of the first block.
 *
 * \param mask  8-bit bitfield describing the desired 7 segment display state
 */
void sb_7seg_setMask(uint8_t mask);

/**
 * \brief Disables the 7-segment displays
 *
 * Any running alarms are unregistered.
 */
void sb_7seg_disable(void);

/** @}*/

#endif

