#ifndef CHECK_H
#define CHECK_H

#include <limits.h>

/**
 * \addtogroup Check
 *
 * \brief Check compiler settings
 *
 * Since GCC offers a bunch of different compiler settings, this header
 * tries to help detecting possible bugs (during compile time) which would
 * lead to incompatibility with the provided libspicboard.
 *
 * @{
 * \file check.h
 * \version \$Rev: 7715 $
 */

#define XSTR(s) STR(s)
#define STR(s) #s

// Check C Standard
#ifdef __STDC__
#if defined(__STDC_VERSION__) && (__STDC_VERSION__ < 201112L)
#if __STDC_VERSION__ >= 199901L
##pragma message ("You are using C99 Standard. Thats okay, but we would suggest to use C11")
#else
#error The minimum support Standard is C99!
#endif
#endif
#else
#error You are using a non standard C compiler
#endif

// Check GNU Extensions
#ifndef __GNUC__
#pragma message ("GCC Extensions are not available - you wont be able to use __flash")
#else
// Set compiler options
#pragma GCC optimize ("unsigned-char")
#pragma GCC optimize ("unsigned-bitfields")
#pragma GCC optimize ("short-enums")
#pragma GCC optimize ("pack-struct")
#endif

// Check CPU clock rate
#ifndef F_CPU
#define F_CPU 16000000UL
#pragma message ("CPU frequency defaults now to 16 MHz")
#elif F_CPU == 1000000UL
#pragma message ("You are using 1 MHz CPU frequency - please make sure that the libspicboard you are using was built for this frequency!")
#elif F_CPU != 16000000UL
#error The CPU Frequency is not supported!
#endif

// Check unsigned char
#if ! defined(__CHAR_UNSIGNED__) || CHAR_MIN < 0
#error chars are signed - you should use unsigned for the SPiCboard (-funsigned-char)
#endif

/**
 * \def CHECK_ENUM_SIZE(VAR, LEN)
 *
 * Aborts compilation if the size of VAR is not euqal LEN
 */
#define CHECK_ENUM_SIZE(VAR, LEN) _Static_assert(sizeof(VAR) == (LEN), "enum " STR(VAR) " has the wrong size and therefore it is not compatible to the libspicboard; check your compiler flags (-fshort-enums)");

/** @}*/

#endif
