MAKEFILE_DIR := $(shell dirname $(realpath $(lastword $(MAKEFILE_LIST))))

LIBSPICBOARDDIR ?= $(MAKEFILE_DIR)
SPICSIMJAR      ?= $(addprefix $(wildcard $(LIBSPICBOARDDIR)/doc/),SPiCsim.jar)

CC   := avr-gcc
SIZE := avr-size

ATMEGA_MODEL = 328pb

AVRDUDE ?= avrdude
AVRDUDE_CONF ?= $(MAKEFILE_DIR)/doc/atmega$(ATMEGA_MODEL).conf
AVRDUDE_FLAGS := -p m$(ATMEGA_MODEL) -c xplainedmini -P usb

ifneq ("$(wildcard $(AVRDUDE_CONF))","")
	AVRDUDE_FLAGS += -C +$(AVRDUDE_CONF)
endif

# Warning! -fwhole-program breaks the linkage of additional files
CFLAGS = -Os -ffreestanding -mmcu=atmega$(ATMEGA_MODEL) -std=gnu11 -funsigned-char -funsigned-bitfields -fshort-enums -fpack-struct -ffunction-sections -fdata-sections -Wextra -std=gnu11 -Wl,--gc-sections -Wall -Werror -pedantic -pedantic-errors -DF_CPU=16000000 -I. -I./libspicboard/ -I$(LIBSPICBOARDDIR)
LDFLAGS ?= -L. -L./libspicboard/ -L$(LIBSPICBOARDDIR) -lspicboard -u sb_init

# Check if avr-gcc supports Atmega model out-of-the-box
ifeq "$(shell avr-gcc --target-help | grep atmega$(ATMEGA_MODEL))" ""
	# Otherwise we use the shipped definitions
	CFLAGS += -B$(LIBSPICBOARDDIR) -B. -D__AVR_DEV_LIB_NAME__=m$(ATMEGA_MODEL)
endif

.PHONY: help clean %.flash %.sim

help::
	@echo
	@/bin/echo -e "\e[1mSPiCboard v3 (generic) Makefile\e[0m"
	@echo
	@/bin/echo -e "\tUsage: \e[3mmake\e[0m [target]"
	@echo
	@echo "Target could either be"
	@echo "	clean         (remove all files created by this Makefile)"
	@echo "	help          (show this)"
	@echo "or consist of the name of a source file and one of the following options:"
	@echo "	[name].flash  (flash to connected SPiCboard)"
	@echo "	[name].sim    (execute in SPiCboard simulator)"
	@echo "	[name].dis    (dissassemble the binary)"
	@echo "Compilation, linkage and translation of the object file is done automatically for the options above, but can also called explicitly:"
	@echo "	[name].elf    (compile and link to a GNU executable)"
	@echo "	[name].hex    (translate the GNU executable to intel hex)"
	@echo
	@/bin/echo -e "Example: In case you want to compile and flash \e[3maufgabe1.c\e[0m, connect your SPiCboard and run"
	@echo "	make aufgabe1.flash"
	@echo
	@echo "This generic version is for single source files only. If you have a more advanced project (e.g. utilizing several source files or modules) you have to create your own Makefile."
	@/bin/echo -e "Copy the contents of $(realpath $(lastword $(MAKEFILE_LIST))) into a \e[3mMakefile\e[0m located in your project directory and add a custom target."
	@/bin/echo -e "In case you are not familar with GNU Make, have a look at \e[4mhttps://www.gnu.org/software/make/manual/make.html\e[0m for further information."
	@echo

clean::
	rm -f *.hex *.elf *.bin *.dis *.o

%.flash: %.hex
	@if lsusb -d 03eb:2145 >/dev/null 2>&1; then \
		$(AVRDUDE) $(AVRDUDE_FLAGS) -U flash:w:$< ; \
	else \
		/bin/echo -e "\t\e[41;37;1m SPiCboard not found! \e[0m"; \
		echo 'Please connect your SPiCboard v3 via USB and retry.' ; \
		exit 1 ; \
	fi

%.sim: %.elf
ifeq ("$(wildcard $(SPICSIMJAR))","")
	@/bin/echo -e "\t\e[41;37;1m SPiCsim.jar not found! \e[0m"
	@echo 'But it is available on the (G)SPiC lecture web page.'
	@/bin/bash -c "select q in \"Let the Makefile download it now\" \"Abort\"; do case \$$q in Abort ) exit 1 ;; *) wget \"https://www4.cs.fau.de/Lehre/SS17/V_SPIC/Uebungen/doc/SPiCsim.jar\" -O \"$(SPICSIMJAR)\" ; break; ;; esac ; done"
endif
	java -jar $(SPICSIMJAR) $<

%.hex: %.elf
	avr-objcopy -O ihex -R .eeprom $< $@

%.bin: %.elf
	avr-objcopy -O binary -R .eeprom $< $@

%.dis: %.elf
	avr-objdump -d $< > $@

%.elf: %.c
	$(CC) $(CFLAGS) -o $@ $^ $(LDFLAGS)
	$(SIZE) $@

