/*
 * IDLflex -- A flexible, generic code generator for CORBA IDL
 *
 * Copyright (C) 2000  Hans Reiser, AspectIX Research Team
 *                     University of Erlangen-Nuernberg, Germany
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License.
 * The full text of the license can be found in the LICENSE.txt file.
 *
 * $Id: IDLDefCompiler.java,v 1.21 2002/02/28 13:57:53 reiser Exp $
 */

package org.aspectix.IDLflex;

import java.io.*;
import java.lang.*;
import java.util.Vector;
import java.util.StringTokenizer;
import java.util.Hashtable;
import java.util.Enumeration;

import org.omg.CORBA.*;

import org.xml.sax.EntityResolver;
import org.xml.sax.InputSource;
import org.xml.sax.SAXException;
import org.xml.sax.SAXParseException;        
import org.w3c.dom.Node;
import org.w3c.dom.Element;
import org.w3c.dom.Document;
import org.w3c.dom.NodeList;

import org.aspectix.IDLflex.IRObj.*;
import org.aspectix.xmlparser.*;

/** This class contains the main compiler routine generateCode and several
 * auxiliar functions used by the compiler.
 */
public class IDLDefCompiler {
    // Maps plugin type names to plugin implementation classes 
    private Hashtable pluginClasses;

    // List of active plugin objects 
    private Vector pluginVector;

    // List of used IDLWriter
    private IDLOutputList writerlist;

    // write path
    String writepath;

    Document        mainDoc;
    Element	    docElement;         // main Element of parsed xml file
    Utility         util;		// Utility Object (Spec from config file)
    Class           writerclass;	// IDLWriter output class (from config file)

    private IDLObjectVector generatePath;
    private IDLObject currentIDL;

    public IDLIterationPseudo getIterationPseudo() {
	int n = generatePath.count_objects();
	for(int i=n-1; i>=0; i--) {
	    IDLObject idlo = generatePath.objectAt(i);
	    if(idlo.is_a("IDLIterationPseudo")) 
		return (IDLIterationPseudo)idlo;
	}
	return null;
    }

    /** This is the constructor of the generic IDL compiler. It has
     *  one parameter which specifies the XML file name. The
     *  constructur runs the XML parser to get the parser tree
     *  reference and then initializes the specified Utility- und
     *  FileWriter-class.
     *
     * @exception IDLflexException thrown on all errors which cause an
     *     aborting of the compilation process.
     */
    

    /** This is the public generateCode method. 
     * It initializes the currentIDL reference;
     *
     * @exception IDLflexException on all errors with abort the compilation
     *     process
     */
    public IDLOutputList generateCode(String compName, IDLObject obj, 
				      int destmode)
	throws IDLflexException 
    {
	writerlist = new IDLOutputList(writerclass, writepath, destmode);
	if(compName==null) compName=getRootCompiler();
	push_currentIDL(obj);
	generateCode(compName, (IDLWriter)null);
	return writerlist;
    }

    /** This is another public generateCode method.
     *  It allows for passing a custom IDLWriter object for code output.
     *
     * @exception IDLflexException on all errors with abort the compilation
     *     process
     */
    public void generateCode(String compName, IDLObject obj, IDLWriter out)
	throws IDLflexException
    {
	writerlist = null;
	push_currentIDL(obj);
	generateCode(compName, out);
    }


    Vector x = new Vector();


    private Document parseFile(String filename) 
	throws org.aspectix.AspectixException
    {
	// Read and parse the XML mapping description into a memory object
        XMLparser p = new IBMXMLparser(new IDLflexEntityResolver());
	x.addElement(p);//test.
        String xmlpath = LocateFile.getXMLSearchPath();
        String fname = LocateFile.findFile(filename, xmlpath);
        if(fname==null) {
            System.err.println("file not found: "+filename);
	    System.exit(1); //TODO
        }
        //System.err.println("parseFile: reading "+fname);
        return p.parse(fname);
    }

    private void insertIncludedFiles(Document doc) throws org.aspectix.AspectixException
    {
	Vector includeFiles = new Vector();
	Vector removeNodes = new Vector();

	Element root = doc.getDocumentElement();
	NodeList chld = root.getChildNodes();
	int n = chld.getLength();
	for(int i=0; i<n; i++) {
	    Node node = chld.item(i);
	    if(node.getNodeName().equals("IMPORT")) {
		Element el = (Element)node;
		String file = el.getAttribute("NAME");
		includeFiles.add(file);
		removeNodes.add(node);
	    }
	}
	
	/* Remove all IMPORT nodes... */
	Enumeration en = removeNodes.elements();
	while(en.hasMoreElements()) {
	    Node node = (Node)en.nextElement();
	    root.removeChild(node);
	}

	/* Parse and insert IMPORT nodes */
	en = includeFiles.elements();
	while(en.hasMoreElements()) {
	    String file = (String)en.nextElement();
	    //System.err.println("...importing "+file);
	    Document impdoc = parseFile(file);
	    //System.err.println("...document is "+impdoc);
	    insertIncludedFiles(impdoc);
	    Element importroot = impdoc.getDocumentElement();
	    NodeList importchld = importroot.getChildNodes();
	    int impn = importchld.getLength();
	    for(int i=0; i<impn; i++) {
		Node addnode = doc.importNode( importchld.item(i), true );
		//System.err.println("     > adding node "+i+"/"+impn+": "+
		//addnode.getNodeName());
		root.appendChild( addnode );
	    }
	}
    }

    /* IDLDefCompiler:
     * Parameter ist eine Hashtable mit folgendem Inhalt:
     * IDLflexXMLPATH -> String
     * IDLflexDTDPATH -> String
     * IDLflexMapping -> String
     * IDLflexIncludePath -> String
     * IDLflexIncludeID -> Vector
     * IDLflexExcludeID -> Vector
     * DEF:xxx -> String     aus -D Kommando-Zeile
     */
    Hashtable config;

    public IDLDefCompiler(Hashtable cfg) throws IDLflexException 
    { 
	// initialize some internal variables
	generatePath = new IDLObjectVector();
	pluginVector = new Vector();
	pluginClasses = new Hashtable();
	config = cfg;

	writepath = (String)config.get("DEF:WRITEPATH");
	if(writepath == null) writepath = ".";
        try {
            mainDoc = parseFile( (String)config.get("IDLflexMapping") );
            insertIncludedFiles(mainDoc);
        } catch (org.aspectix.AspectixException a) {
            throw new IDLflexException("Error while parsing mapping: "+
				       a.getMessage());
        }
	docElement = mainDoc.getDocumentElement();

	// initialize class util (org.aspecitx.IDLflex.Utility) 
	String utilName="";
	try {
	    Element el = docElement;
	    if(el==null) {
		throw new IDLflexException("*** empty XML document\n");
	    }
	    else {
		utilName = el.getAttribute("UTILITY");
		util = (Utility)(Class.forName(utilName).newInstance());
		util.init(this);
	    }
	} catch(Exception e) {
	    e.printStackTrace();
	    throw new IDLflexException("*** Cannot create instance of"+
	       "specified UTILITY class '"+utilName+"'\n"+
	       e.toString());
	}
	
	// initialize class writerclass (org.aspecitx.IDLflex.IDLFileWriter)
	try {
	    Element el = docElement;
	    String wName = el.getAttribute("WRITER");
	    writerclass = Class.forName(wName);
	} catch(Exception e) {
	    throw new IDLflexException("*** Cannot load WRITER class:\n"+
				       e.toString());
	}
    }

    /** debugging function to print out the XML parser tree.
     */
    /*
    public void dump() {
	com.sun.xml.tree.TreeWalker tw = new com.sun.xml.tree.TreeWalker(docElement);
	Node n = tw.getCurrent();
	while(n != null) {
	    System.out.println("*** Node: "+n.getNodeName());
	    System.out.println("--- Data: "+n.getNodeValue());
	    n = tw.getNext();
	}
    }
    */

    /** adds a mapping from plugin name to implemenation class
     */
    public void registerPlugin(String name, String cls)
    {
	try {
	    Class c = Class.forName(cls);
	    pluginClasses.put(name.toLowerCase(), c);
	} catch (Exception e) {
	    System.err.println("registerPlugin: class "+cls+" not found.");
	}
    }

    /** auxiliar function to get the name of the root XML component
     *  (specified in the ROOT-attribute of the IDLflex-tag).
     */
    public String getRootCompiler() {
	Element el = docElement;
	if(el==null) return null;
	return el.getAttribute("ROOT");
    }

    private IDLObject remove_alias(IDLObject obj) {
	while( obj.is_a("AliasObj") ) {
	    try { obj= obj.getContent("RESOLVE"); }
	    catch(IDLflexException e) {}
	}
	return obj;
    }

    private Vector splitIfCondition(String cond) {
	int pos;
	Vector retval = new Vector();
	while( (pos=cond.indexOf("|")) >= 0 ) {
	    retval.addElement( cond.substring(0,pos) );
	    cond = cond.substring(pos+1);
	}
	retval.addElement( cond );
	return retval;
    }

    /** private function to check the type of an IDLObject. The
     * special symbols | and ! are used to check alternatives and
     * negate comparison.  Putting "RESOLVE." in front of an IDL
     * object name makes the comparison after resolving aliases,
     * i.e. the type of the aliased object is check, not of the alias
     * itself (which is of type "AliasObj").  
     */
    private boolean checkType(IDLObject obj, String type) {
	int pos;
	boolean negate, result;
	Vector cond = splitIfCondition(type);
	String what;

	for(int i=0; i<cond.size(); i++) {
	    what = (String)(cond.elementAt(i));
	    if( what.startsWith("!") ) {
		negate = true;
		what = what.substring(1);
	    } else {
		negate = false;
	    }
	    if( what.toLowerCase().startsWith("resolve.") ) {
		obj = remove_alias(obj);
		what = what.substring(8);
	    }
	    //System.err.println("checkType: what="+what+", obj="+obj);
	    if(what.equalsIgnoreCase("TypedefObj")) {
		result = obj.is_a("UnionObj") || obj.is_a("StructObj")||
		    obj.is_a("AliasObj") || obj.is_a("EnumObj");
	    }
	    else result = obj.is_a(what);

	    if(negate) result = !result;
	    if(result) return true;
	}
	return false;
    }

    /** This private function checks Attributs by calling the Utility
     *  class. OR and NOT expressions are handled the same way as in
     *  checkType.
     */
    private boolean checkAttr(IDLObject obj, String type) 
    throws IDLflexException 
    {
	int pos;
	boolean negate, result;
	Vector cond = splitIfCondition(type);
	String what;

	for(int i=0; i<cond.size(); i++) {
	    what = (String)(cond.elementAt(i));
	    if( what.startsWith("!") ) {
		negate = true;
		what = what.substring(1);
	    } else {
		negate = false;
	    }
	    if( what.toLowerCase().startsWith("resolve.") ) {
		obj = remove_alias(obj);
		what = what.substring(8);
	    }
	    result = util.getAttribute(what, obj);
	    if(negate) result = !result;
	    if(result) return true;
	}
	return false;
    }

    /** private function to check an &lt;IF&gt; expression. Depending
     * of whether TYPE or COND was specified, the verification of the
     * condition is delegated to checkType or checkAttr.
     *
     * The return value is 0 or 1 for false or true, or -1 if neither
     * TYPE nor COND was specified.  */
    int checkIf(Element current) throws IDLflexException 
    {
	String type=current.getAttribute("TYPE");
	if(!type.equals("")) {
	    if(checkType(currentIDL, type)) {
		return 1; 
	    }
	    return 0;
	}
	String attr=current.getAttribute("COND");
	if(!attr.equals("")) {
	    if(checkAttr(currentIDL, attr)) return 1;
	    else return 0;
	}
	throw new IDLflexException("condition with neither TYPE nor COND");
    }

    /** returns a reference to an XML element corresponding to the
     *  COMPONENT with the specified name.
     */
    Element getComponentElement(String compName) 
	throws IDLflexException
    {
	Element el = docElement;
	if(el!=null) {
	    Node n=el.getFirstChild();
	    while(n!=null) {
		if (n.getNodeName().equals("COMPONENT")) {
		    el = (Element)n;
		    if (el.getAttribute("NAME").equals(compName))
			return el;
		}
		n = n.getNextSibling();
	    }
	}
	throw new IDLflexException("Component "+compName+" does not exist!");
    }

    /** returns a reference to an XML element in the outmost level
     *  which is of the specified type (which may be TABLE, MAPTABLE
     *  or COMPONENT) and has the specified name.  
     */
    Element getBasicElement(String type, String name) {
	Element el = docElement;
	if(el==null) return null;
	Node n=el.getFirstChild();
	while(n!=null) {
	    if(n.getNodeName().equals(type)) {
		el = (Element)n;
		if(name==null || name.equals("")) return el;
		if( el.getAttribute("NAME").equals(name) ) return el;
	    }
	    n = n.getNextSibling();
	}
	return null;
    }

    /** Obtains a reference to an XML TABLE with the specified name,
     *  and returns its content as a list of names.
     */
    public final String[] getTable(String name) {
	Element el = getBasicElement("TABLE", name);
	if(el==null) return null;
	Node n=el.getFirstChild();
	String sep = el.getAttribute("SEPARATOR");
	String dt=n.getNodeValue();
	StringTokenizer st = new StringTokenizer(dt,sep);
	String[] retval = new String[st.countTokens()];
	int i=0;
	while(st.hasMoreTokens()) {
	    retval[i] = st.nextToken().trim();
	    i++;
	}
	return retval;
    }

    /** Obtains the content of an XML MAPTABLE entry represented by
     *  a Java Hastable.
     */
    public final Hashtable getMapTable(String name) {
	Element el = getBasicElement("MAPTABLE", name);
	if(el==null) return null;
	Node n = el.getFirstChild();
	Hashtable ht = new Hashtable();
	while(n!=null) {
	    if(n.getNodeName().equals("MAP")) {
		Element mapel = (Element)n;
		String idl_name = mapel.getAttribute("TAG");
		String map_name = mapel.getAttribute("MAP");
		ht.put(idl_name, map_name);
	    }
	    n = n.getNextSibling();
	}
	return ht;
    }


    /** All actions caused by a &lt;FILE&gt; entity.
     *<ul>
     *  <li> get filename using the Utility class
     *  <li> create fill filename defined by 'WRITEPATH'
     *  <li> open the file -> output stream (IDLFileWriter)
     *  <li> execute the content of &lt;FILE&gt;
     *  <li> close the file.
     *</ul>
     */
    private void fileTagHandler( Node current, String fileSpec ) 
	throws IDLflexException
    {
	IDLWriter newwriter;

	String fileName = util.getName(fileSpec, currentIDL);
	Debug.print(Debug.TRACE, Debug.IDLflex, "Creating "+fileName);
	try {
	    newwriter=writerlist.open(fileName, currentIDL.getName("id"), "");
	} catch(IDLflexException e) {
	    throw new IDLflexException("*** Cannot create file writer"+
				       " for file '"+fileName+"'\n"+
				       e.toString());
	}

	IDLObject oldref = util.getReferenceObject();
	util.setReferenceObject(currentIDL);
	try {
	    generateCode(current, newwriter);
	    try {
		newwriter.close();
	    } catch(Exception e) {
		throw new IDLflexException("*** Cannot close file writer"+
					   "for file '"+fileName+"'\n"+
					   e.toString());
	    }
	} finally {
	    util.setReferenceObject(oldref);
	}
    }

    /** search a plugin object of the specified name. 
     * 
     * @exception IDLflexException if no plugin object with the
     *     specified name exists.
     */
    public Plugin getPlugin(String name) throws IDLflexException
    {
	Plugin x;
	for(int i=pluginVector.size()-1; i>=0; i--) {
	    x = (Plugin)(pluginVector.elementAt(i));
	    if(x.name.equalsIgnoreCase(name))
		return x;
	}
	throw new IDLflexException("plugin object with name "+
				   name+" does not exist!");
    }

    /** All actions caused by &lt;PLUGIN&gt;.
     *  NEW:xxx creates a new instance of xxx;
     *  DEL removes plugin;
     *  all other commands are passed to the plugin object implementation.
     */
    private void pluginTagHandler(Node n, IDLObject obj)
	throws IDLflexException
    {
	String name = ((Element)n).getAttribute("ID").toLowerCase();
	String cmd = ((Element)n).getAttribute("OP").toLowerCase();
	Plugin x;

	if(cmd.equalsIgnoreCase("NEW"))
	    cmd="NEW:Counter";  // to be removed soon

	if(cmd.toUpperCase().startsWith("NEW:")) {
	    String spectype = cmd.substring(4).toLowerCase();
	    Class c = (Class)pluginClasses.get(spectype);
	    try {
		x = (Plugin)(c.newInstance());
		x.name = name;
	    } catch(Exception e) {
		throw new IDLflexException("creating plugin object "+
					   name+" ("+cmd+") failed.");
	    }
	    pluginVector.addElement(x);
	} else {
	    x = getPlugin(name);
	    if(cmd!=null) {
		if(cmd.equalsIgnoreCase("DEL"))
		    pluginVector.removeElement(x);
		else
		    x.execCmd(cmd);
	    }
	}
    }


    /** is equivalent to generateCode(node, writer), where
     *  node is a reference to the XML component with the name
     *  compName.
     *
     * @exception IDLflexException bei allen Fehlern, die zum Abbruch
     *     des Kompilier-Vorgangs fhren
     */
    private void generateCode(String compName, IDLWriter writer)
	throws IDLflexException
    {
	Debug.print(Debug.TRACE, Debug.IDLflex, 
		    "generateCode called for "+compName);
	Element el = getComponentElement(compName);
	Node n = el.getFirstChild();
	generateCode(n, writer);
    }


    /** resolves an IDL-object into a new object by iterately calling
     *  its getContent method.
     *
     *  prefix specifies a hierarchical name, with the components
     *  separated by semicolon. getContent is called iterately for all
     *  components starting on the left.
     *
     *  @exception IDLflexException
     *    is thrown, if calling getContent for a component of prefix
     *    returns no new IDLObject.
     */
    private static IDLObject resolveObject(IDLObject obj, String prefix) 
	throws IDLflexException 
    {
	int pos;
	IDLObject newobj;
	while( true ) {
	    pos=prefix.indexOf(".");
	    if(pos>=0) newobj=obj.getContent( prefix.substring(0,pos) );
	    else newobj=obj.getContent( prefix );
	    if(newobj==null) {
		throw new IDLflexException("resolveObject: reached null Object while resolving prefix "+prefix+" on "+obj.getName("id")+"/"+obj);
	    }
	    obj = newobj;
	    if( pos<0 ) break;
	    prefix = prefix.substring(pos+1);
	}
	return obj;
    }

    /** handles the "OBJ" attributes used in &lt;CALL&gt; and &lt;GET&gt;. 
     *  Just delegates all work to resolveObject(). 
     */
    private IDLObject resolveOBJattr(Element current, IDLObject obj) 
    throws IDLflexException
    {
	String use = current.getAttribute("OBJ");
	if(use==null || use.equals("")) return obj;
	else if(use.equals("UPPER")) 
	    return generatePath.getUpperObject(1);
	else return resolveObject(obj, use);
    }


    /** Test if an object is to be included or excluded according to
     *  -I switches.
     *  If the object is a module, it is sufficient to have a matching
     *     prefix of a -I entry. Otherwise, the whole object ID has to
     *     match.
     *  Rejectt rules always require a full match!
     *  returns true, if Object should be included!
     */
    private boolean checkIncludeExclude( IDLObject obj ) 
	throws IDLflexException
    {
	String scopedname = obj.getName("fullname");

	if(scopedname!=null && !scopedname.equals("")) {

	    if(!scopedname.startsWith("::"))
		scopedname = "::"+scopedname;  // workaround for JavaORB bug
	
	    // Verify accept-rules (if there are any...)
	    int i;
	    Vector include = (Vector)config.get("IDLflexIncludeID");
	    if(include==null) include = new Vector();
	    int len = include.size();
	    for(i=0; i<len; i++) {
		if( scopedname.startsWith((String)include.elementAt(i)) )
		    break;
		if( ((String)include.elementAt(i)).startsWith(scopedname) &&
		    obj.is_a("ModuleObj") ) break;
	    }
	    if(len>0 && i>=len) {
		Debug.print(Debug.TRACE, Debug.IDLflex, 
			    "not including "+scopedname);
		return false;   // no accept rule found...
	    }

	    // Verify reject-rules...
	    Vector exclude = (Vector)config.get("IDLflexExcludeID");
	    len=exclude.size();
	    for(i=0; i<len; i++) {
		if( scopedname.startsWith((String)exclude.elementAt(i)) ){
		    Debug.print(Debug.TRACE, Debug.IDLflex, 
				"rejecting "+scopedname);
		    return false;
		}
	    }
	    return true;
	}
	else return true;
	// should never be reached:
	// return true;
    }

    private IDLObject[] reverselist(IDLObject[] list) {
	int len = list.length;
	IDLObject retlist[] = new IDLObject[len];
	for(int i=0; i<len; i++) {
	    retlist[i] = list[len-i-1];
        }
	return retlist;
    }

    /** private helper routine to get a list of objects to iterate on,
     *  taking -I and -X (include and exclude) switches into account.
     *  -I and -X will only be respected if iterating on the MEMBER of
     *  a module or of the root idl object.
     */
    private IDLObject[] getIterateContentList( IDLObject obj, String spec )
	throws IDLflexException
    {
	int ind = spec.indexOf(":");
        String modif = "";
	if(ind>0) {
	    modif = spec.substring(ind+1);
	    spec = spec.substring(0,ind);
	}
	if( !( obj.is_a("ContainerObj",false)||obj.is_a("ModuleObj") ) ||
	    !spec.equals("MEMBER") )
	{
	    IDLObject[] list = obj.getContentList(spec);
	    if(modif.equalsIgnoreCase("reverse")) {
		list = reverselist(list);
	    }
	    return list;
	}

	IDLObject[] list = obj.getContentList(spec);
	Vector v = new Vector(list.length);
	for(int i=0; i<list.length; i++) {
	    if( checkIncludeExclude( list[i] ) )
		v.addElement(list[i]);
	}
	list = new IDLObject[v.size()];
	for(int i=0; i<v.size(); i++) {
	    list[i] = (IDLObject)v.elementAt(i);
	}
	if(modif.equalsIgnoreCase("reverse")) {
	    list = reverselist(list);
	}
	return list;
    }


    /* * * * tag specific generate functions * * * */

    private IDLObject resolveOBJ(Element current) 
	throws IDLflexException
    {
	return resolveOBJattr(current, currentIDL);
    };
    private void push_currentIDL(IDLObject obj) {
	generatePath.addObject(obj);
	currentIDL = obj;
	//System.out.println("pushing (len="+generatePath.count_objects()+
	//		   ") -> currentIDL="+currentIDL);
    };
    private void pull_currentIDL() {
	generatePath.removeLastObject();
	currentIDL = generatePath.getLastObject();
	//System.out.println("pulling (len="+generatePath.count_objects()+
	//		   ") -> currentIDL="+currentIDL);
    }

    
    private void generateFILE(Element current, IDLWriter writer) 
	throws IDLflexException
    {
	push_currentIDL(resolveOBJ(current));
	try {
	    String fileSpec = current.getAttribute("SPEC");
	    fileTagHandler(current.getFirstChild(), fileSpec);
	} finally {
	    pull_currentIDL();
	}
    }


    private void generateIF(Element current, IDLWriter writer) 
	throws IDLflexException
    {
	push_currentIDL(resolveOBJ(current));
	try {
	    if( checkIf(current) == 1 ) {
		generateCode(current.getFirstChild(), writer);
	    }
	} finally {
	    pull_currentIDL();
	}
    }


    private void generateSWITCH(Element current, IDLWriter writer) 
	throws IDLflexException
    {
	push_currentIDL(resolveOBJ(current));
	try {
	    Node child = current.getFirstChild();
	    String childt;
	    while(child!=null) {
		childt = child.getNodeName();
		if(childt.equals("CASE")) {
		    generatePath.addNode(child);
		    push_currentIDL(resolveOBJ((Element)child));
		    try {
			if(checkIf((Element)child)==1 ) {
			    Node exec;
			    try {
				exec=((Element)child).getFirstChild();
			    } catch(Exception ex) {
				System.err.println(child);
				ex.printStackTrace();
				throw new IDLflexException("getFirstChild");
			    }
			    generateCode(exec, writer);
			    break;
			}
		    } finally { 
			pull_currentIDL(); 
			generatePath.removeLastNode();
		    }
		} else if (childt.equals("DEFAULT")) {
		    generatePath.addNode(child);
		    push_currentIDL(resolveOBJ((Element)child));
		    try {
			Node exec=child.getFirstChild();
			generateCode(exec, writer);
			break;
		    } finally {
			pull_currentIDL();
			generatePath.removeLastNode();
		    }
		}
		child=child.getNextSibling();
	    }
	} finally {
	    pull_currentIDL();
	}
    }

    private void generateCALL(Element current, IDLWriter writer) 
	throws IDLflexException
    {
	push_currentIDL(resolveOBJ(current));
	try {
	    String comp = current.getAttribute("NAME");
	    generateCode(comp, writer);
	} finally {
	    pull_currentIDL();
	}
    }


    private void generateITERATE(Element current, IDLWriter writer)
	throws IDLflexException
    {
	push_currentIDL(resolveOBJ(current));
	try {
	    String spec=((Element)current).getAttribute("NAME");
	    if(spec.equals("")) 
		throw new IDLflexException("ITERATE: missing NAME");

	    Node chld = current.getFirstChild();
	    IDLObject[] objlist = getIterateContentList(currentIDL, spec);

	    if(objlist==null) {
		throw new IDLflexException("ITERATE: no such spec "+spec+
					   " in Object "+currentIDL);
	    } else {
		IDLIterationPseudo iip = new IDLIterationPseudo();
		iip.count = objlist.length;
		push_currentIDL( iip );
		//System.err.println("ITERATE: spec = "+spec+"  # = "+iip.count);
		try {
		    for(int i=0; i<objlist.length; i++) {
			iip.index = i;
			push_currentIDL( objlist[i] );
			try {
			    generateCode( chld, writer);
			} finally {
			    pull_currentIDL();
			}
		    }
		} finally {
		    pull_currentIDL();
		}
	    }
	} finally {
	    pull_currentIDL();
	}
    }

    private void generateERROR(Element current, IDLWriter writer)
	throws IDLflexException
    {
	push_currentIDL(resolveOBJ(current));
	try {
	    String err="";
	    Node text = current.getFirstChild();
	    while(text!=null) {
		if (text.getNodeName().equals("#text"))
		    err+=text.getNodeValue();
		else 
		    ; // sollte ned sein....(?)
		text=text.getNextSibling();
	    }
	    if (err.equals("")) 
		err="ERROR-Tag in xml-File without Text";
	    System.err.println(currentIDL);
	    System.err.println(currentIDL.getClass());
	    throw new IDLflexException(err);
	} finally {
	    pull_currentIDL();
	}
    }


    private void generateGET(Element current, IDLWriter writer)
	throws IDLflexException
    {
	push_currentIDL(resolveOBJ(current));
	try {
	    String param = current.getAttribute("T");
	    String dt=util.getName(param, currentIDL);
	    try {
		if(writer!=null) writer.write(dt);
	    } catch(Exception e) { 
		System.err.println("GET("+param+"): " + e);
		e.printStackTrace(); 
	    }
	} finally {
	    pull_currentIDL();
	}
    }
    
    private void generateSBOX(Element current, IDLWriter writer)
	throws IDLflexException
    {
	push_currentIDL(resolveOBJ(current));
	try {
	    String name = current.getAttribute("NAME");
	    IDLStringWriter strw = new IDLStringWriter();
	    generateCode(current.getFirstChild(), strw);
	    try { strw.close(); } catch(Exception e) {/*FIXME*/}
	    util.savebox.put( name, strw.toString() );
	} finally {
	    pull_currentIDL();
	}
    }

    /** General code generation methode; is called with the references
     *  to an XML node and an IDL object.
     *
     *  reads the content of the XML node and executes it step by step.
     */
    public void generateCode(Node current, IDLWriter writer) 
	throws IDLflexException
    {
	Debug.print(Debug.TRACE, Debug.IDLflex, 
		    "generateCode called:");
	try {
	    while(current!=null) {
		Debug.print(Debug.TRACE, Debug.IDLflex, 
			    "looping: \n"+
			    "current Node name is "+current.getNodeName()+
			    "\nnode value is "+current.getNodeValue()+
			    "\nIDLObject is "+currentIDL+"\n");

		generatePath.addNode( current );
		String t;
		t = current.getNodeName();
		if(t.equals("FILE")) {
		    generateFILE( (Element)current, writer);
		} else if(t.equals("IF")) {
		    generateIF( (Element)current, writer);
		} else if(t.equals("SWITCH")) {
		    generateSWITCH( (Element)current, writer);
		} else if(t.equals("CALL")) {
		    generateCALL( (Element)current, writer);
		} else if(t.equals("ITERATE")) {
		    generateITERATE( (Element)current, writer);
		} else if(t.equals("ERROR")) {
		    generateERROR( (Element)current, writer);
		} else if(t.equals("#text")) {
		    String dt=current.getNodeValue();
		    if(!dt.equals("")) {
			try {
			    if(writer!=null) writer.write(dt);
			} catch(Exception e) { 
			    System.err.println("#text: " + e.toString()); 
			}
		    }
		} else if(t.equals("GET")) {
		    generateGET( (Element)current, writer );
		} else if(t.equals("P")) {
			try {
			    if(writer!=null) writer.write("\r");
			} catch(Exception e) { 
			    System.err.println("P: " + e.toString()); 
			}
		}
		// Save-Boxes
		else if(t.equals("SBOX")) {
		    generateSBOX( (Element)current, writer );
		} else if(t.equals("SBOXDEL")) {
		    String name = ((Element)current).getAttribute("NAME");
		    util.savebox.del(name);
		}
		// Plugin-Tag
		else if(t.equals("PLUGIN")) {
		    pluginTagHandler( current, currentIDL );
		}
		generatePath.removeLastNode();

		current=current.getNextSibling();
	    }
	} 
	catch(IDLflexException e) {
	    if(e.hasContext()) throw(e);
	    throw new IDLflexException(generatePath, e.getMessage());
	}
	finally {
	}
    }
}
