/*
 * IDLflex -- A flexible, generic code generator for CORBA IDL
 *
 * Copyright (C) 2000  Hans Reiser, AspectIX Research Team
 *                     University of Erlangen-Nuernberg, Germany
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 * 
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You find a copy of the license in the LICENSE.txt file in the
 *  root of the software distribution; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *  Please read the README.txt file first.
 *
 *  $Id: IDLflexMain.java,v 1.20 2002/07/29 12:22:25 reiser Exp $
 */

package org.aspectix.IDLflex;

import org.aspectix.IDLparser.pub.*;
import org.aspectix.IDLflex.*;
import org.aspectix.IDLflex.IRObj.*;
import java.util.*;
import java.io.*;


/** IDLflexMain implements the main method, which is responsible for the
 *  initialization of the CORBA environment and the Interface Repository.
 *  External modules are responsible for vendor-specific stuff 
 *  (IR_JavaORB, IR_ORBcus).
 *
 *  Most of the code is used for parsing command line arguments and
 *  setting up the IR connection.
 */
final public class IDLflexMain {
    private static int verbose=0;

    //static org.omg.CORBA.Object obj = null;
    //public static org.omg.CORBA.Repository ir = null;

    static Vector ORBArguments;
    static {
	ORBArguments = new Vector();
    }

    private static void printVersionAndExit()
    {
	System.out.println("IDLflex version "+Version.VERSIONSTRING);
	System.exit(0);
    }

    private static String argsFileName = null;
    private static String argsMapping = null;

    public static class _Attribute {
	String[] names;
	String[] values;
	_Attribute(Vector attr) {
	    names = new String[attr.size()];
	    values = new String[attr.size()];
	    for(int i=0; i<attr.size(); i++) {
		String data = ((String)attr.elementAt(i));
		int pos = data.indexOf("=");
		if(pos>0) {
		    names[i] = data.substring(0,pos);
		    values[i] = data.substring(pos+1);
		} else {
		    names[i] = data;
		    values[i] = "";
		}
		Debug.print(Debug.TRACE, Debug.IDLflex,
			    "defined: "+names[i]+" = "+values[i]);
	    }
	}
	boolean isDefined(String s) {
	    int len=names.length;
	    for(int i=0; i<len; i++) {
		if(names[i].equals(s)) return true;
	    }
	    return false;
	}
	String getDefined(String s) {
	    int len=names.length;
	    for(int i=0; i<len; i++) {
		if(names[i].equals(s)) return values[i];
	    }
	    return null;
	}
    }
    public static _Attribute argsAttributes;
    public static Vector argsInclude;
    public static Vector argsExclude;
    public static Vector argsIDLInclude;
    public static String argsHandon = "";

    /**	Command line parsing.
     */
    public static int process_args( String args[] ) {
	int state=0;
	Vector Attributes = new Vector();
	Vector Include = new Vector();
	Vector Exclude = new Vector();
	Vector IDLInclude = new Vector();

	for(int i=0; i<args.length; i++) {
	    String data=args[i];
	    if(state==0) {
		if(args[i].length()>2) data=args[i].substring(2);
		else data="";
		if(args[i].startsWith("-D")) state=1;
		else if(args[i].startsWith("-I")) state=2;
		else if(args[i].startsWith("-X")) state=3;
		else if(args[i].startsWith("-m")) state=4;
		else if(args[i].startsWith("-i")) state=6;
		else if(args[i].startsWith("-P")) { 
		    argsHandon = args[i+1]; 
		    i++;
		}    
		else if(args[i].startsWith("-V")) {
		    printVersionAndExit();
		}
		else if(args[i].startsWith("-v")) verbose++; 
		else if(args[i].equals("--")) { state=5; }
		else if(args[i].startsWith("-")) {
		    System.err.println("Illegal argument: "+args[i]);
		    return -1;
		} else {
		    if(argsFileName==null) argsFileName=args[i];
		    else {
			System.err.println("Invalid argument");
		    }
		}
		if(data.equals("")) continue;
	    }
	    if(state==1) {
		Attributes.addElement(data); state=0;
	    } else if(state==2) { 
		Include.addElement(data); state=0;
	    } else if(state==3) { 
		Exclude.addElement(data); state=0;
	    } else if(state==6) { 
		IDLInclude.addElement(data); state=0;
	    } else if(state==4) {
		if(argsMapping==null) argsMapping=data;
		else {
		    System.err.println("Duplicate Option '-m'");
		    break; //return -1;
		}
		state=0;
	    } else if(state==5) {
		ORBArguments.addElement(data);
	    }
	}
	if(state!=0 && state!=5) {
	    System.err.println("Option missing argument:"+args[args.length-1]);
	    //return -1;
	}
	argsAttributes = new _Attribute(Attributes);

	argsInclude = Include;
	argsExclude = Exclude;
	if(argsMapping==null)
	    argsMapping="Java/JavaMapping.xml";
	argsIDLInclude = IDLInclude;

	Debug.setVerbose(verbose);
	if(argsFileName==null) return -1;
	return 0;
    }


    public static String getMapName(String fname) {
	String xmlpath = LocateFile.getXMLSearchPath();
	return LocateFile.findFile(fname, xmlpath);
    }

    private static String IDLflexHome;


    /** main method: 
     *<ul>
     *  <li>parse command line arguments
     *  <li>Instantiate IDLcompiler (parses the XML mapping program)
     *  <li>Parse the IDL object tree
     *  <li>run the compiler
     *</ul>
     */
    public static void main( String args[] )
    {
        // IR_ORBacus has to 'kill' external IR on exit...
	//	java.lang.Runtime.runFinalizersOnExit(true);

	// 1. Parse command line
	IDLflexHome = java.lang.System.getProperty("IDLflexHome",".");
	int res = process_args( args );
	if(res<0) {
	    System.err.println("Usage: ");
	    System.err.println("java org.aspectix.IDLflex.IDLflexMain [-D<defines>] [-I<includeIDs>] [-X<excludeIDs>] [-i<includepath>] [-m<Mapping.xml>] <filename>");
	    System.exit(1);
	}


	// 2. Initialize compiler (read XML mapping file)
	Debug.print(Debug.INFO, Debug.IDLflex,
		    ">> IDLflexMain: Initializing the compiler object...");
	IDLcompiler compiler = null;
	try {
	    compiler = new IDLcompiler(argsMapping);
	} catch(IDLflexException ex) {
	    Debug.print(Debug.IDLflex, ex);
	    System.exit(1);
	}

	// 3.
	Debug.print(Debug.INFO, Debug.IDLflex, 
		    ">> IDLflexMain: parsing IDL...");
	IDLObject iobj = null;
	try {
	    iobj = compiler.getIDLObjects( "FILE:"+argsFileName );
	    //iobj = compiler.getIDLObjects("JAVAORBIDL:"+argsFileName);
	    //iobj = compiler.getIDLObjects("JAVAORBIOR:"+argsFileName);
	} catch(IDLflexException e) {
	    Debug.print(Debug.IDLflex, e);
	    System.exit(1);
	}


	// 4. and run the compiler...
	Debug.print(Debug.INFO, Debug.IDLflex, 
		    ">> IDLflexMain: Running the compiler...");
	try {
	    if(true) {
		IDLOutputList out = compiler.generateCodeFiles( iobj );
	    } else {
		IDLOutputList out = compiler.generateCodeStreams( iobj );
		Enumeration en = out.getFiles().keys();
		while(en.hasMoreElements()) {
		    String file = (String)en.nextElement();
		    System.out.println("***** content of "+file);
		    java.io.Reader r = out.getFileReader(file);
		    java.io.BufferedReader br = new BufferedReader(r);
		    String s;
		    while( (s=br.readLine())!=null )
			System.out.println(s);
		}
	    }
	} catch( Exception e ) {
	    Debug.print(Debug.IDLflex, e);
	    System.exit(1);
	}

	//if(irbase!=null) { irbase.cleanup(); irbase=null; }
	Debug.print(Debug.INFO, Debug.IDLflex, "done.");
	System.exit(0);
    }

    protected void finalize() throws Throwable
    {
	//if(irbase!=null) irbase.cleanup();
    }
}
