/*
 * IDLflex -- A flexible, generic code generator for CORBA IDL
 *
 * Copyright (C) 2000  Hans Reiser, AspectIX Research Team
 *                     University of Erlangen-Nuernberg, Germany
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License.
 * The full text of the license can be found in the LICENSE.txt file.
 *
 * $Id: FimoObjectCreator.java,v 1.21 2003/11/10 21:44:10 reiser Exp $
 */

package org.aspectix.IDLflex.IRObj;

import org.omg.CORBA.IRObject;
import org.aspectix.IDLparser.pub.*;
import org.aspectix.IDLparser.tree.LookupUtil;
import java.util.Iterator;
import java.util.Hashtable;
import org.aspectix.IDLflex.IDLflexException;


/** This class is used to create IDL objects from Interface Repository
 *  objects. Depending on the type of the IR definition, the
 *  corresponding IDL object is created.  
 */
public class FimoObjectCreator {
    private static boolean verbose = false;

    /** Hashtable for created IDLObjects.
     */
    static Hashtable ht;
    static {
	ht = new Hashtable();
    }

    /** Translate IDL object ID into a java.lang.Object used in lookup
     *  hashtable.
     */
    private static java.lang.Object getID(IDLNode node) {
	try {
	    IDLNamed n = (IDLNamed)node;
	    return n.getScopedName();
	} catch(Exception e) {};
	try {
	    IDLBaseType b = (IDLBaseType)node;
	    return "__BASETYPE__"+b.getName();
	} catch(Exception e) {};
	return node;
    }

    /** Method get tries to return a reference to an already existing object.
     *  if no such object is found, the behavior depends on the
     *  paramter create: if it is set, a new object will be created,
     *  otherwise null is returned.
     *  mode: 1=create object if it does not exist
     *        2=initialize object with add node content even if it already
     *          exists (used for module reopening (=appending))
     */
    public static IDLObject get(IDLNode node, int mode) 
	throws IDLflexException 
    {
	if(node instanceof IDLScopedName) {
	    try {
		node = ((IDLScopedName)node).getReferencedType();
		if (node instanceof IDLInterf) {
		    boolean x = ((IDLInterf)node).forwardDecl();
		    if(x==true) {
			node = LookupUtil.instance().lookupInterfaceDefinition(((IDLInterf)node).getScopedName());
		    }
		    x = ((IDLInterf)node).forwardDecl();
		}
	    } catch(Exception ex) {
		System.err.println("FimoObjectCreater: get: failed to "+
				   "resolve scoped name!"+ex);
	    }
	}

        IDLObject obj=null;
	////Hack for VS
	////if( !(mode>0&&(node instanceof IDLOpDecl || node instanceof IDLParamDecl) )){
	   obj = (IDLObject)ht.get( getID(node) );
        ////}
	if(obj!=null) {
	    //System.err.println("Found: "+getID(node));
	    //System.err.println("obj is "+obj);
	    //System.err.println("obj id is "+obj.getName("id"));
	    if(mode==2) {
		init_object(obj, node);
		return null;
	    }
	    return obj;
	} else {
	    if(mode>0) {
		if(verbose) System.err.println("Creating: "+getID(node) );
		IDLObject o = create_obj(node);
		if(o!=null) {
		    // First we have to register the object. Some loop
		    // in the definitions might have to lookup the 
		    // current object in the hash table in its
		    // init_members function.
		    ht.put( getID(node), o);
		    init_object(o, node);
		    //o.init_members();
		} else {
                    if(verbose)
                        System.err.println("ObjectCreator: create failed or skipped for definition of "+node);
		}
		return o;
	    } else {
		return null;
	    }
	}
    }

    /**	Just an alias to
     *	{@link #get(IDLNode,int) get}(iro, 1); 
     */
    public static IDLObject get(IDLNode node) 
	throws IDLflexException
    {
	if(node==null) 
	    throw new IDLflexException("ObjectCreator: reference object is null");
	return get(node, 1);
    }



    
    /** Creates an IDLObject according to the type of the IRObject.
     */
    private static IDLObject create_obj(IDLNode node) {
	IDLObject idlobj;
	if(verbose) {
	    System.err.println("FimoObjectCreator::create_object: "+
			       "called with type "+node);
	    System.err.println("class is "+node.getClass().getName());
	}
	if (node instanceof IDLArray) {
	    idlobj = new ArrayObj();
	} else if(node instanceof IDLAttr) {
	    idlobj = new AttributeObj();
	} else if (node instanceof IDLConstDef) {
	    idlobj = new ConstantObj();
	} else if (node instanceof IDLExcept) {
	    idlobj = new ExceptionObj();
	} else if (node instanceof IDLInterf) {
/*
	this does not work:
        referencing the forward declaration requires that the Interface object was created.
        thus we create it here (at declaration time), and initialize it later (at definition)

	    if(((IDLInterf)node).forwardDecl()) {
		if(verbose) System.err.println(" INFO: create_obj: skipping forward decl");
		return null;
	    }
*/
	    idlobj = new InterfaceObj();
	} else if (node instanceof IDLModule) {
	    idlobj = new ModuleObj();
	} else if (node instanceof IDLOpDecl) {
	    idlobj = new OperationObj();
	} else if (node instanceof IDLTypedef) {
	    //idlobj = new TypedefObj();
	    idlobj = new AliasObj();
	}
	/* else if (node instanceof Alias) 
	   case DefinitionKind._dk_Alias:
	   idlobj = new AliasObj();
	   break;
	*/
	else if (node instanceof IDLStruct) {
	    idlobj = new StructObj();
	} else if (node instanceof IDLUnion) {
	    idlobj = new UnionObj();
	} else if (node instanceof IDLEnum) {
	    idlobj = new EnumObj();
	} 
	else if (node instanceof IDLBaseType) {
	    idlobj = new PrimitiveObj();
	}
	else if (node instanceof IDLString) {
	    idlobj = new StringObj();
	} else if (node instanceof IDLSequence) {
	    idlobj = new SequenceObj();
	} else if (node instanceof IDLSpec) {
	   
	   idlobj = new ContainerObj();
	}
	else if (node instanceof IDLFixed) {
	    idlobj = new FixedObj();
	} else if (node instanceof IDLTypeDecl) {
	    System.err.println("ERROR: IDLTypeDecl not yet handled for "+
			       ((IDLTypeDecl)node).getName()+" -- "+
			       ((IDLTypeDecl)node).getScopedName() );
	    try {
		throw new RuntimeException("und raus hier...");
	    } catch(Exception ex) {
		ex.printStackTrace();
	    }
	    return null;
	}
	else if (node instanceof IDLValueType) {
	    idlobj = new ValueTypeObj();
	}
	else if (node instanceof IDLNative) {
	    idlobj = new NativeObj();
	}
	else {
	    /* IDLCase IDLConst IDLContext (IDLException) IDLNamed 
	     * IDLNode IDLParamDecl IDLRaises IDLScope* (IDLStackIterator)
	     * (IDLTracer) (IDLType) IDLTypeDecl (IDLTypeUtil)
	     */
	    if(verbose) {
                System.err.println("FimoObjectCreator::create_object: "+
                                   "unknown object type "+node);
                System.err.println("class is "+node.getClass().getName());
            }
	    return  null;
	}
	return idlobj;
    }

    private static void init_interface_object(IDLObject obj, IDLNode node) 
	throws IDLflexException
    {
	IDLInterf id = (IDLInterf)node;
	if(id.forwardDecl()) {
	    //System.err.println("init_interface_object: no initialization for forward decl");
	    return;
	}

	try {
	    Iterator it = id.getDirectBaseInterfaces();
	    while(it.hasNext()) {
		IDLInterf base = (IDLInterf)it.next();
		InterfaceObj iobj = (InterfaceObj)get(base);
		obj.addToContentList("BASE", iobj);
	    }
	} catch(IDLException ex) {};
	
	Iterator it = id.getLocallyDefinedMembers();
	while(it.hasNext()) {
	    IDLNode member = (IDLNode)it.next();
	    IDLObject iobj = get( member );
	    if(iobj==null) {
		System.err.println("FIMO: iobj is null for member "+member);
	    }
	    obj.addToContentList("MEMBER", iobj);
	}

	it = id.getVisibleMembers();
	while(it.hasNext()) {
	    IDLNode member = (IDLNode)it.next();
	    IDLObject iobj = get( member );
	    obj.addToContentList("ALLMEMBER", iobj);
	}

	obj.setAttribute("abstract", id.abstractIf() );
	obj.setAttribute("local", id.localIf());
    }
    

    private static void init_valuetype_object(IDLObject obj, IDLNode node) 
	throws IDLflexException
    {
	IDLValueType vtd = (IDLValueType)node;
	if(vtd.forwardDecl()) {
	    //System.err.println("init_valuetype_object: no initialization for forward decl");
	    return;
	}

	try {
	    Iterator it = vtd.getDirectBaseValueTypes();
	    while(it.hasNext()) {
		IDLValueType base = (IDLValueType)it.next();
		ValueTypeObj iobj = (ValueTypeObj)get(base);
		obj.addToContentList("BASE", iobj);
	    }
	} catch(IDLException ex) {
	    ex.printStackTrace();
	}

	Iterator it = vtd.getSupportedInterfaces();
	while(it.hasNext()) {
	    IDLInterf base = (IDLInterf)it.next();
	    InterfaceObj iobj = (InterfaceObj)get(base);
	    obj.addToContentList("SUPPORTS", iobj);
	}
	
	it = vtd.getLocallyDefinedMembers();
	while(it.hasNext()) {
	    IDLNode member = (IDLNode)it.next();
	    IDLObject iobj = get( member );
	    obj.addToContentList("MEMBER", iobj);
	}

	it = vtd.getVisibleMembers();
	while(it.hasNext()) {
	    IDLNode member = (IDLNode)it.next();
	    IDLObject iobj = get( member );
	    obj.addToContentList("ALLMEMBER", iobj);
	}

	obj.setAttribute("abstract", vtd.abstractvalue());
	obj.setAttribute("custom", vtd.custom());
	obj.setAttribute("boxed", vtd.boxed());
	obj.setAttribute("truncatable", false); // TODO FIXME !!!
    }
    

    private static void init_string_obj(IDLObject obj, IDLNode node, 
				   boolean bound, boolean wide)
	throws IDLflexException
					       
    {
	if(bound) {
	    IDLString s = (IDLString)node;
	    try {
		obj.setValue("bound", new Integer(s.getLength()));
	    } catch(IDLException ex) {};
	}
	else
	    obj.setValue("bound", new Integer(-1));
	obj.setAttribute("bound", bound);
	obj.setAttribute("wide", wide);
    }


    private static void init_primitive(IDLObject obj, IDLNode node)
	throws IDLflexException
    {
	IDLBaseType base = (IDLBaseType)node;
	String s = base.getName();
	String name;
	if(s.equals("unsigned_long")) name="tk_ulong";
	else if(s.equals("unsigned_short")) name="tk_ushort";
	else if(s.equals("long_long")) name="tk_longlong";
	else if(s.equals("unsigned_long_long")) name="tk_ulonglong";
	else if(s.equals("long_double")) name="tk_longdouble";
        else if(s.equals("Object")) name="tk_objref";
	else if(s.equals("ValueBase")) name="tk_value";
	else name = "tk_"+base.getName();
	obj.setName("name", name);
    }

    private static void init_operation(IDLObject obj, IDLNode node)
	throws IDLflexException
    {
	IDLOpDecl opd = (IDLOpDecl)node;
	if(opd.isOneway()) {
	    obj.setAttribute("oneway", true);
	}
	
	Iterator it = opd.getParams();
	while(it.hasNext()) {
	    IDLParamDecl param = (IDLParamDecl)it.next();
	    ParameterObj paro = new ParameterObj();
	    paro.setName("name", param.getName());

	    String attr = param.getAttribute();
	    if(attr.equals("in")) {
		paro.setAttribute("inarg", true);
	    } else if(attr.equals("out")) {
		paro.setAttribute("outarg", true);
	    } else if(attr.equals("inout")) {
		paro.setAttribute("inoutarg", true);
	    } else {
		System.err.println("init_operation: unkown attr "+attr);
	    }
	    
	    IDLObject base = get( param.getIDLType() );
	    paro.setContent("BASE", base);
	    paro.setContent("CONTAINER", obj);
	    obj.addToContentList("PARAM", paro);
	}
	
	IDLRaises raises = opd.getRaises();
	it = null;
	try {
	    if(raises!=null) it = raises.getExceptions();
	} catch(IDLException ex) {};
	if(it!=null) {
	    while(it.hasNext()) {
		IDLExcept exc = (IDLExcept)it.next();
		IDLObject except = get( exc );
		obj.addToContentList("EXCEPT", except);
	    }
	}
	
	// set return
	IDLObject ret = get( opd.getIDLType() );
	obj.setContent("RETURN", ret);
    }

    private static void init_union(IDLObject obj, IDLNode node) 
	throws IDLflexException
    {
	IDLUnion ud = (IDLUnion)node;
	IDLObject discr = get( ud.getSwitchType() );
	obj.setContent("DISCR", discr);
	
	Iterator it = ud.getCases();
	while(it.hasNext()) {
	    IDLCase icase = (IDLCase)it.next();
	    Iterator labels=null;
	    try {
		labels = icase.getDiscriminators( ud.getSwitchType() );
	    } catch( IDLException ex ) {
		System.err.println("init_union: cannot get discriminator!");
		ex.printStackTrace();
	    }
	    if(labels==null) continue;
	    while(labels.hasNext()) {
		Object olabel = labels.next();
		UnionMemberObj uobj = new UnionMemberObj();
		boolean isdef = false;
		if(olabel.equals("default")) isdef = true;
		uobj.setAttribute("isdefault", isdef);
		uobj.setContent("CONTAINER", obj);
		uobj.setContent("SUPER", obj);
		uobj.setContent("BASE", get( icase.getIDLType() ) );
		uobj.setName("name", icase.getName() );
		if(isdef) {
		    uobj.setValue("discr", new Long(0));
		} else {
		    long l = 0;
		    if( olabel instanceof java.math.BigDecimal ) {
			l =((java.math.BigDecimal)olabel).longValue();
		    } else if( olabel instanceof java.math.BigInteger ) {
			l =((java.math.BigInteger)olabel).longValue();
		    } else if( olabel instanceof java.lang.String ) {
			l =(long)(((String)olabel).charAt(0));
		    } else {
			System.err.println("ERROR: unsupported union switch type "+olabel.getClass().getName());
		    }
		    uobj.setValue("discr", new Long(l));
		}	
		obj.addToContentList("ELEMENTS", uobj);
	    }
	}

	obj.setValue("defdiscr", null); // trigger automatic setting!
    }


    private static void init_object(IDLObject obj, IDLNode node) 
	throws IDLflexException
    {
	if( node == null ) {
	    System.err.println("Error: FimoObjectCreater.init_object called "+
			       "with node==null");
	    return;
	}
	if(verbose) {
	    System.err.println("FimoObjectCreator::init_object: "+
			       "called with type "+node);
	    System.err.println("class is "+node.getClass().getName());
	}
	/*
	if(node instanceof IDLNative) {
	   String name = ((IDLNative)node).getName();
	   obj.setName("name", name);
	   return;
	}*/

	/* Init global values for all kinds of objects */
	try {
	    IDLNamed n = (IDLNamed)node;
	    String name = n.getLocalName();
	    obj.setName("name", name);
	    /* TODO: FIXME
	       besser gar keine ID setzen, als eine falsche: 
	       dann ermittelt IDLObject seine ID selbst (IDL:.../1.0)
	       String id = n.getLocalName();
	       obj.setName("id", id);
	    */
	} catch(Exception ex) { 
	    if(verbose) System.err.println(" INFO: node without name: "+obj);
	}
	try {
	    /* TODO: das ist scho ned wirklich schn, oder? */
	    org.aspectix.IDLparser.tree.IDLEntityNode next =
		(org.aspectix.IDLparser.tree.IDLEntityNode)node;
	    while(true) {
		next = next.getFather();
		if(next==null) {
		    break;
		}
		// TODO: Add IDLValue...
		if(next instanceof IDLExcept ||
		   next instanceof IDLStruct ||
		   next instanceof IDLUnion ||
		   next instanceof IDLInterf ||
		   next instanceof IDLModule ||
	 	   next instanceof IDLValueType ||
		   next instanceof IDLSpec ) 
		{
		    break;
		}
	    };
	    if(next!=null) {
		IDLObject defined_in = get( next );
		obj.setContent("CONTAINER", defined_in);
	    }
	} catch(Exception e) {
	    System.err.println(" INFO: node without container: "+obj);
	}

	Iterator iter = null;
	/* Init common state of ContainerObj's (i.e. members) */
	if ( node instanceof IDLModule )
	    iter = ((IDLModule)node).getDefinitions();
	else if ( node instanceof IDLSpec )
	    iter = ((IDLSpec)node).getDefinitions();

	/*
	else if ( node instanceof IDLExcept )
	    iter = ((IDLExcept)node).getMembers();
	else if ( node instanceof IDLStruct ) 
	    iter = ((IDLStruct)node).getMembers();
	  (macht das sinn???) [orig. Typedef == Union/Struct/Alias/Enum]
	else if ( node instanceof IDLTypedef )
	else if ( node instanceof IDLUnion )
	    iter = ((IDLUnion)node).getCases();
	*/

	if(iter != null) {
	    while(iter.hasNext()) {
		IDLNode child = (IDLNode)iter.next();
		if(child instanceof IDLPreproc) {
			// Preprocessor pragma nodes need special handling
			String text = ((IDLPreproc)child).getText();
			// TODO: allow white space between pragma and prefix
			if(text.startsWith("#pragma prefix")) {
				text=text.substring(14);
				text=text.trim();
				text=text.substring(1,text.length()-1);
				obj.setName("prefix",text);
			}
		}	
		IDLObject iobj = get(child, 2);
		// What to do here?
		// (1) get() on Forward references returns null, which
		//     can simply be skipped
		// (2) reopening of modules requires special treatment:
		//     we have to add all new definitions in the idl tree
		//     node to the already created member element.
		//     currently, get(xxx,2) return null in that case.
		if(iobj!=null) {
		    obj.addToContentList("MEMBER", iobj);
		}
	    }
	}

	/* Now initialize values particular for my type of object */
	if (node instanceof IDLTypedef) {
	    /*case DefinitionKind._dk_Alias: */
	    IDLTypedef itd = (IDLTypedef)node;
	    IDLObject base = get( itd.getIDLType() );
	    obj.setContent("BASE", base);
	}
	else if (node instanceof IDLArray) {
	    IDLArray ard = (IDLArray)node;
	    IDLObject base = get( ard.getIDLType() );
	    int len = -1;
	    try {
		len = ard.getLength();
	    } catch(IDLException ex) {};
	    obj.setContent("BASE", base);
	    obj.setValue("length", new Integer(len));
	}
	else if (node instanceof IDLAttr) {
	    IDLAttr ad = (IDLAttr)node;
	    IDLObject base = get( ad.getIDLType() );
	    obj.setContent("BASE", base);
	    obj.setAttribute("readonly", ad.isReadonly() );
	    /* TODO: initialize is_public */
	}
	else if (node instanceof IDLConstDef) {
	    IDLConstDef cd = (IDLConstDef)node;
	    IDLObject base = null;
	    try {
		base = get( cd.getIDLType() );
	    } catch(IDLException ex) {
		System.err.println("constant: cannot get base type!!!");
	    }
	    obj.setContent("BASE", base);
	    try {
		// Fimo Konstanten:
		// Mit getValue() bekomme ich:
		// Einen String fr boolean (TRUE/FALSE), [w]char, [w]string
		// Ein BigDecimal fr alle numerischen Datentypen.
		// Ein BigInteger fr Enums... :-)
		// (auf jeden Fall ein java.lang.Number!)
		// ConstObj:value wird genau darauf gesetzt.
		// TODO: das muss noch sauber gemacht werden...
		Object o = cd.getValue();
		if(o instanceof String || o instanceof java.math.BigDecimal ||
		   o instanceof java.math.BigInteger ) 
		{
		    obj.setValue("value", o);
		} else {
		    throw new IDLflexException("ERROR: Unexpected Object in constant value for "+obj+": "+o+" / "+o.getClass());
		}
	    } catch(IDLException ex) {
		System.err.println("constant: cannot get value!!!");
	    }
	} else if (node instanceof IDLEnum) {
	    IDLEnum ed = (IDLEnum)node;
	    Iterator it = ed.getNames();
	    int i=0;
	    while(it.hasNext()) {
		IDLScopedName n = (IDLScopedName)it.next();
		IDLObject iobj = new EnumMemberObj();
		iobj.setName("name", n.getText());
		iobj.setValue("value", new Integer(i)); i++;
		iobj.setContent("SUPER", obj);
		iobj.setContent("CONTAINER", obj.getContent("CONTAINER"));
		obj.addToContentList("MEMBER", iobj);
	    }
	} else if (node instanceof IDLExcept) {
	    IDLExcept ed = (IDLExcept)node;
	    Iterator it = ed.getMembers();
	    while(it.hasNext()) {
		IDLTypeDecl n = (IDLTypeDecl)it.next();
		IDLObject idlo = new StructMemberObj();
		idlo.setName("name", n.getName() );
		IDLObject membertype = get( n.getIDLType() );
		idlo.setContent("BASE", membertype);
		idlo.setContent("CONTAINER", obj.getContent("CONTAINER"));
		obj.addToContentList("ELEMENTS", idlo);
	    }
	} else if (node instanceof IDLFixed) {
	    IDLFixed fd = (IDLFixed)node;
	    try {
		obj.setValue("digits", new Integer(fd.getTotal()));
		obj.setValue("scale", new Integer(fd.getFractional()));
	    } catch(IDLException ex) {
		System.err.println("fixed: failed to get information!");
	    }
	} else if (node instanceof IDLInterf) {
	    init_interface_object(obj, node);
	} else if (node instanceof IDLValueType) {
	    init_valuetype_object(obj, node);
	} else if (node instanceof IDLModule) {
	    /* Nothing to do (?) */
	} else if (node instanceof IDLNative) {
	    /* TODO: Insert Native */
	    System.err.println("WARNING: native statement found...");
	} else if (node instanceof IDLOpDecl) {
	    init_operation(obj, node);
	} else if (node instanceof IDLBaseType) {
	    init_primitive(obj, node);
	} else if (node instanceof IDLSequence) {
	    /* In the sequence TypeCode, bound==0 means unbounded sequence... */
	    IDLSequence sd = (IDLSequence)node;
	    try {
		int len = sd.getLength();
		if(len>0)
		    obj.setValue("bound", new Integer(len)); 
		else
		    obj.setValue("bound", new Integer(0));
	    } catch(Exception ex) {
		System.err.println("sequence: getLength failed!");
	    }
	    try {
		IDLObject base = get( sd.getIDLType() );
		obj.setContent("BASE", base);
	    } catch(Exception ex) {
		System.err.println("sequence: no base type!");
	    }
	} else if (node instanceof IDLString) {
	    IDLString is = (IDLString)node;
	    int len=-1;
	    try {
		len = is.getLength();
	    } catch(IDLException ex) {
		System.err.println("incomplete string definition");
	    }
	    init_string_obj(obj, node, len>0, is.isWide());
	} else if (node instanceof IDLStruct) {
	    IDLStruct ed = (IDLStruct)node;
	    Iterator it = ed.getMembers();
	    while(it.hasNext()) {
		IDLTypeDecl n = (IDLTypeDecl)it.next();
		IDLObject idlo = new StructMemberObj();
		idlo.setName("name", n.getName() );
		IDLObject membertype = get( n.getIDLType() );
		idlo.setContent("BASE", membertype);
		idlo.setContent("CONTAINER", obj.getContent("CONTAINER"));
		obj.addToContentList("ELEMENTS", idlo);
	    }
	} else if (node instanceof IDLTypedef) {
	    // nothing to do
	} else if (node instanceof IDLUnion) {
	    init_union(obj, node);
	} else if (node instanceof IDLSpec) {
	    // TODO: (probably nothing to do)
	}
    }
}
