/*
 * IDLflex -- A flexible, generic code generator for CORBA IDL
 *
 * Copyright (C) 2000  Hans Reiser, AspectIX Research Team
 *                     University of Erlangen-Nuernberg, Germany
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License.
 * The full text of the license can be found in the LICENSE.txt file.
 *
 * $Id: IDLObject.java,v 1.11 2002/02/26 10:46:10 reiser Exp $
 */

package org.aspectix.IDLflex.IRObj;

import java.lang.Object;
import org.aspectix.IDLflex.IDLflexException;
import org.aspectix.IDLflex.Debug;
import java.util.*;
import org.aspectix.IDLflex.IRObj.util.*;


/** common base class for all IDLObjects.
 * 
 * support for getName("name"/"fullname"/"id")
 *
 * for IR containers, getContent("CONTAINER") gets the IDL object
 * corresponding to this IR container.
 */
public class IDLObject {
    // public variables

    /** In idl-objects which directly correspond to an interface repository
     *  objects, this reference points to this interface repository object.
     *  The use of this variable is deprecated.
     */
    //public IRObject referenceObject;

    // Variables for access within IDLObject.IRObj package
    IDLObject defined_in;
    String name;
    String fullname;
    String id, prefix, version;
    Object defkind;   // in fact a org.omg.CORBA.DefKind object...

    // privat:
    //private Container ir_container;

    /** public constructor for an empty IDLObject (without name and ID).
     */
    public IDLObject() {
	//referenceObject = null;
	name = ""; 
	id = null; version = null; prefix = null;
	defined_in = null;
	defkind = null;
    }

    public IDLObject[] VectorToIDLObjects(Vector v) {
	IDLObject[] objs;
	int len = v.size();
	objs = new IDLObject[len];
	for(int i=0; i<len; i++) {
	    objs[i] = (IDLObject)(v.elementAt(i));
	}
	return objs;
    }

    /** public constructor for an IDLObject which is initialized via an
     * IRObject.
     * 
     * name and ID is initialized directly from the IR object.
     * the container idl object reference is initialized in the
     * init_members funktion.
     */
    /*
      public IDLObject(IRObject iro) {
	this();
	referenceObject = iro;
	if(iro==null) {
	    System.err.println("Error: IDLObject initializer called with argument null"); }
	else if (iro.def_kind()==null) {
	    System.err.println("Error: IDLObject initializer got def_kind null"); }
	
	kind= iro.def_kind().value();
	
	try {
	    Contained c = ContainedHelper.narrow(iro);
	    name = c.name();
	    // This is a HACK: Bugfix for JavaORB bug!
	    // (according to CORBA standard, IDL names do NOT start with
	    // an underscore)
	    if(name.startsWith("_")) name=name.substring(1);
	    
	    fullname = c.absolute_name();
	    id = c.id();
	    ir_container = c.defined_in();
	} catch(Exception e) {
	    // RepositoryDef, PrimitiveDef, ArrayDef, etc
	    // (unbenannte Objekte) diese haben keinen Namen.... :-)
	}
	finally
	{
	Debug.print( Debug.TRACE, Debug.IRO, 
	    "IDLObject(IRObject): n='"+name+"', fn='"+
	    fullname+"', id='"+id+"', kd="+kind
	    );
	    }
	    }
    */

    /**
     */
    public Object getValue(String spec) throws IDLflexException
    {
	if(spec.equalsIgnoreCase("defkind")) return defkind;
	throw new IDLflexException("IDLObject: getValue: value"+spec+
				   " does not exist!");
    }

    /** the general getName function accepts the parameters "name",
     * "fullname" and "id".
     */
    public String getName(String spec) throws IDLflexException {
	if(name==null) 
	    System.err.println("DEBUG: IDLObject:getName:: name==null");
	if(spec.equalsIgnoreCase("name")) return name;
	if(spec.equalsIgnoreCase("fullname")) {
	    String retval;
	    if(defined_in==null) retval=name;
	    else retval=defined_in.getName("fullname") + "::" + name;
	    return retval;
	}
	if(spec.equalsIgnoreCase("nameforid")) {
	    if(prefix!=null) {
		if(name==null||name.equals("")) return prefix;
		else return prefix+"/"+name;
	    }
	    if(defined_in==null) return name;
	    String s = defined_in.getName("nameforid");
	    if(s==null || s.equals("")) return name;
            return s+"/"+name;
	}
	if(spec.equalsIgnoreCase("id")) {
	    if(id!=null) return id;

	    StringBuffer ret = new StringBuffer();
	    ret.append("IDL:");
	    ret.append(getName("nameforid"));
	    ret.append(":");
	    if(version!=null) ret.append(version);
	    else ret.append("1.0");
	    id = ret.toString();
	    return id;
	}
	if(spec.equalsIgnoreCase("version")) {
	    if(version!=null) return version;
	    return "1.0";
	}
	if(spec.equalsIgnoreCase("prefix")) {
	    return prefix;
	}
	throw new IDLflexException("IDLObject getName: invalid name: "+spec);
    }

    /** the general getAttribute throws an Exception; all attributes
     *  are handled by the derived classes
     */
    public boolean getAttribute(String spec) throws IDLflexException {
	throw new IDLflexException("getAttribute: invalid spec: "+spec);
    }

    /** the general getContent function only supports the parameter
     *  "CONTAINER", which returns the object wherein the current object
     *  is contained in (not applicable for all objects).
     */
    public IDLObject getContent(String spec) throws IDLflexException {
	if(spec.equalsIgnoreCase("CONTAINER")) {
	    if(defined_in!=null) return defined_in;
	    return null;
	}
	throw new IDLflexException("IDLObject: getContent: no such spec: "+
				   spec+" in "+this);
    }
    /** the general getContentList throws an Exception; all Lists are
     *  handled by the derived classes.
     */
    public IDLObject[] getContentList(String spec) throws IDLflexException {
	throw new IDLflexException("IDLObject: getContentList: no such spec: "
				   +spec+" in "+this);
    }



    // generic setter methods
    public void setValue(String spec, Object value) 
	throws IDLflexException 
    {
        if (spec.equalsIgnoreCase("defkind")) defkind = value;
        else 
	    throw new IDLflexException("IDLObject: setValue: value" +
					spec + " does not exist!");
    }
    public void addToValueList(String spec, Object value) 
	throws IDLflexException 
    {
        throw new IDLflexException("IDLObject: addToValueList: value list " + 
				   spec + " does not exist!");
    }
    public void setName(String spec, String name) throws IDLflexException {
        if (spec.equalsIgnoreCase("name")) {
	    this.name = name;
	} else if (spec.equalsIgnoreCase("id")) {
            if (id != null) 
		throw new IDLflexException("Attempt to reset repository id!");
            id = name;
        } else if (spec.equalsIgnoreCase("prefix")) {
	    prefix = name;
	} else if (spec.equalsIgnoreCase("version")) {
            if ((id != null) || (version != null))
                throw new IDLflexException("Attempt to reset version!");
            version = name;
        }
        else throw new IDLflexException("IDLObject: setName: invalid name: "+
					spec);
    }

    public void setAttribute(String spec, boolean attribute)
	throws IDLflexException
    {
        throw new IDLflexException("IDLObject: setAttribute: invalid spec: "+
				   spec);
    }
    
    public void setContent(String spec, IDLObject content) 
	throws IDLflexException 
    {
        if (spec.equalsIgnoreCase("CONTAINER")) 
	    defined_in = content;
        else throw new IDLflexException("IDLObject: setContent: no such spec: "
					+ spec + " in " + this);
    }
    
    public void addToContentList(String spec, IDLObject content) 
	throws IDLflexException
    {
        throw new IDLflexException("IDLObject: addToContentList: no such spec: "
				   + spec + " in " + this);
    }


    /** Shortcut to {@link #is_a(String,boolean) is_a(spec, true)}.
     */
    public boolean is_a(String spec) {
	return is_a(spec, false);
    }


    /** Returns true if this object is of type what. If the second parameter
     *	is false the most derived type of this object is compared. Otherwise
     *	what may also be any base type.
     */
    public boolean is_a(String spec, boolean inherit) {
	if(spec.equalsIgnoreCase("IDLObject")) return true;
	else return false;		// IDLObject does not inherit...
    }

    public Iterator iterator() {
        Iterator i = (Iterator) IDLIteratorUtil.create(this);
        try {
            IDLObject[] v = this.getContentList("MEMBER");
            for(int n = 0; n < v.length; n++) {
                i = IDLIteratorUtil.concat(i, v[n].iterator());
            }
        } catch (org.aspectix.IDLflex.IDLflexException e) {
            // nothing to do -- just don't concat any new 
            // members to the Iterator
        } 
        return i;
    }

}



