/*
 * IDLflex -- A flexible, generic code generator for CORBA IDL
 *
 * Copyright (C) 2000  Hans Reiser, AspectIX Research Team
 *                     University of Erlangen-Nuernberg, Germany
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License.
 * The full text of the license can be found in the LICENSE.txt file.
 *
 * $Id: ObjectCreator.java,v 1.7 2001/11/21 16:14:39 reiser Exp $
 */

package org.aspectix.IDLflex.IRObj;

import org.omg.CORBA.IRObject;
import org.omg.CORBA.*;
import java.util.Hashtable;
import org.aspectix.IDLflex.IDLflexException;


/** This class is used to create IDL objects from Interface Repository
 *  objects. Depending on the type of the IR definition, the
 *  corresponding IDL object is created.  
 */
public class ObjectCreator {

    /** Hashtable for created IDLObjects.
     */
    static Hashtable ht;
    static {
	ht = new Hashtable();
    }

    /** Translate IDL object ID into a java.lang.Object used in lookup
     *  hashtable.
     */
    private static java.lang.Object getID(IRObject iro) {
	try {
	    Contained c=ContainedHelper.narrow(iro);
	    return c.id();
	} catch(Exception e) {};
	try {
	    IDLType idl = IDLTypeHelper.narrow(iro);
	    return idl.type();
	} catch(Exception e) {};
	return iro;
    }

    /** Method get tries to return a reference to an already existing object.
     *  if no such object is found, the behavior depends on the
     *  paramter create: if it is set, a new object will be created,
     *  otherwise null is returned.
     */
    public static IDLObject get(IRObject iro, boolean create) 
	throws IDLflexException 
    {
	IDLObject obj = (IDLObject)ht.get( getID(iro) );
	if(obj!=null) {
	    return obj;
	} else {
	    if(create) {
		IDLObject o = create_obj(iro);
		if(o!=null) {
		    // First we have to register the object. Some loop
		    // in the definitions might have to lookup the 
		    // current object in the hash table in its
		    // init_members function.
		    ht.put( getID(iro), o);
		    init_object(o, iro);
		    //o.init_members();
		} else {
		    System.err.println("ObjectCreator: create failed on "+iro);
		}
		return o;
	    } else {
		return null;
	    }
	}
    }

    /**	Just an alias to
     *	{@link #get(org.omg.CORBA.IRObject,boolean) get}(iro, true); 
     */
    public static IDLObject get(IRObject iro) 
	throws IDLflexException
    {
	if(iro==null) 
	    throw new IDLflexException("ObjectCreator: reference object is null");
	return get(iro, true);
    }




    /** Creates an IDLObject according to the type of the IRObject.
     */
    private static IDLObject create_obj(IRObject iro) {
	IDLObject idlobj;
	switch(iro.def_kind().value()) {
	case DefinitionKind._dk_Attribute:
	    idlobj = new AttributeObj();
	    break;
	case DefinitionKind._dk_Constant:
	    idlobj = new ConstantObj();
	    break;
	case DefinitionKind._dk_Exception:
	    idlobj = new ExceptionObj();
	    break;
	case DefinitionKind._dk_Interface:
	    idlobj = new InterfaceObj();
	    break;
	case DefinitionKind._dk_Module:
	    idlobj = new ModuleObj();
	    break;
	case DefinitionKind._dk_Operation:
	    idlobj = new OperationObj();
	    break;
	case DefinitionKind._dk_Typedef:
	    idlobj = new TypedefObj();
	    break;
	case DefinitionKind._dk_Alias:
	    idlobj = new AliasObj();
	    break;
	case DefinitionKind._dk_Struct:
	    idlobj = new StructObj();
	    break;
	case DefinitionKind._dk_Union:
	    idlobj = new UnionObj();
	    break;
	case DefinitionKind._dk_Enum:
	    idlobj = new EnumObj();
	    break;
	case DefinitionKind._dk_Primitive:
	    {
		PrimitiveDef pd = PrimitiveDefHelper.narrow(iro);
		if( pd.kind()==PrimitiveKind.pk_string)
		    idlobj = new StringObj();
		else if( pd.kind()==PrimitiveKind.pk_wstring)
		    idlobj = new StringObj();
		else
		    idlobj = new PrimitiveObj();
	    }
	    break;
	case DefinitionKind._dk_String:
	case DefinitionKind._dk_Wstring:
	    idlobj = new StringObj();
	    break;
	case DefinitionKind._dk_Sequence:
	    idlobj = new SequenceObj();
	    break;
	case DefinitionKind._dk_Array:
	    idlobj = new ArrayObj();
	    break;
	case DefinitionKind._dk_Repository:
	    idlobj = new ContainerObj();
	    break;
	case DefinitionKind._dk_Fixed:
	    idlobj = new FixedObj();
	    break;
	default:
	    return  null;
	}
	return idlobj;
    }

    private static void init_interface_object(IDLObject obj, IRObject iro) 
	throws IDLflexException
    {
	InterfaceDef id = InterfaceDefHelper.narrow(iro);

	InterfaceDef[] base = id.base_interfaces();
	for(int i=0; i<base.length; i++) {
	    InterfaceObj iobj = (InterfaceObj)ObjectCreator.get(base[i]);
	    obj.addToContentList("BASE", iobj);
	}
	
	Contained[] cseq = id.contents( DefinitionKind.dk_all, false );
	for(int i=0; i<cseq.length; i++) {
	    IDLObject iobj = ObjectCreator.get( cseq[i] );
	    obj.addToContentList("MEMBER", iobj);
	}

	/*
	cseq = id.contents( DefinitionKind.dk_all, true );
	for(int i=0; i<cseq.length; i++) {
	    IDLObject iobj = ObjectCreator.get( cseq[i] );
	    obj.addToContentList("ALLMEMBER", iobj;
	}
	*/
	
	boolean abstractif=false;
	TypeCode tc = id.describe_interface().type;
	if(tc.kind()==TCKind.tk_objref) abstractif=false;
	else if(tc.kind()==TCKind.tk_abstract_interface) abstractif=true;
	else System.err.println("InterfaceDef is neither tk_objref nor tk_abstract_interface");
	obj.setAttribute("abstractif", abstractif);
    }
    

    private static void init_string_obj(IDLObject obj, IRObject iro, 
				   boolean bound, boolean wide)
	throws IDLflexException
					       
    {
	obj.setAttribute("bound", bound);
	obj.setAttribute("wide", wide);
	if(bound) {
	    if(wide)
		obj.setValue("bound", new 
		    Integer(WstringDefHelper.narrow(iro).bound()));
	    else 
		obj.setValue("bound", new 
		    Integer(StringDefHelper.narrow(iro).bound()));
	}
	else
	    obj.setValue("bound", new Integer(-1));
    }


    private static void init_primitive(IDLObject obj, IRObject iro)
	throws IDLflexException
    {
	PrimitiveDef pd = PrimitiveDefHelper.narrow(iro);
	int TCKindValue = pd.type().kind().value();
	String name;
	switch(TCKindValue) {
	case TCKind._tk_boolean:
	    name = "tk_boolean"; break;
	case TCKind._tk_char:
	    name = "tk_char"; break;
	case TCKind._tk_wchar:
	    name = "tk_wchar"; break;
	case TCKind._tk_octet:
	    name = "tk_octet"; break;
	case TCKind._tk_string:
	    name = "tk_string"; break;
	case TCKind._tk_wstring:
	    name = "tk_wstring"; break;
	case TCKind._tk_short:
	    name = "tk_short"; break;
	case TCKind._tk_ushort:
	    name = "tk_ushort"; break;
	case TCKind._tk_long:
	    name = "tk_long"; break;
	case TCKind._tk_ulong:
	    name = "tk_ulong"; break;
	case TCKind._tk_longlong:
	    name = "tk_longlong"; break;
	case TCKind._tk_ulonglong:
	    name = "tk_ulonglong"; break;
	case TCKind._tk_float:
	    name = "tk_float"; break;
	case TCKind._tk_double:
	    name = "tk_double"; break;
	case TCKind._tk_fixed:
	    name = "tk_fixed"; break;
	case TCKind._tk_longdouble:
	    name = "tk_longdouble"; break;
	case TCKind._tk_TypeCode:
	    name = "tk_TypeCode"; break;
	case TCKind._tk_any:
            name = "tk_any"; break;
	case TCKind._tk_objref:
	    name = "tk_objref"; break;
	default:
	    name = "##UNSUPPORTED PRIMITIVE TCKIND VALUE "+TCKindValue+"##";
	}
	obj.setName("name", name);
    }

    private static void init_operation(IDLObject obj, IRObject iro)
	throws IDLflexException
    {
	OperationDef opd = OperationDefHelper.narrow(iro);
	
	// set param
	ParameterDescription[] pd = opd.params();
	//param = new IDLObject[pd.length];
	for(int i=0; i<pd.length; i++) {
	    ParameterObj paro = new ParameterObj();
	    paro.setName("name", pd[i].name);

	    if(pd[i].mode.equals(org.omg.CORBA.ParameterMode.PARAM_IN)) {
		paro.setAttribute("inarg", true);
	    } else if(pd[i].mode.equals(org.omg.CORBA.ParameterMode.PARAM_OUT)) {
		paro.setAttribute("outarg", true);
	    } else if(pd[i].mode.equals(org.omg.CORBA.ParameterMode.PARAM_INOUT)) {
		paro.setAttribute("inoutarg", true);
	    }

	    IDLObject base = ObjectCreator.get( pd[i].type_def );
	    paro.setContent("BASE", base);
	    paro.setContent("CONTAINER", obj);
	    obj.addToContentList("PARAM", paro);
	}
	
	// set except
	ExceptionDef[] ex = opd.exceptions();
	//except = new IDLObject[ex.length];
	for(int i=0; i<ex.length; i++) {
	    IDLObject except = ObjectCreator.get(ex[i]);
	    obj.addToContentList("EXCEPT", except);
	}
	
	// set return
	//System.err.println("result: "+opd.result());
	//System.err.println("result_def: "+fullname+" "+opd.result_def());
	IDLObject ret = ObjectCreator.get( opd.result_def() );
	obj.setContent("RETURN", ret);
    }

    private static void init_union(IDLObject obj, IRObject iro) 
	throws IDLflexException
    {
	UnionDef ud = UnionDefHelper.narrow(iro);
	IDLObject discr = ObjectCreator.get(ud.discriminator_type_def());
	obj.setContent("DISCR", discr);

	UnionMember[] umlist = ud.members();
	for(int i=0; i<umlist.length; i++) {
	    UnionMemberObj uobj = new UnionMemberObj();
	    boolean isdef = false;
	    try {
		if( umlist[i].label.type().kind().equals(TCKind.tk_octet) )
		    if( umlist[i].label.extract_octet()==0 ) isdef=true;
	    } catch(BAD_OPERATION e) {}
	    uobj.setAttribute("isdefault", isdef);
	    uobj.setContent("CONTAINER", obj);
	    uobj.setContent("SUPER", obj);
	    uobj.setContent("BASE", get(umlist[i].type_def) );
	    uobj.setName("name", umlist[i].name);
	    Any a = umlist[i].label;
	    Long lval;
	    if(isdef) {
		lval = new Long(0);
	    } else if(a.type().kind().equals(TCKind.tk_char)) {
		lval = new Long((long)a.extract_char());
	    } else if (a.type().kind().equals(TCKind.tk_wchar)) {
		lval = new Long((long)a.extract_wchar());
	    } else {
		String s = org.aspectix.IDLflex.Utility.extractAnyValue(a);
		lval = new Long(s);
	    }
	    uobj.setValue("discr", lval);
	    obj.addToContentList("ELEMENTS", uobj);
	}

	obj.setValue("defdiscr", null); // trigger automated computation
    }


    private static void init_object(IDLObject obj, IRObject iro) 
	throws IDLflexException
    {
	if( iro == null ) {
	    System.err.println("Error: ObjectCreater.init_object called with"+
			       " iro==null");
	    return;
	}

	/* Init global values for all kinds of objects */
	obj.setValue("defkind", iro.def_kind());
	try {
	    Contained c = ContainedHelper.narrow(iro);
	    String name = c.name();
	    // This is a HACK: Bugfix for JavaORB bug!
	    // (according to CORBA standard, IDL names do NOT start with
	    // an underscore)
	    if(name.startsWith("_")) name=name.substring(1);
	    obj.setName("name", name);
	    String id = c.id();
	    obj.setName("id", id);
	    IDLObject defined_in = ObjectCreator.get( c.defined_in() );
	    obj.setContent("CONTAINER", defined_in);

	    String fullname = c.absolute_name();
	    // JavaORB workaround
	    if(!fullname.startsWith("::")) fullname="::"+fullname;
	    String f2 = obj.getName("fullname");
	    if (!fullname.equals(f2)) {
		System.err.println("init_object:WARNING: fullname mismatch?!");
		System.err.println("f1: "+fullname);
		System.err.println("f2: "+f2);
	    }
	} catch(Exception e) {
	    System.err.println("init_object: no container found (harmless, no need to worry)...");
	    System.err.println(e.getMessage());
	    // nothing to worry, probably just an object without name..
	    // (like RepositoryDef, PrimitiveDef, ArrayDef, ...)
	}

	/* Init common state of ContainerObj's (i.e. members) */
	switch(iro.def_kind().value()) {
	case DefinitionKind._dk_Exception:
	case DefinitionKind._dk_Module:
	case DefinitionKind._dk_Struct:
	case DefinitionKind._dk_Typedef:
	case DefinitionKind._dk_Union:
	case DefinitionKind._dk_Value:
	case DefinitionKind._dk_Repository:
	    Container cont = ContainerHelper.narrow(iro);
	    Contained[] seq = cont.contents( DefinitionKind.dk_all, false );
	    for(int i=0; i<seq.length; i++) {
		IDLObject iobj = ObjectCreator.get(seq[i]);
		obj.addToContentList("MEMBER", iobj);
	    }
	    break;
	}

	/* Now initialize values particular for my type of object */
	switch(iro.def_kind().value()) {
	case DefinitionKind._dk_Alias:
	    {
		AliasDef ad = AliasDefHelper.narrow( iro );
		IDLObject base = ObjectCreator.get( ad.original_type_def() );
		obj.setContent("BASE", base);
		break;
	    }
	case DefinitionKind._dk_Array:
	    {
		ArrayDef ard = ArrayDefHelper.narrow(iro);
		IDLObject base = ObjectCreator.get(ard.element_type_def());
		int len = ard.length();
		obj.setContent("BASE", base);
		obj.setValue("length", new Integer(len));
		break;
	    }
	case DefinitionKind._dk_Attribute:
	    {
		AttributeDef ad = AttributeDefHelper.narrow(iro);
		IDLObject base = ObjectCreator.get( ad.type_def() );
		obj.setContent("BASE", base);
		if(ad.mode()==org.omg.CORBA.AttributeMode.ATTR_NORMAL)
		    obj.setAttribute("readonly", false);
		else
		    obj.setAttribute("readonly", true);
		/* TODO: initialize is_public */
		break;
	    }
	case DefinitionKind._dk_Constant:
	    {
		ConstantDef cd = ConstantDefHelper.narrow(iro);
		IDLObject base = ObjectCreator.get( cd.type_def() );
		obj.setContent("BASE", base);
		obj.setValue("value", cd.value());
		break;
	    }
	case DefinitionKind._dk_Enum:
	    {
		EnumDef ed = EnumDefHelper.narrow(iro);
		String[] memlist = ed.members();
		for(int i=0; i<memlist.length; i++) {
		    IDLObject iobj = new EnumMemberObj();
		    iobj.setName("name", memlist[i]);
		    iobj.setValue("value", new Integer(i));
		    iobj.setContent("PARENT", obj);
		    iobj.setContent("CONTAINER", obj.getContent("CONTAINER"));
		    obj.addToContentList("MEMBER", iobj);
		}
		break;
	    }
	case DefinitionKind._dk_Exception:
	    {
		ExceptionDef ed = ExceptionDefHelper.narrow(iro);
		StructMember[] smlist = ed.members();
		for(int i=0; i<smlist.length; i++) {
		    IDLObject idlo = new StructMemberObj();
		    idlo.setName("name", smlist[i].name );
		    IDLObject membertype = 
			ObjectCreator.get( smlist[i].type_def );
		    idlo.setContent("BASE", membertype);
		    idlo.setContent("CONTAINER", obj.getContent("CONTAINER"));
		    obj.addToContentList("ELEMENTS", idlo);
		}
		break;
	    }
	case DefinitionKind._dk_Fixed:
	    {
		FixedDef fd = FixedDefHelper.narrow(iro);
		obj.setValue("digits", new Integer(fd.digits()));
		obj.setValue("scale", new Integer(fd.scale()));
		break;
	    }
	case DefinitionKind._dk_Interface:
	    {
		init_interface_object(obj, iro);
		break;
	    }
	case DefinitionKind._dk_Module:
	    break;

	// TODO: Insert Native

	case DefinitionKind._dk_Operation:
	    init_operation(obj, iro);
	    break;
	    
	case DefinitionKind._dk_Primitive:
	    {
		PrimitiveDef pd = PrimitiveDefHelper.narrow(iro);
		// String Obj: refObj, IsBound, IsWide
		if( pd.kind()==PrimitiveKind.pk_string) {
		    init_string_obj(obj, iro, false, false);
		} else if( pd.kind()==PrimitiveKind.pk_wstring) {
		    init_string_obj(obj, iro, false, true);
		} else {
		    init_primitive(obj, iro);
		}
		break;
	    }
	case DefinitionKind._dk_Sequence:
	    {
		SequenceDef sd = SequenceDefHelper.narrow(iro);
		obj.setValue("bound", new Integer(sd.bound())); 
		IDLObject base = ObjectCreator.get( sd.element_type_def() );
		obj.setContent("BASE", base);
		break;
	    }
	case DefinitionKind._dk_String:
	    init_string_obj(obj, iro, true, false);
	    break;
	case DefinitionKind._dk_Wstring:
	    init_string_obj(obj, iro, true, true);
	    break;
	case DefinitionKind._dk_Struct:
	    {
		StructDef sd = StructDefHelper.narrow(iro);
		StructMember[] smlist = sd.members();
		for(int i=0; i<smlist.length; i++) {
		    IDLObject idlo = new StructMemberObj();
		    idlo.setName("name", smlist[i].name );
		    IDLObject membertype = ObjectCreator.get( smlist[i].type_def );
		    idlo.setContent("BASE", membertype);
		    idlo.setContent("CONTAINER", obj.getContent("CONTAINER"));
		    obj.addToContentList("ELEMENTS", idlo);
		}
		break;
	    }
	case DefinitionKind._dk_Typedef:
	    // nothing to do
	    break;
	case DefinitionKind._dk_Union:
	    // TODO
	    break;
	case DefinitionKind._dk_Repository:
	    // TODO: (probably nothing to do)
	    break;
	}
    }
}
