/*
 * IDLflex -- A flexible, generic code generator for CORBA IDL
 *
 * Copyright (C) 2000  Hans Reiser, AspectIX Research Team
 *                     University of Erlangen-Nuernberg, Germany
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License.
 * The full text of the license can be found in the LICENSE.txt file.
 *
 * $Id: UnionObj.java,v 1.7 2001/11/19 16:36:36 reiser Exp $
 */

package org.aspectix.IDLflex.IRObj;

import java.lang.Object;
import org.aspectix.IDLflex.IDLflexException;
import java.util.Enumeration;
import java.util.Vector;
import org.omg.CORBA.*;

/** implementation object of an IDL union.
 *
 * the list of contained definition elements can be accessed in two ways:
 * either a flat list (with some multiple entries which just differ by 
 * their labels) can be obtained, or a list with unique definition
 * entries, each of them having a sublist with their labels (if they
 * have multiple labels)
 *
 *<ul>
 * <li><em>getAttribute("needsdef")</em>
 *     true, if the union requires an implicit default declaration, i.e.
 *     if the existing labels do not cover completely the domain of the
 *     discriminator data type.
 * <li><em>getName/getValue("defdiscr")</em>
 *     returns the smalles possible value (index number) of the default
 *     label, either as a String or as an Long object.
 * <li><em>getContent("DISCR")</em>
 *     returns the IDL object defining the discriminator data type
 * <li><em>getContent("RDISCR")</em>
 *     returns the IDL object defining the discriminator data type.
 *     If the discriminator type is an alias, the type referenced by
 *     the alias is returned, i.e. the alias is resolved.
 * <li><em>getContentList("ELEMENTS")</em>
 *     returns the flat list of all contained UnionMemberObj objects
 * <li><em>getContentList("UNIQUE")</em>
 *     returns a list of all unique UnionMemberObj defining union
 *     entries.
 *</ul>
 */
public class UnionObj extends ContainerObj {
    Vector union_flat;
    Vector union_unique;
    IDLObject discriminator;
    IDLObject r_discriminator;  // alias-resolved discriminator 
    boolean implicit_default;
    long def_discr_value;


    public UnionObj() {
	union_flat = new Vector();
	union_unique = new Vector();
    }

    /** extract the value of union label from an Any */
    static long getLabelNumber(Object a) throws IDLflexException
    { 
	// Numeric switch types and Enum
	try {
	    return ((java.lang.Number)a).longValue();
	} catch(ClassCastException ex) {}

	// Boolean switch type
	if(a.equals("TRUE")) return 1;
	if(a.equals("FALSE")) return 0;

	// char/wchar switch type
	if (a instanceof String) return Long.parseLong((String)a);

	throw new IDLflexException("UnionObj:getLabelNumber: unsupported");
    }


    private static int value_maxval(IDLObject type) 
	throws IDLflexException
    {
	String tk=type.getClass().toString(); // just for debugging

	while(type instanceof AliasObj) {
	    type = type.getContent("RESOLVE");
	}

	if(type instanceof PrimitiveObj) tk=type.getName("name");
	else if (type instanceof StringObj) {
	    if(type.getAttribute("wide")) tk="tk_wstring";
	    else tk="tk_string";
	} else if (type instanceof EnumObj) {
	    tk="tk_enum";
	}

	if( tk.equals("tk_short") || tk.equals("tk_ushort") )
	    return 65535;
	if( tk.equals("tk_long") || tk.equals("tk_ulong") ||
	    tk.equals("tk_longlong") || tk.equals("tk_ulonglong") )
	    return 2000000000; // Not the correct value, but this doesn't
	                       // matter.
	if( tk.equals("tk_boolean")) return 1;
	if( tk.equals("tk_char") || tk.equals("tk_octet") )
	    return 255;
	if( tk.equals("tk_wchar") )
	    return 65535;
	if( tk.equals("tk_enum") ) {
	    return type.getContentList("MEMBER").length-1;
	}
	System.err.println("WARNING: UnionObj: unknown type range for discriminator of type"+tk);
	return 999999999;
    }


    void init_union_discr() throws IDLflexException
    {
	// Check if there is an explicit "default:" case
	implicit_default=true;
	Enumeration en = union_unique.elements();
	while(en.hasMoreElements()) {
	    UnionMemberObj umem = (UnionMemberObj)en.nextElement();
	    if( umem.getAttribute("isdefault") ) {
		implicit_default=false;
		break;
	    }
	}

	// If not, check if the cases span all possible values of the 
	// discriminator type
	if(implicit_default) {
	    if( union_flat.size() > value_maxval( discriminator ) ) {
		implicit_default=false;
		return; // No need to scan for default value - there is none!
	    }
	}

	// set def_discr_value to first value not used by any case. 
	// (Maybe this could be done more efficient!)
	// This does not consider negative values correctly! (todo)
	int max=value_maxval( discriminator );
    outer:
	for(def_discr_value=0; def_discr_value<max; def_discr_value++) {
	    en = union_flat.elements();
	    while(en.hasMoreElements()) {
		UnionMemberObj umem =  (UnionMemberObj)en.nextElement();
		try {
		    if( umem.getAttribute("isdefault") ) 
			continue;
		    if( ((Long)umem.getValue("discr")).longValue() == def_discr_value )
			continue outer;
		} catch(IDLflexException e) {
		    System.err.println("in UnionObj-init: "+e);
		    break outer;
		}
	    }
	    break;
	}
    }

    public boolean getAttribute(String spec) throws IDLflexException {
	if(spec.equalsIgnoreCase("needsdef")) 
	    return implicit_default;
	return super.getAttribute(spec);
    }

    // return default discriminator value numerically
    public String getName(String spec) throws IDLflexException 
    {
	if(spec.equalsIgnoreCase("defdiscr")) 
	    return ""+def_discr_value;
	return super.getName(spec);
    }

    public Object getValue(String spec) throws IDLflexException 
    {
	if(spec.equalsIgnoreCase("defdiscr")) 
	    return new Long(def_discr_value);
	return super.getValue(spec);
    }

    public IDLObject getContent(String spec) throws IDLflexException {
	if(spec.equalsIgnoreCase("DISCR")) 
	    return discriminator;
	else if(spec.equalsIgnoreCase("RDISCR")) 
	    return r_discriminator;
	else return super.getContent(spec);
    }

    public IDLObject[] getContentList(String spec) throws IDLflexException {
	if(spec.equalsIgnoreCase("ELEMENTS")) 
	    return VectorToIDLObjects(union_flat);
	else if(spec.equalsIgnoreCase("UNIQUE")) 
	    return VectorToIDLObjects(union_unique);
	else return super.getContentList(spec);
    }

    // generic setter methods
    public void setValue(String spec, Object val) throws IDLflexException 
    {
	if(spec.equalsIgnoreCase("defdiscr")) {
	    init_union_discr(); /// compute value automatically
	    if(val!=null) {
		// allow setting value defined by ObjectCreator
		def_discr_value = ((Long)val).longValue();
	    }
	}
	else super.setValue(spec, val);
    }

    public void setContent(String spec, IDLObject content)
	throws IDLflexException 
    {
	// TODO: Nur DISCR zulassen...
        if (spec.equals("DISCR") || spec.equals("RDISCR")) {
            discriminator = content;
            r_discriminator = content;
            while (r_discriminator.is_a("AliasObj")) {
                r_discriminator = r_discriminator.getContent("RESOLVE");
            }
        } else super.setContent(spec, content);
    }
    
    public void addToContentList(String spec, IDLObject content)
	throws IDLflexException 
    {
        if (spec.equals("ELEMENTS")) {
	    // first, add to "flat" structure...
	    IDLObject previous = null;
	    try {
		previous = (IDLObject)union_flat.lastElement();
	    } catch(Exception ex) {}
	    union_flat.addElement(content);
	    
	    // then, add a copy to "unique" structure
	    String casename = content.getName("name");
	    // TODO: maybe we should do a full search here?!
	    if( previous==null || 
		!casename.equals(previous.getName("name")) )
	    {
		UnionMemberObj uuobj = new UnionMemberObj();
		uuobj.setName("name", casename);
		uuobj.setContent("BASE", content.getContent("BASE"));
		uuobj.setValue("discr", content.getValue("discr"));
		uuobj.setContent("SUPER", this);
		uuobj.setContent("CONTAINER", this);
		union_unique.addElement(uuobj);
	    }
	    UnionMemberObj lastcase = (UnionMemberObj)union_unique.lastElement();
	    lastcase.addToContentList("ELEMENTS",content);
	    if(content.getAttribute("isdefault"))
		lastcase.setAttribute("isdefault", true);
	}
        else if (spec.equals("UNIQUE")) {
	    System.err.println("UnionObj: ERROR: addTCL called with UNIQUE");
	}
	else super.addToContentList(spec, content);
    }

    public boolean is_a(String spec, boolean inherit) {
	if(spec.equalsIgnoreCase("UnionObj")) return true;
	if(inherit) return super.is_a(spec, inherit);
	else return false;
    }
}
