
/*
 * IDLflex -- A flexible, generic code generator for CORBA IDL
 *
 * Copyright (C) 2000  Hans Reiser, AspectIX Research Team
 *                     University of Erlangen-Nuernberg, Germany
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License.
 * The full text of the license can be found in the LICENSE.txt file.
 *
 * $Id: Utility.java,v 1.13 2002/09/25 15:03:56 reiser Exp $
 */

package org.aspectix.IDLflex;

import java.util.*;
import org.omg.CORBA.*;
import org.w3c.dom.Node;
import org.w3c.dom.Element;
import org.aspectix.IDLflex.IRObj.*;

/**
 *
 * This base class provides the basic unterface for the language mapping
 * specific Utility class, as well as several auxiliar functions.
 *
 *<p>Interaction with the XML document:                      <br>
 *<ul>
 *<li><em>String getName(String what, IDLObject obj)</em><br>
 *       is called by &lt;GET T="what"/&gt;                
 * 
 *<li><em>boolean getAttribute(String what, IDLObject obj)</em><br>
 *       is called by &lt;IF ATTR="what"&gt; ... &lt;/IF&gt;  
 *     (or &lt;COND ATTR="what"/&gt;)                       
 *</ul>
 */
public abstract class Utility {
    IDLDefCompiler compiler=null;
    IDLObject refObj=null;
    private static Hashtable basic_mapping;
    private String[]    outerModules;
    //private Hashtable   outerMagic;
    private String      defaultModule;
    public  SaveBox 	savebox;

    // public constructor
    public Utility() { }

    /** Initialize the Utility class.
     *  
     *<p>This read the maptable "BasicMapping" to be used by
     *   get_basic_mapping()
     *<p>Furthermore, it initializes the SaveBox and the data structures
     *   necessary for handling the defines USEMODULE and DEFMODULE
     */
    public void init(IDLDefCompiler idlc) {
	compiler = idlc;
	basic_mapping = compiler.getMapTable("BasicMapping");

	savebox = new SaveBox();

	String s = IDLflexMain.argsAttributes.getDefined("USEMODULE");
	if(s!=null) {
	    StringTokenizer st = new StringTokenizer(s,":");
	    int cnt = st.countTokens();
	    outerModules = new String[cnt];
	    for(int i=0; i<cnt; i++) outerModules[i] = st.nextToken();
	} else outerModules = null;
	
	/*
	outerMagic = new Hashtable();
	outerMagic.put("CORBA", "omg.org"); // will be reversed!
        outerMagic.put("PortableServer", "omg.org");
	outerMagic.put("Messaging", "omg.org");
	outerMagic.put("DynamicAny", "omg.org");
	outerMagic.put("MessageRouting", "omg.org");
	outerMagic.put("PortableInterceptor", "omg.org");
	outerMagic.put("Dynamic", "omg.org");
	*/

	defaultModule = IDLflexMain.argsAttributes.getDefined("DEFMODULE");

	compiler.registerPlugin("Counter","org.aspectix.IDLflex.Counter");
    }

    public void setReferenceObject(IDLObject ref) { refObj = ref; }
    public IDLObject getReferenceObject() { return refObj; }

    private static int object2tckvalue(IDLObject obj) {
	if(obj!=null) {
	    if(obj.is_a("PrimitiveObj")) {
		try {
		    return ((Integer)obj.getValue("tckind")).intValue();
		} catch(IDLflexException ex) {
		    System.err.println("fatal error: PrimitiveObject does not"+
				       "have tckind value!");
		    throw new RuntimeException();
		}
	    } else if(obj.is_a("StringObj")) {
		try {
		    if(obj.getAttribute("wide")) return TCKind._tk_wstring;
		    else return TCKind._tk_string;
		} catch(Exception e) { e.printStackTrace(); };
	    } else if(obj.is_a("FixedObj")) {
		return TCKind._tk_fixed;
	    }
	}
	return -1;
    }

    /** Translate the argument, which should define a primitive data type,
     *  into the expression specified by the maptable "BasicMapping".
     *
     *  @exception IDLflex.IDLflexException
     *      if the argument does not correspond to a primitive data type
     */
    public final static String get_basic_mapping(IDLObject obj) 
	throws IDLflexException 
    {
	String basicname = null;
	if( obj instanceof PrimitiveObj )
	    basicname = obj.getName("name");
	else if (obj instanceof StringObj ) {
	    if(obj.getAttribute("wide")) basicname="tk_wstring";
	    else basicname="tk_string";
	} else throw new IDLflexException("not a basic type");
	String res = (String)basic_mapping.get(basicname);
	return res;
    }

    public final static String tckvalue2string(int tckvalue) 
    throws IDLflexException 
    {
	switch(tckvalue) {
	case TCKind._tk_null:
	    return "tk_null";
	case TCKind._tk_void:
	    return "tk_void";
	case TCKind._tk_boolean:
	    return "tk_boolean";
	case TCKind._tk_char:
	    return "tk_char";
	case TCKind._tk_wchar:
	    return "tk_wchar";
	case TCKind._tk_octet:
	    return "tk_octet";
	case TCKind._tk_string:
	    return "tk_string";
	case TCKind._tk_wstring:
	    return "tk_wstring";
	case TCKind._tk_short:
	    return "tk_short";
	case TCKind._tk_ushort:
	    return "tk_ushort";
	case TCKind._tk_long:
	    return "tk_long";
	case TCKind._tk_ulong:
	    return "tk_ulong";
	case TCKind._tk_longlong:
	    return "tk_longlong";
	case TCKind._tk_ulonglong:
	    return "tk_ulonglong";
	case TCKind._tk_float:
	    return "tk_float";
	case TCKind._tk_double:
	    return "tk_double";
	case TCKind._tk_fixed:
	    return "tk_fixed";
	case TCKind._tk_longdouble:
	    return "tk_longdouble";
	case TCKind._tk_TypeCode:
	    return "tk_TypeCode";
	case TCKind._tk_enum:
	    return "tk_enum";
	case TCKind._tk_any:
	    return "tk_any";
	case TCKind._tk_objref:
	    return "tk_objref";
	case TCKind._tk_struct:
	    return "tk_struct";
	case TCKind._tk_union:
	    return "tk_union";
	case TCKind._tk_sequence:
	    return "tk_sequence";
	case TCKind._tk_array:
	    return "tk_array";
	case TCKind._tk_alias:
	    return "tk_alias";
	case TCKind._tk_except:
	    return "tk_except";
	case TCKind._tk_abstract_interface:
	    return "tk_abstract_interface";
	case TCKind._tk_value:
	    return "tk_value";
	case TCKind._tk_value_box:
	    return "tk_value_box";
	case TCKind._tk_native:
	    return "tk_native";
	default:
	    throw new IDLflexException("Cannot find basic mapping for type "+
				       tckvalue );
	}
    }

    /** Returns the IDL name which correspond to a primitive data type
     * in the IDL representation.
     *
     *  @exception IDLflex.IDLflexException
     *      if the argument does not correspond to a primitive data type
     */
    public final static String get_primitive_name(IDLObject obj) 
	throws IDLflexException 
    {
	if(obj instanceof PrimitiveObj) {
	    String name = obj.getName("name");
	    if(name.equals("tk_objref")) 
		return "Object";
	    if(name.equals("tk_value"))
		return "Value";
	    return name.substring(3); // strip the "tk_"
	} else if (obj instanceof StringObj) {
	    return "String"; // ? braucht es das?
	} else
	    throw new IDLflexException("Cannot find basic mapping for "+obj);
    }
    
    public final static String extractUnionDiscrType(IDLObject obj)
	throws IDLflexException
    {
	TypeCode discrtype;
	if(obj.is_a("UnionMemberObj")) 
	    discrtype=(TypeCode)obj.getContent("SUPER").getValue("discrtype");
	else 
	    discrtype=(TypeCode)obj.getValue("discrtype");

	return tckvalue2string(discrtype.kind().value());
    }


    /*
    public final static String extractAnyType(Any a) throws IDLflexException
    {
	return tckvalue2string(a.type().kind().value());
    }
    */

    /** Simple helper function to replace \r \n \t in a string with
     *  their escape sequences. 
     */
    public static String escapeString(String input) {
	String x, retval="";
	StringTokenizer st = new StringTokenizer(input,"\r\n\t",true);
	while(st.hasMoreTokens()) {
	    x = st.nextToken();
	    if(x.equals("\r")) retval += "\\r";
	    else if(x.equals("\n")) retval += "\\n";
	    else if(x.equals("\t")) retval += "\\t";
	    else retval += x;
	}
	return retval;
    }


    /** Extract a value from an CORBA Any. This provides a usefull
     *  basic behavior for many data types. Handling of same specific
     *  data types requireing different treatment must be handled in
     *  the mapping specific Utility class.
     */
    public final static String extractAnyValue(Any a)
    {
	TypeCode tc = a.type();
	switch(tc.kind().value()) {
	case TCKind._tk_boolean: 
	    return a.extract_boolean()?"1":"0";
	case TCKind._tk_char:
	    return "'"+a.extract_char()+"'";
	case TCKind._tk_wchar:
	    return "'"+a.extract_wchar()+"'";
	case TCKind._tk_octet:
	    return String.valueOf( (int)a.extract_octet() );
	case TCKind._tk_string:
	    String s = a.extract_string();
	    // TODO: Better solution for this...
	    if(s.startsWith("\"")) {  // JavaORB bug workaround
		s=s.substring(1,s.length()-1);
	    }
	    return s;
	case TCKind._tk_wstring:
	    String st = a.extract_wstring();
	    if(st.startsWith("\"")) {  // JavaORB bug workaround
		st=st.substring(1,st.length()-1);
	    }
	    return st;
	case TCKind._tk_short:
	    return String.valueOf( a.extract_short() );
	case TCKind._tk_ushort:
	    return String.valueOf( a.extract_ushort() );
	case TCKind._tk_long:
	    return String.valueOf( a.extract_long() );
	case TCKind._tk_ulong:
	    return String.valueOf( a.extract_ulong() );
	case TCKind._tk_longlong:
	    return String.valueOf( a.extract_longlong() );
	case TCKind._tk_ulonglong:
	    return String.valueOf( a.extract_ulonglong() );
	case TCKind._tk_float:
	    return String.valueOf( a.extract_float() );
	case TCKind._tk_double:
	    return String.valueOf( a.extract_double() );
	case TCKind._tk_fixed:
	    return String.valueOf( a.extract_fixed() );
	case TCKind._tk_longdouble:
	    return "<longdouble: unsupported>";
	}
	System.err.println("extractAnyConst: Unsupportet Type "+tc.kind().value()+" in Any "+a);
	return "";
    }


    // name resolving via the interface repository
    private static Hashtable iro_resolver;
    static { iro_resolver = new Hashtable(); }

    /** Resolves the complete path to the top of the IDL representation.
     *
     * The list of names, which consists of the name of the specified
     * object and of all surrounding modules (or other containers), is
     * stored in an UtilityResolvedPath object.
     *
     * @see org.aspectix.IDLflex.UtilityResolvedPath */
    public final UtilityResolvedPath resolve(IDLObject obj) {
	if(iro_resolver.contains(obj)) {
	    return (UtilityResolvedPath)(iro_resolver.get(obj));
	} else {
	    UtilityResolvedPath p = real_resolve(obj);

	    // length<=1; object not contained in a module, add default module
	    // if no referenceObject, then it is not a IDL object, just
	    //    a named member object, and needn't add default module...

	    /* TODO.... (referenceObject so nicht ok; -> obj==Modul anders
	       testen..
	    if(p.length()<=1 && obj.referenceObject!=null) {
		if(obj.referenceObject.def_kind()!=DefinitionKind.dk_Module) {
		    if(defaultModule!=null) 
			p.add0(defaultModule, DefinitionKind.dk_Module);
		}
	    }
	    */

	    /*
	    if(outerMagic!=null) {
		String s = p.name(0);
		String magic = (String)outerMagic.get(s);
		if(magic!=null) {
		    StringTokenizer st = new StringTokenizer(magic,".");
		    while(st.hasMoreTokens()) {
			String tk = (String)st.nextToken();
			ModuleObj m = new ModuleObj();
			try {
			    m.setName("name", tk);
			} catch(IDLflexException ex) {};
			p.add0(tk, m);
		    }
		}
	    }
	    */
	    iro_resolver.put(obj, p);
	    return p;
	}
    }

    private final UtilityResolvedPath real_resolve(IDLObject obj) {
	UtilityResolvedPath p = null;

/*
	try {
	    // handle prefix omg.org
	    String prefix = obj.getName("prefix");
	    if(prefix!= null && (prefix.equals("omg.org") || prefix.equals("org.omg")) ) {
		p = new UtilityResolvedPath();
		p.add("org", new ModuleObj());
		p.add("omg", new ModuleObj());
		if(!obj.getName("name").equals(""))
		    p.add(obj);
		return p;
	    }
	} catch(Exception e) {};
*/

	try {
	    IDLObject idlo = obj.getContent("CONTAINER");
	    if(idlo!=null) {
		p = real_resolve( idlo );
	    }
	} catch(Exception e) {};
	
	if(p!=null) {
	    p.add(obj);
	}
	else {
	    p=new UtilityResolvedPath();
	    // falls -D USEMODULE=blaa::blubb -> blaa und blubb aussen rum
	    if(outerModules!=null) {
		for(int i=0; i<outerModules.length; i++) {
		    // TODO: berprfen, ob man das ModuleObj initialisieren
		    // sollte... (vermutlich nicht notwendig...)
		    p.add(outerModules[i], new ModuleObj());
		}
	    }
	}
	return p;
    }

    /** Called from the XML-document by <GET T="name"/> it is used to
     *  generate text for the argument "name" passed as the paramter
     *  what.
     *
     *  The current IDL reference object is passed as the second
     *  paramter
     *
     * @exception IDLflexException
     *     if the text generation failed (invalid 'what'-paramter)
     */
    public String getName(String what, IDLObject obj) 
    throws IDLflexException
    {
	String spec; String sub;
	int i = what.indexOf(":");
	if( i==-1 ) { spec=what; sub=""; }
	else { spec=what.substring(0,i); sub=what.substring(i+1); }

	if(spec.equalsIgnoreCase("DEF")) {
	    String s = IDLflexMain.argsAttributes.getDefined(sub);
	    if(s!=null) return s;
	    throw new IDLflexException("No such definition: "+sub);
	}
	else if(spec.equalsIgnoreCase("IDL")) {
	    if(sub.equals("")) return obj.getName("name");
	    else return obj.getName(sub);
	} 
	else if(spec.equalsIgnoreCase("SBOX")) { 
	    String s = savebox.get( sub );
	    if(s==null) {
		throw new IDLflexException("SBOX("+sub+") does not exist!");
	    }
	    return s;
	} else if(spec.equalsIgnoreCase("PLUGIN")) {
	    Plugin sp = compiler.getPlugin(sub);
	    return sp.getName(""); 
	}
	else if(spec.equalsIgnoreCase("NAME")) {
	    return obj.getName("name");
	}
	else if(spec.equalsIgnoreCase("LIST")) {
	    String field;
	    int j = sub.indexOf(":");
	    if( j==-1 ) { field="MEMBER"; }
	    else { 
		field=sub.substring(j+1);
		sub=sub.substring(0,j);
	    }
	    if(sub.equalsIgnoreCase("Count")) {
		IDLObject[] list = obj.getContentList(field);
		return ""+list.length;
	    }
	    throw new IDLflexException("list:"+sub+": invalid expression");
	}
	else if(spec.equalsIgnoreCase("LOOP")) {
	    IDLIterationPseudo iip = compiler.getIterationPseudo();
	    if(iip==null) {
		throw new IDLflexException("<GET LOOP:.../> used outside "+
					   "iteration context!");
	    }
	    if(sub.equalsIgnoreCase("Index")) {
		return String.valueOf( iip.index );
	    } else if(sub.equalsIgnoreCase("Count")) {
		return String.valueOf( iip.count );
	    } else if(sub.equalsIgnoreCase("RIndex")) {
		return String.valueOf( iip.count-iip.index-1 );
	    }
	    throw new IDLflexException("LOOP:"+sub+": invalid expression");
	} 
	throw new IDLflexException("Utility: getName: invalid spec: "+what
				   + " on Object "+obj);
    }

    /** Called from the XML-document by <IF ATTR="name"/> it is used
     *  to verify the attribute "name" passed in the paramter attr.
     *
     *  The current IDL reference object is passed as the second
     *  paramter
     *
     *  @exception IDLflexException
     *    if the verification of the attribute failes 
     *    (invalid 'attr'-paramter) 
     */
    public boolean getAttribute(String attr, IDLObject o)
	throws IDLflexException 
    {
	int pos;

	String spec; String sub;
	if( (pos=attr.indexOf(":"))>0 ) {
	    spec=attr.substring(0,pos); sub=attr.substring(pos+1);
	} else { spec=attr; sub=""; }

	if(spec.equalsIgnoreCase("DEF")) {
	    return IDLflexMain.argsAttributes.isDefined(sub);
	}

	if(spec.equalsIgnoreCase("LOOP")) {
	    IDLIterationPseudo iip = compiler.getIterationPseudo();
	    if(iip==null) {
		throw new IDLflexException("LOOP:... used outside of"+
					   "iteration context!");
	    }
	    if(sub.equalsIgnoreCase("First")) 
		return (iip.index==0);
	    else if(sub.equalsIgnoreCase("Notfirst")) 
		return (iip.index>0);
	    else if(sub.equalsIgnoreCase("Last")) 
		return (iip.index==iip.count-1);
	    else if(sub.equalsIgnoreCase("Notlast")) 
		return (iip.index<iip.count-1);
	    throw new IDLflexException("LOOP:"+sub+": invalid attribute");
	}

	if(spec.equalsIgnoreCase("HAVE")) {
	    if(sub.equalsIgnoreCase("RETURN")) {
		IDLObject robj = o.getContent("RETURN");
		if(robj==null) throw new IDLflexException("get Attribute: HAVE: no such Content RETURN in Object "+o);
		if(robj.is_a("PrimitiveObj")) {
		    if( robj.getName("name").equals("tk_void") ) return false;
		}
		return true;
	    }
		
	    IDLObject[] list = o.getContentList(sub);
	    if(list==null) throw new IDLflexException("getAttribute: HAVE: no such ContentList '"+sub+"' in Object "+o);
	    if(list.length>0) return true; else return false;
	}
	
	if(spec.equalsIgnoreCase("IDL")) {
	    return o.getAttribute(sub);
	}
	if(spec.equalsIgnoreCase("OBJ")) {
	    System.err.println("Use of deprecated 'OBJ:' -- "+
			       "use 'IDL:' instead!");
	    return o.getAttribute(sub);
	}
	if(spec.equalsIgnoreCase("CORBA")) {
		UtilityResolvedPath p = resolve(o);
		try {
			if(p.name(0).equals("org") &&
		   	   p.name(1).equals("omg") ) return true;
		} catch(Exception ex) {}
		return false;
	}

	throw new IDLflexException("Unsupported Attribute "+attr+"\n");
    }
}

	
