/*
 * IDLflex -- A flexible, generic code generator for CORBA IDL
 *
 * Copyright (C) 2000  Hans Reiser, AspectIX Research Team
 *                     University of Erlangen-Nuernberg, Germany
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License.
 * The full text of the license can be found in the LICENSE.txt file.
 *
 * $Id: IDLUtility.java,v 1.3 2001/11/26 17:09:23 aanenni Exp $
 */

package org.aspectix.IDLflex.mapping;

import java.util.Vector;
import java.util.StringTokenizer;
import org.omg.CORBA.*;
import org.aspectix.IDLflex.UtilityResolvedPath;
import org.aspectix.IDLflex.IDLflexException;
import org.aspectix.IDLflex.NameCollisionResolver;
import org.aspectix.IDLflex.IDLDefCompiler;
import org.aspectix.IDLflex.IRObj.*;
import org.aspectix.IDLflex.Utility;

/** This class provides Java-specific extensions to the base Utility
 * class.  
 */
public class IDLUtility extends org.aspectix.IDLflex.Utility {

    NameCollisionResolver suffixCollisionResolver;
    NameCollisionResolver keywordCollisionResolver;
    NameCollisionResolver fullCollisionResolver;

    static String currentPackagePrefix="";
    public void setReferenceObject(IDLObject obj) {
	super.setReferenceObject(obj);
	try {
	    if(obj!=null)
		currentPackagePrefix = getPackageName(obj);
	    else currentPackagePrefix = "";
	} catch(Exception e) { 
	    System.err.println(" IDLUtility: setReferenceObject failed: ");
	    e.printStackTrace();
	    currentPackagePrefix=""; 
	}
    }

    /** Initialization.
     *
     * After initializing the super class, this function creates instances
     * of the generic collision resolver paramterized for the Java
     * language mapping.
     */
    public void init(IDLDefCompiler compiler) {
	super.init(compiler);
    }
    
    private String constructName(IDLObject obj, String suffix) {
	return constructName(obj, suffix, "");
    };
    private String constructName(IDLObject obj, String suffix, String prefix) {
	try {
	    String s = get_basic_mapping(obj);
	    if(suffix.equals("")) return s;

	    // This is not needed any more:
	    // if we have a suffix for a Primitive/String/Fixed-Obj, this
	    // should be "Holder", and we create the name 
	    // org.omg.CORBA.XXXHolder, where XXX is the last part of the
	    // basic name convertet to uppercase
	    int dot = s.lastIndexOf(".");
	    if(dot>=0) s = s.substring(dot+1);
	    s = s.substring(0,1).toUpperCase() + s.substring(1);
	    return "org.omg.CORBA."+s+"Holder";
	} catch(Exception e) {};
	
	// Otherwise we have a "normal" composite IDL type.
	UtilityResolvedPath res=resolve(obj);
	if(res.length()==0) {
	    System.err.println("constructName: empty Path! obj="+obj);
	    return "";
	}

	int i;
	String retval = "";
	
	for(i=0; i<res.length()-1; i++) {
	    /*
	    if(res.defkind(i)==DefinitionKind.dk_Interface) {
		if( res.defkind(i+1)!=DefinitionKind.dk_Attribute &&
		    res.defkind(i+1)!=DefinitionKind.dk_Operation &&
		    res.defkind(i+1)!=DefinitionKind.dk_Constant )   
		    
		    retval += fullCollisionResolver.resolve(res.name(i))
			+ "Package.";
	    }
	    else if(res.defkind(i)==DefinitionKind.dk_Struct||
		    res.defkind(i)==DefinitionKind.dk_Union||
		    res.defkind(i)==DefinitionKind.dk_Exception) {
		retval += fullCollisionResolver.resolve(res.name(i))
		    + "Package.";
	    } else {
		retval += fullCollisionResolver.resolve(res.name(i))+".";
	    }
	    */
	    retval += res.name(i)+"::";
	};
	if(res.defkind(i)==DefinitionKind.dk_Attribute||
	   res.defkind(i)==DefinitionKind.dk_Operation) {
	    retval += prefix + res.name(i) 
		+ suffix;
	} else {
	    retval += prefix+res.name(i)+suffix;
	}
	return retval;
    }


    // getFullName:
    // fully scoped name of IDL element
    private String getFullName(IDLObject obj, String type) 
	throws IDLflexException	
    {	
	if(type.equals("")) {
	    return constructName(obj, "");
	}

	if(type.equals("basicidl")) {
	    return get_primitive_name(obj);
	}

	if(type.equals("decl")) {
	    //while( obj.is_a("AliasObj") ) {
	    //obj = obj.getContent("RESOLVE");
	    //}
	    if( obj.is_a("SequenceObj")|| obj.is_a("ArrayObj") ) {
		String s = getFullName( obj.getContent("BASE"), "decl" );
		return s+"[]";
	    }
	    return constructName(obj, "");
	}

	throw new IDLflexException("getFullName (type="+type+") failed");
    }

    private String getPackageName(IDLObject obj) 
	throws IDLflexException
    {
	String str = constructName(obj, "");
	int i = str.lastIndexOf(':');
	if(i==-1) return "";
	else return str.substring(0,i);
    }



    //----------------------------------------------------------------------
    /** This method is the overloaded method for &lt;GET &gt; text
     *  substitution.
     * I suppose that most of it can be removed in the IDL mapping
     *
     *  supported arguments for 'whatstr':
     *<ul><li><em>PACKAGEDEF</em>:
     *       Creates a 'package XXX;' statement, if the class which currently
     *       is created resides within a package.
     *    <li><em>PACKAGENAME</em>:
     *       Returns the name of the surrounding package of the specified
     *       object, i.e. all but the last part of its fully scoped name.
     *    <li><em>FILE:subspec</em>:
     *       Returns the filename corresponding to the class name returned
     *       by TYPE:subspec.
     *    <li><em>TYPE:subspec</em>:
     *       Creates a Java type declaration for the referenced IDL object
     *       If the object is located outside the package of the file 
     *       currently created, a fully scoped name is used. Name collision
     *       is done for all componentes of the fully scoped (or simple) 
     *       name.
     *    <li><em>CONSTVAL</em>:
     *       Value of a constant, formatted for Java syntax
     *    <li><em>DISCR:xxx</em>:
     *       xxx: num for numeric, sym for symbolic represenation of
     *       discriminator label (UnionMemberObj) or default discr (UnionObj)
     *    <li><em>DISCR_ADD_UNDERSCORE</em>:
     *       If exists a label called "discriminator" within an union, return
     *       an underscore '_'. Used for name collision resolving between
     *       union labels and the auto-generated 'discriminator' function.
     *</ul>
     *<p> valid values for 'subspec' are:
     *<ul><li><em>decl</em> basic data type class name
     *    <li><em>helper</em> helper class name
     *	  <li><em>holder</em> holder class name
     *	  <li><em>skeleton</em> skeleton class name
     *   <li><em>tie</em> tie class name
     *	  <li><em>signatureif</em> signature interface class name
     *	  <li><em>stub</em> stub class name
     *	  <li><em>newarray</em> declaration of array of objects of the
     *	          given class, to be used in a new clause. I.e.
     *            create "basicclassname[lenX][][]..[]". X is a
     *		  value obtained from the "Special"-Object named 
     *  	  INDEX, the number of [] pairs is obtained by the
     *		  dimensionality of the array/sequence.
     *	  <li><em>basicidl</em> IDL name of a basic IDL definition
     *</ul>
     */
    public String getName(String whatstr, IDLObject obj) 
	throws IDLflexException
    {
	// get Token name
	String what;
	String sub;
	int pos;
	if( (pos=whatstr.indexOf(":")) >=0 ) {
	    what=whatstr.substring(0,pos);
	    sub=whatstr.substring(pos+1);
	} else { what=whatstr; sub=""; }

	if(what.startsWith("FILE")) return sub;

	// write value of a constant
	if(what.equals("CONSTVAL")) {
	    Any constval = (Any)(obj.getValue("value"));
	    String typ = "sorry"; //Utility.extractAnyType(constval); to do!!!
	    if(typ.equals("tk_enum")) {
		/* JavaORB bug: the value of an Enum is not stored in a
		 * Any of type tk_enum, but of type tk_long. Thus, the
		 * XML document handles the Enum specially (it produces a
		 * from_int( <GET T="CONSTVAL"/> ) ). Thus, we have to
		 * return the numeric value of the Enum here, not the
		 * symbolic name.
		 */
		return ""+EnumObj.valueFromAny( constval );
	    } else if (typ.equals("tk_boolean")) {
		return constval.extract_boolean()?"true":"false";
	    } else if (typ.equals("tk_string")||typ.equals("tk_wstring")) {
		String s =Utility.extractAnyValue(constval);
		return "\""+Utility.escapeString(s)+"\"";
	    } else if (typ.equals("tk_char")||typ.equals("tk_wchar")) {
		// TODO: aus extract die ' ' raus tun...
		//return "'"+Utility.extractAnyValue(constval)+"'";
		return Utility.extractAnyValue(constval);
	    } else {
		return Utility.extractAnyValue(constval);
	    }
	}
	
	// get CORBA fully scoped names
	else if(what.equals("TYPE")) {
	    if(sub.equals("name")) {
		String objname = obj.getName("name");
		return objname;
	    }

	    String s = getFullName(obj, sub);

	    System.err.println("TYPE:"+sub+": current="+currentPackagePrefix);
	    System.err.println("       fullname="+s);

	    if(currentPackagePrefix.equals(""))
		return s;
	    if( s.startsWith(currentPackagePrefix) ) {
		if(s.lastIndexOf(":")>currentPackagePrefix.length())
		    return s;
		else
		    return s.substring( currentPackagePrefix.length()+1 );
	    } else return s;
	}

	return super.getName( whatstr, obj );
    }

    /** getAttribute implementation for Java mapping
     *  one special function is included here:
     *  the helper function mapping for a IDL interface requires tho
     *  create a narrow(java.lang.Object obj) method iff the interface
     *  is abstract or if it derived from any abstract class. The
     *  attribute "abstractbase" verifies this condition.
     */
    public boolean getAttribute(String attr, IDLObject obj) 
    throws IDLflexException {
	if(attr.equals("abstractbase")) {
	    if(!obj.is_a("InterfaceObj")) {
		throw new IDLflexException("attribute 'abstractbase' only "+
					   "available for InterfaceObj");
	    }
	    if(obj.getAttribute("abstractif")) return true;
	    IDLObject[] list = obj.getContentList("ALLBASE");
	    for(int i=0; i<list.length; i++)
		if( list[i].getAttribute("abstractif") ) return true;
	    return false;
	}
	return super.getAttribute(attr, obj);
    }
}
	
