/*
 * IDLflex -- A flexible, generic code generator for CORBA IDL
 *
 * Copyright (C) 2000  Hans Reiser, AspectIX Research Team
 *                     University of Erlangen-Nuernberg, Germany
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License.
 * The full text of the license can be found in the LICENSE.txt file.
 *
 * $Id: JavaFileWriter.java,v 1.3 2000/12/15 15:32:36 fzhauck Exp $
 */

package org.aspectix.IDLflex.mapping;

import java.io.*;
import java.util.StringTokenizer;

/** This class implements a simple state machine for formatting Java Code.
  * '{' increments indentation level, '}' decrements it,
  * ';' significates new line; comments are left unchanged
  * some whitespace is removed
  */
public class JavaFileWriter extends org.aspectix.IDLflex.IDLFileWriter
{
    final static int SKIPWS = 0;
    final static int CONST1 = 1;
    final static int CONST2 = 2;
    final static int CONST2b = 3;
    final static int CONST0 = 4;
    final static int NORMAL = 5;
    final static int STRING1 = 6;
    final static int STRING2 = 7;
    int indentlevel;
    int state;
    boolean last_was_newline = false;
    public JavaFileWriter() {
	super();
	indentlevel=0;
	state=SKIPWS;
    }

    private void indent() throws IOException {
	int len=indentlevel*4;
	String s;
	while(len>=8) { super.write("\t"); len-=8; }
	while(len>0) { super.write(" "); len--; }
    }
    public void write(String s) throws IOException {
	StringTokenizer st=new StringTokenizer(s,"/*;{} \r\n\t\"'",true);
	while(st.hasMoreTokens()) {
	    // super.write("<"+state+">");
	    String z = st.nextToken();
	    if(z.length()>1) {
		if(state==CONST2b) state=CONST2;
		else if(state==CONST0) state=NORMAL;
		else if(state==SKIPWS) state=NORMAL;
		super.write(z);
		continue;
	    }
	    char ch = z.charAt(0);
	    if(ch=='\r') {
		super.write("\n");
		indent();
		continue;
	    }
	    switch(state) {
	    case CONST1:
		if(ch=='\r'||ch=='\n') {
		    state=SKIPWS;
		    super.write("\n");
		    indent();
		} else super.write(z);
		break;
	    case CONST2:
		if(ch=='*') state=CONST2b;
		super.write(z);
		break;
	    case CONST2b:
		if(ch=='/') {
		    super.write(z+"\n");
		    state=SKIPWS;
		    indent();
		} else if(ch=='*') {
		    super.write(z); // stay at CONST2b
		} else {
		    state=CONST2;
		    super.write(z);
		}
		break;
	    case CONST0:
		if(ch=='/') state=CONST1;
		else if(ch=='*') state=CONST2;
		super.write(z);
		break;
	    case SKIPWS:
		//if(last_was_newline && ch=='\n') {
		   // super.write("\n");
		//}
		if(ch=='\r') super.write("\n");
		if(ch==' '||ch=='\t'||ch=='\r'||ch=='\n')		    
		    continue;
		state=NORMAL;
		// break-through to NORMAL...
	    case NORMAL:
		if(ch=='/') {
		    super.write(z); state=CONST0;
		} else if(ch==';') {
		    super.write(z+"\n");
		    state=SKIPWS;
		    indent();
		} else if(ch=='{') {
		    super.write(z+"\n");
		    state=SKIPWS;
		    indentlevel++;
		    indent();
		} else if(ch=='}') {
		    indentlevel--;
		    super.write("\n");
		    indent();
		    super.write("}\n");
		    state=SKIPWS;
		    indent();
		} else if(ch==' '||ch=='\t'||ch=='\r'||ch=='\n') {
		    super.write(" ");
		    state=SKIPWS;
		} else if(ch=='"') {
		    super.write(z);
		    state=STRING1;
		} else if(ch=='\'') {
		    super.write(z);
		    state=STRING2;
		} else if(ch==';') {
		    super.write(";\n");
		    state=SKIPWS;
		    indent();
		}
		else super.write(z);
		break;
	    case STRING1:
		if(ch=='"') state=NORMAL;
		super.write(z);
		break;
	    case STRING2:
		if(ch=='\'') state=NORMAL;
		super.write(z);
		break;
	    }
	    last_was_newline=(ch=='\n')?true:false;
	}
    }
}
