/*
 * IDLflex -- A flexible, generic code generator for CORBA IDL
 *
 * Copyright (C) 2000  Hans Reiser, AspectIX Research Team
 *                     University of Erlangen-Nuernberg, Germany
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License.
 * The full text of the license can be found in the LICENSE.txt file.
 *
 * $Id: JavaUtility.java,v 1.14 2002/03/25 18:13:09 aanenni Exp $
 */

package org.aspectix.IDLflex.mapping;

import java.util.Vector;
import java.util.StringTokenizer;
import org.omg.CORBA.*;
import org.aspectix.IDLflex.UtilityResolvedPath;
import org.aspectix.IDLflex.IDLflexException;
import org.aspectix.IDLflex.NameCollisionResolver;
import org.aspectix.IDLflex.IDLDefCompiler;
import org.aspectix.IDLflex.IRObj.*;
import org.aspectix.IDLflex.Utility;

/** This class provides Java-specific extensions to the base Utility
 * class.  
 */
public class JavaUtility extends org.aspectix.IDLflex.Utility {

    NameCollisionResolver suffixCollisionResolver;
    NameCollisionResolver keywordCollisionResolver;
    NameCollisionResolver fullCollisionResolver;

    static String currentPackagePrefix="";
    public void setReferenceObject(IDLObject obj) {
	super.setReferenceObject(obj);
	try {
	    if(obj!=null)
		currentPackagePrefix = getPackageName(obj);
	    else currentPackagePrefix = "";
	} catch(Exception e) { 
	    System.err.println(" JavaUtility: setReferenceObject failed: ");
	    e.printStackTrace();
	    currentPackagePrefix=""; 
	}
    }

    /** Initialization.
     *
     * After initializing the super class, this function creates instances
     * of the generic collision resolver paramterized for the Java
     * language mapping.
     */
    public void init(IDLDefCompiler compiler) {
	super.init(compiler);
	fullCollisionResolver = new NameCollisionResolver(compiler.getTable("ReservedSuffix"), compiler.getTable("ReservedWords") );
	suffixCollisionResolver = new NameCollisionResolver(compiler.getTable("ReservedSuffix"), null);
	keywordCollisionResolver = new NameCollisionResolver(null, compiler.getTable("ReservedWords"));
    }
    
    protected String constructName(IDLObject obj, String suffix) {
	return constructName(obj, suffix, "");
    };
    protected String constructName(IDLObject obj, String suffix, String prefix) {
	try {
	    if(obj instanceof PrimitiveObj && obj.getName("name").equals("tk_value")
	       && suffix.equals("") )
	    {
		return "java.io.Serializable";
	    }
	    String s = get_basic_mapping(obj);
	    if(suffix.equals("")) return s;
            // For most primitive types, we have to create a org.omg.CORBA.XY,
            // where X is the uppercase version of the basic name x, and Y is
            // the suffix. For complex basic types (Object, ValueBase), we just
	    // append to suffix to the mapped name.
            // We do not support prefixes for those cases (indeed, only Helper
            // and Holder classes should be required)
	    if(s.startsWith("org.omg")) {
		return s+suffix;
            } else {	
		// strip prefix e.g. for java.lang.String -> org.omg.CORBA.StringHolder
	        int dot = s.lastIndexOf(".");
	        if(dot>=0) s = s.substring(dot+1);
	        s = s.substring(0,1).toUpperCase() + s.substring(1);
	        return "org.omg.CORBA."+s+"Holder";
	    }
	} catch(Exception e) {
	};

	if(obj.is_a("NativeObj")) {
	    try {
		System.err.println("Access to native type "+obj.getName("fullname"));
	    } catch(IDLflexException ex) {};
	    System.err.println("Mapping to java.lang.Object!");
	    if(suffix.equals("")) return "java.lang.Object";
	    System.err.println("Access to suffix "+suffix+" not adequately supported.");
	}
	
	// Otherwise we have a "normal" composite IDL type.
	UtilityResolvedPath res=resolve(obj);
	if(res.length()==0) {
	    System.err.println("constructName: empty Path! obj="+obj);
	    return "";
	}

	int i;
	String retval = "";
	
	for(i=0; i<res.length()-1; i++) {
	    IDLObject resobj = res.obj(i);
	    if(resobj instanceof InterfaceObj) {
		if( !(resobj instanceof AttributeObj) &&
		    !(resobj instanceof OperationObj) &&
		    !(resobj instanceof ConstantObj) ) 
		{
		    retval += fullCollisionResolver.resolve(res.name(i))
			+ "Package.";
		}
	    }
	    else if( resobj instanceof StructObj ||
		     resobj instanceof ExceptionObj ||
		     resobj instanceof UnionObj ) 
	    {
		retval += fullCollisionResolver.resolve(res.name(i))
		    + "Package.";
	    } else {
		retval += fullCollisionResolver.resolve(res.name(i))+".";
	    }
	};
	
	if( res.obj(i) instanceof AttributeObj ||
	    res.obj(i) instanceof OperationObj ) {
	    // Usually, TYPE:name should be used for Operation and Attribute
	    // names...
	    retval += prefix + keywordCollisionResolver.resolve(res.name(i)) 
		+ suffix;
	} else {
	    retval += prefix+fullCollisionResolver.resolve(res.name(i))+suffix;
	}
	return retval;
    }

    // getJavaFullName: Name-Translation with Java:
    // supported types:  (basic has been removed...)
    //                  - 'basic'  -->  resolved name for objects
    //                             -->  basicmapping resolved type for primitives
    //                             -->  base type for array/sequence
    //                  - 'decl'   -->  like basic, but with [] appended for array/sequence,
    //                                  and with resolving of aliases
    //                  - 'holder', 'helper', 'operationif', 'signaturif', 'stub'
    //                             -->  basic name, with appended suffix

    protected String getJavaFullName(IDLObject obj, 
					  String type) 
	throws IDLflexException
    {
	if(type.equals("")) 
	    return constructName(obj, "");

	if(type.equals("holder")) {
	    // If 'obj' is an alias, we should use the Holder class of the
	    // original object -- unless this is an anonymous sequence or
	    // array object, where we have a proper Holder class..
	    //
	    // dl9rdz 26/02/2002 is this really true???
	    // It breaks compiling Dynamic.idl and messes up things!
	    // TODO: Check this.
	    // Current fix: If obj resolves to SequenceObj or ArrayObj in any
	    // number of iterations, use Holder of the original obj
	    IDLObject original = obj;
	    while( obj.is_a("AliasObj") ) {
		IDLObject newobj = obj.getContent("RESOLVE");
		if(newobj.is_a("SequenceObj")||newobj.is_a("ArrayObj")) {
		    obj = original;
		    break;
		}
		obj = newobj;
	    }
	    return constructName(obj, "Holder");
	}
	if(type.equals("helper"))
	    return constructName(obj, "Helper");
	if(type.equals("operationif")) {
	    boolean abstr = obj.getAttribute("abstractif");
	    return constructName(obj, abstr?"":"Operations");
	}
	if(type.equals("skeleton")) {
	    if( getAttribute("DEF:POA",obj) )
		return constructName(obj, "POA");
	    else 
		return constructName(obj, "ImplBase", "_");
	}
	if(type.equals("tie")) {
	    if( getAttribute("DEF:POA",obj) )
		return constructName(obj, "POATie");
	    else
		return constructName(obj, "Tie");
	}
	if(type.equals("signatureif"))
	    return constructName(obj, "");
	if(type.equals("stub")) {
	    return constructName(obj, "Stub", "_");
	}

	if(type.equals("newarray")) {
	    int nested=0;
	    while(true) {
		if(obj.is_a("AliasObj")) {
		    obj=obj.getContent("RESOLVE");
		} else if(obj.is_a("SequenceObj")||obj.is_a("ArrayObj")) {
		    obj = obj.getContent("BASE"); nested++;
		} else break;
	    }
	    String s = getJavaFullName( obj, "decl" ) + "[len" +
		getName("PLUGIN:INDEX", obj) + "]";
	    for(int i=1; i<nested; i++) s+="[]";
	    return s;
	}

	if(type.equals("basicidl")) {
	    return get_primitive_name(obj);
	}

	if(type.equals("decl")) {
	    while( obj.is_a("AliasObj") ) {
		obj = obj.getContent("RESOLVE");
	    }
	    if( obj.is_a("SequenceObj")|| obj.is_a("ArrayObj") ) {
		String s = getJavaFullName( obj.getContent("BASE"), "decl" );
		return s+"[]";
	    }
	    return constructName(obj, "");
	}

	throw new IDLflexException("getJavaFullName (type="+type+") failed");
    }

    protected String getJavaPath(IDLObject obj, String what)
	throws IDLflexException
    {
	String str = getJavaFullName(obj, what);
	str = str.replace('.',java.io.File.separatorChar);
	return str+".java";
    }

    protected String getJavaName(IDLObject obj, String what)
	throws IDLflexException
    {
	String str = getJavaFullName(obj, what);
	int i = str.lastIndexOf('.');
	return str.substring(i+1);
    }

    protected String getPackageName(IDLObject obj) 
	throws IDLflexException
    {
	String str = constructName(obj, "");
	int i = str.lastIndexOf('.');
	if(i==-1) return "";
	else return str.substring(0,i);
    }



    //----------------------------------------------------------------------
    /** This method is the overloaded method for &lt;GET &gt; text
     *  substitution.
     *
     *  supported arguments for 'whatstr':
     *<ul><li><em>PACKAGEDEF</em>:
     *       Creates a 'package XXX;' statement, if the class which currently
     *       is created resides within a package.
     *    <li><em>PACKAGENAME</em>:
     *       Returns the name of the surrounding package of the specified
     *       object, i.e. all but the last part of its fully scoped name.
     *    <li><em>FILE:subspec</em>:
     *       Returns the filename corresponding to the class name returned
     *       by TYPE:subspec.
     *    <li><em>TYPE:subspec</em>:
     *       Creates a Java type declaration for the referenced IDL object
     *       If the object is located outside the package of the file 
     *       currently created, a fully scoped name is used. Name collision
     *       is done for all componentes of the fully scoped (or simple) 
     *       name.
     *    <li><em>CONSTVAL</em>:
     *       Value of a constant, formatted for Java syntax
     *    <li><em>DISCR:xxx</em>:
     *       xxx: num for numeric, sym for symbolic represenation of
     *       discriminator label (UnionMemberObj) or default discr (UnionObj)
     *    <li><em>DISCR_ADD_UNDERSCORE</em>:
     *       If exists a label called "discriminator" within an union, return
     *       an underscore '_'. Used for name collision resolving between
     *       union labels and the auto-generated 'discriminator' function.
     *</ul>
     *<p> valid values for 'subspec' are:
     *<ul><li><em>name</em> unscoped name of current element (attribute,
     *                      method, struct/union/exception member, enum
     *                      element, const name. Keyword collisions are
     *                      resolved, Prefix/Suffix collisions are not.
     *    <li><em>decl</em> data type primitive or class name, fully scoped if
     *                      the definition is located outside of the current
     *                      java package defined by the inner-most FILE tag.
     *                      (Probably this has to be modified if we need to
     *                      generate inner classes - we don't at the moment.)
     *                      Both keyword and prefix/suffix collisions are
     *                      resolved. The be used by all classes/interfaces
     *                      and data types.
     *    <li><em>helper</em> helper class name
     *	  <li><em>holder</em> holder class name
     *	  <li><em>skeleton</em> skeleton class name
     *   <li><em>tie</em> tie class name
     *	  <li><em>signatureif</em> signature interface class name
     *	  <li><em>stub</em> stub class name
     *	  <li><em>newarray</em> declaration of array of objects of the
     *	          given class, to be used in a new clause. I.e.
     *            create "basicclassname[lenX][][]..[]". X is a
     *		  value obtained from the "Special"-Object named 
     *  	  INDEX, the number of [] pairs is obtained by the
     *		  dimensionality of the array/sequence.
     *	  <li><em>basicidl</em> IDL name of a basic IDL definition
     *</ul>
     */
    public String getName(String whatstr, IDLObject obj) 
	throws IDLflexException
    {
	// get Token name
	String what;
	String sub;
	int pos;

	if(!whatstr.startsWith("JAVA:")) {
	    return super.getName(whatstr, obj);
	}
	whatstr=whatstr.substring(5);

	if( (pos=whatstr.indexOf(":")) >=0 ) {
	    what=whatstr.substring(0,pos);
	    sub=whatstr.substring(pos+1);
	} else { what=whatstr; sub=""; }

	// generate "package x.y.z;" statement if within a scope
	if(what.equals("PkgDecl")) {
	    String s = getPackageName(obj);
	    if(s.equals("")) return "";
	    else return "package "+s+";";
	} else if(what.equals("PkgName")) {
	    // fully scoped Name of the surrounding package
	    return getPackageName(obj);
	}

	// write value of a constant
	else if(what.equals("ConstVal")) {
	    java.lang.Object constval = obj.getValue("value");
	    if(constval==null) {
		System.err.println("constval is null for "+obj);
	    }
	    IDLObject type = obj.getContent("base");

	    String typ = "";
	    if( type instanceof PrimitiveObj ) typ = type.getName("name");
	    if( type instanceof StringObj ) typ = "tk_string";
	    if( type instanceof EnumObj ) typ = "tk_enum";

	    if(typ.equals("tk_enum")) {
		/* JavaORB bug: the value of an Enum is not stored in a
		 * Any of type tk_enum, but of type tk_long. Thus, the
		 * XML document handles the Enum specially (it produces a
		 * from_int( <GET T="CONSTVAL"/> ) ). Thus, we have to
		 * return the numeric value of the Enum here, not the
		 * symbolic name.
		 */
		// In the constant, there should be a numeric value...
		return constval.toString();
	    } else if (typ.equals("tk_boolean")) {
		if (((String)constval).equalsIgnoreCase("TRUE") )
		    return "true";
		else
		    return "false";
	    } else if (typ.equals("tk_string")||typ.equals("tk_wstring")) {
		return "\""+(String)constval+"\"";
	    } else if (typ.equals("tk_char")||typ.equals("tk_wchar")) {
		//return "'"+Utility.extractAnyValue(constval)+"'";
		//return Utility.extractAnyValue(constval);
		return "'"+(String)constval+"'";
	    }
	    return constval.toString();   // Number or String
	}
	
	// write symbolic value of a union discriminator
	// (Enum is handled seperatly in the XML document)
	// [ DISCRVAL:num is the same as IDL:defdiscr resp. IDL:discr ]
	// [ DISCRVAL and DISCRVAL:sym are equivalent ]
	else if(what.equals("DiscrVal")) {
	    String which;
	    if(obj.is_a("UnionObj")) which="defdiscr"; else which="discr";
	    long value = ((Long)(obj.getValue(which))).longValue();
	    if(sub.equals("num")) 
		return String.valueOf(value);

	    IDLObject discr;
	    if(obj.is_a("UnionMemberObj")) 
		discr = obj.getContent("SUPER").getContent("RDISCR");
	    else
		discr = obj.getContent("RDISCR");
		
	    if(discr.is_a("PrimitiveObj")) {
		if(discr.getAttribute("tk_boolean"))
		    return value==0?"false":"true";
		else if(discr.getAttribute("tk_char")||
			discr.getAttribute("tk_wchar"))
		    return "'" + ((char)value) + "'";
		else
		    return String.valueOf(value);
	    } else {
		// This case should not happen, as Enum is handled in the
		// XML-document separately, aliases are already resolved
		// completly, and other complex data types are not supported
		// as discriminator; TODO: do something if this branch occurs.
		System.err.println("JavaUtility: DISCRVAL: unsupported discriminator type!");
	    }
	}

	// generate mapping specific file names
	else if(what.equals("FILE")) {
	    return getJavaPath(obj, sub);
	}

	// get JAVA mapped names
	else if(what.equals("TYPE")) {
	    if(sub.equals("name")) {
		String objname = obj.getName("name");
		String retval=keywordCollisionResolver.resolve(objname);
		if(retval.equals("")) {
		    System.err.println("JAVA:TYPE:name: empty name for "+obj);
		}
		return retval;
	    }

	    String s = getJavaFullName(obj, sub);
	    if(s==null) {
		System.err.println("DEBUG: JavaFullName is null for "+obj);
		System.err.println("DEBUG: Request is TYPE:"+sub);
		System.err.println("DEBUG: obj name is "+obj.getName("name"));
	    }

	    if(currentPackagePrefix.equals(""))
		return s;
	    if( s.startsWith(currentPackagePrefix) ) {
		if(s.lastIndexOf(".")>currentPackagePrefix.length())
		    return s;
		else
		    return s.substring( currentPackagePrefix.length()+1 );
	    } else return s;
	}

	if(obj.is_a("UnionObj")) {
	    if(what.equals("DiscrAddUnderscore")) {
		// iterate over all members. if an item is named discriminator
		// add an _ to it. repeat.
		String discr="discriminator";
		IDLObject[] ml = obj.getContentList("UNIQUE");
	    outer:
		while(true) {
		    for(int k=0; k<ml.length; k++) {
			if(ml[k].getName("name").equals(discr)) {
			    discr = "_" + discr;
			    continue outer;
			}
		    }
		    int i = discr.lastIndexOf("_");
		    if(i<0) return ""; else return discr.substring(0,i+1);
		}
	    }
	}
	return super.getName( whatstr, obj );
    }

    /** getAttribute implementation for Java mapping
     *  one special function is included here:
     *  the helper function mapping for a IDL interface requires tho
     *  create a narrow(java.lang.Object obj) method iff the interface
     *  is abstract or if it derived from any abstract class. The
     *  attribute "abstractbase" verifies this condition.
     */
    public boolean getAttribute(String attr, IDLObject obj) 
    throws IDLflexException {
	if(attr.equals("JAVA:AbstractBase")) {
	    if(!obj.is_a("InterfaceObj")) {
		throw new IDLflexException("attribute 'abstractbase' only "+
					   "available for InterfaceObj");
	    }
	    if(obj.getAttribute("abstractif")) return true;
	    IDLObject[] list = obj.getContentList("ALLBASE");
	    for(int i=0; i<list.length; i++)
		if( list[i].getAttribute("abstractif") ) return true;
	    return false;
	}
	return super.getAttribute(attr, obj);
    }
}
	
