package org.aspectix.IDLparser;

import antlr.*;
import java.io.*;
import org.aspectix.IDLparser.grammar.*;
import org.aspectix.IDLparser.tree.*;
import org.aspectix.IDLparser.pub.*;
import org.aspectix.IDLparser.util.*;

public class ValidatingParser {

    /* The abstract syntax tree (its root node)  */
    private SpecNode ast = null;

    private boolean verbose = false;

    public void setVerbose() {
        verbose = true;
    }

    /**
     * Print each node of the AST together with it's name, class and line number.
     */
    public void printParseInfo() throws IDLException {
        if(ast == null) throw new IDLException("error(ValidatingParser): no AST generated yet "+
                                               "- cannot print parse info");
        System.out.println("\nPRINTING PARSE INFO...");
        ast.printParseInfo();
    }
    
    /**
     * Make ANTLR print a tree representation of the AST using getText() 
     * and getType() methods for printout.
     * Note: The node's text member will be used in a strange way by the IDLCompiler, 
     * therefore occasional "bad" names or nil may occur in this output.
     */
    public void printTreeInfo() throws IDLException {
        System.out.println("\nPRINTING ANTLR TREE OUTPUT...");
        if(ast == null) throw new IDLException("error(ValidatingParser): no AST generated yet "+
                                               "- cannot print tree info");
        IDLTreeVisitor visitor = new IDLTreeVisitor();
        visitor.visit((CommonAST) ast);
    }

    /**
     * Parse and validate an IDL input file.
     * @param fileName input file name
     * @return a validated abstract syntax tree
     * @exception if file can't be found, if parse error, if error in parser
     */
    public IDLNode parse(InputStream instream) throws IDLException {
        if(verbose) System.out.println("\nParsing IDL file...");
	ast = createAST( instream );  
        if(verbose) System.out.println("\nSetting scope...");
        ast.setScope();  
        if(verbose) System.out.println("\nChecking syntax...");
        LookupUtil.instance().setAST(ast);
        ast.setSequence();
        ast.checkClashes();
        ast.resolveReferences();
        ast.checkIdentifierUsage();
        return (IDLNode)ast;
    }

    private SpecNode createAST(InputStream instream) throws IDLException {
        try {
            IDLParser parser = new IDLParser(new IDLLexer(instream));
            parser.setASTFactory((ASTFactory) new IDLNodeFactory()); 
            parser.setASTNodeType("org.aspectix.IDLparser.tree.IDLEntityNode");  
            /* start parser at rule "specification" = root of IDL grammar */
            parser.specification();
            /* get an abstract syntax tree from the parser */
            return (SpecNode)parser.getAST();  
        } catch(RecognitionException r) {
	    throw new IDLException("parse error at line "+LocationInfo.getLocationInfo(r.getLine())+":\n"+r.getMessage());
	    
	} catch(TokenStreamRecognitionException t) {
	    throw new IDLException("parse error at line "+LocationInfo.getLocationInfo(t.recog.getLine())+":\n"+t.getMessage());
        } catch(TokenStreamException s) {
	    throw new IDLException("error while parsing: "+s.getMessage());
        }
    }
}





