header {
	package org.aspectix.IDLparser.grammar;

	import org.aspectix.IDLparser.tree.*;
	import org.aspectix.IDLparser.pub.*;
	import org.aspectix.IDLparser.util.*;
}

/**
 *  This is a complete parser for the IDL language as defined
 *  by the CORBA 2.0 specification.  It will allow those who
 *  need an IDL parser to get up-and-running very quickly.
 *  Though IDL's syntax is very similar to C++, it is also
 *  much simpler, due in large part to the fact that it is
 *  a declarative-only language.
 *  
 *  Some things that are not included are: Symbol table construction
 *  (it is not necessary for parsing, btw) and preprocessing (for
 *  IDL compiler #pragma directives). You can use just about any
 *  C or C++ preprocessor, but there is an interesting semantic
 *  issue if you are going to generate code: In C, #include is
 *  a literal include, in IDL, #include is more like Java's import:
 *  It adds definitions to the scope of the parse, but included
 *  definitions are not generated.
 *
 *  Jim Coker, jcoker@magelang.com
 */
/* 
 * This file by Jim Coker has been extended by Andrea Nenni. It now 
 * represents the IDL language according to CORBA 2.1, with on additional
 * rule "native". Java-Code for creation of an abstract syntax tree has
 * been added.
 */
class IDLParser extends Parser;
options {
	exportVocab=IDL;
	buildAST = true;
	defaultErrorHandler=false;
//	analyzerDebug = true;
}

tokens {
	MODULE="module"<AST=ModuleNode>;
	INTERF="interface"<AST=InterfNode>;
	STRUCTURE="struct"<AST=StructNode>;
	UNION="union"<AST=UnionNode>;
	ENUM="enum"<AST=EnumNode>;
	RAISES="raises"<AST=RaisesNode>;
	TYPEDEF="typedef"<AST=TypedefNode>;
	CONSTANT="const"<AST=ConstDefNode>;
	ATTRIBUTE="attribute"<AST=AttrNode>;
	EXCEPT="exception"<AST=ExceptNode>;
	STRING="string"<AST=StringNode>;
	FIXED="fixed"<AST=FixedNode>;
	WSTRING="wstring"<AST=StringNode>;
	SEQUENCE="sequence"<AST=SequenceNode>;
	CONTEXT="context"<AST=ContextNode>;
	NATIVE="native"<AST=NativeNode>;
	VALUETYPE="valuetype"<AST=ValueTypeNode>;
	BASETYPE<AST=BaseTypeNode>;
	SPEC<AST=SpecNode>;	
	INHERIT<AST=InheritNode>;
	SUPPORT="supports"<AST=SupportNode>;
	ARRAY<AST=ArrayNode>;
	CONST_EXP<AST=ConstExpNode>;
	SCOPED_NAME<AST=ScopedNameNode>;
	TYPE_DECL<AST=TypeDeclNode>;
	OP_DECL<AST=OpDeclNode>;
	PARAM_DECL<AST=ParamDeclNode>;
	CASE_EXP<AST=CaseNode>;
	LITERAL<AST=LiteralNode>;
        PREPROC<AST=PreprocNode>;
}

specification
	:   (definition)+
        {   
            #specification = #([SPEC], #specification);
		}
	;


definition 
	:   (   preprocline   /* hr addition */
		| type_dcl SEMI!
		| const_dcl SEMI!
		| except_dcl SEMI!
		| ( ("local"|"abstract")? "interface" ) => interf SEMI!
		| module SEMI!
		| ( ("custom"|"abstract")? "valuetype" ) => value_type SEMI!
		)
	;

/* hr addition */
preprocline
	: ( p:PREPROC_DIRECTIVE! 
	    { 
		 #preprocline= #([PREPROC], #preprocline);
		 ((IDLEntityNode)#preprocline).setLine(((IDLEntityNode)#p).getLine()); 
		 #preprocline.setText(#p.getText()); 
            }
          );


module
	:   MODULE^ 
		name:identifier! 
		LCURLY! (definition)+ RCURLY!  
        { 
			#MODULE.setText(#name.getText());
        }
	;

interf
	:
	  (l:"local"! | a:"abstract"!)? INTERF^
		name:identifier!
		(i:opt_inheritance_spec b:LCURLY! (export)*  RCURLY!)?
		{
            	    #INTERF.setText(#name.getText());
        	    if(#i != null) {
				#INTERF.setInherit();
		    }
	            if(#a != null) {
				#INTERF.setAbstractIf();
		    }
		    if(#l != null) {
				#INTERF.setLocalIf();
		    }
	            if(#b == null) {
				#INTERF.setForwardDecl();
		    }
		}	
	;

export
	:   (   type_dcl SEMI!
		|   const_dcl SEMI!
		|   except_dcl SEMI!
		|   attr_dcl SEMI!
		|   op_dcl SEMI!
		)
	;

opt_inheritance_spec
	:   a:COLON! scoped_name_list  
        { 
            #opt_inheritance_spec = #([INHERIT], #opt_inheritance_spec);
            ((IDLEntityNode)#opt_inheritance_spec).setLine(((IDLEntityNode)#a).getLine());
        }
	| 
	;

scoped_name_list
	:    a:scoped_name (COMMA! b:scoped_name
			{
				((IDLEntityNode)#b).setUsed();
			}
		)* 
		{
			((IDLEntityNode)#a).setUsed();
		}
	;

scoped_name
{ String text = ""; }
	:   (a:SCOPEOP!)? b:identifier! 
		(SCOPEOP! c:identifier! 
			{
				text  = text.concat("::"+#c.getText());
			} 
		)*
        {
			text = #b.getText()+text;
			if(#a != null) text = "::"+text;
			#scoped_name = #([SCOPED_NAME], #scoped_name);
			#scoped_name.setText(text);
			((IDLEntityNode)#scoped_name).setLine(((IDLEntityNode)#b).getLine());
			if(#a != null) ((ScopedNameNode) #scoped_name).setScopeResolution();
        }
	;


const_dcl
	:   CONSTANT^ b:const_type a:identifier! ASSIGN! const_exp
        {
            #CONSTANT.setText(#a.getText());
			((IDLEntityNode)#b).setUsed();
        }
	;

const_type 
	:   int_or_float
	|   char_type 
	|   boolean_type 
	|   string_type 
	|   scoped_name
	|   wide_char_type
	|   wide_string_type
	|   fixed_pt_const_type
	;


/*   EXPRESSIONS   */
const_exp
	:   or_expr
	;

or_expr   
	:   xor_expr 
		(a:OR! or_expr                   
			{
				#or_expr = #([CONST_EXP], #or_expr);
				#or_expr.setText(#a.getText());
				((IDLEntityNode)#or_expr).setLine(((IDLEntityNode)#a).getLine());
			}
		)? 
	;



xor_expr
	:    and_expr   
        (  a:XOR! xor_expr  
      		{	
				#xor_expr = #([CONST_EXP], #xor_expr);
				#xor_expr.setText(#a.getText());
				((IDLEntityNode)#xor_expr).setLine(((IDLEntityNode)#a).getLine());
			}
        )?
	;


and_expr
	:    shift_expr         
		(  a:AND! 
			and_expr
			{	
				#and_expr = #([CONST_EXP], #and_expr);
				#and_expr.setText(#a.getText());
				((IDLEntityNode)#and_expr).setLine(((IDLEntityNode)#a).getLine());
			}
			
		)? 
	;

shift_expr 
	:   add_expr
	    (  a:SHIFT!
			shift_expr
			{	
				#shift_expr = #([CONST_EXP], #shift_expr);
				#shift_expr.setText(#a.getText());
				((IDLEntityNode)#shift_expr).setLine(((IDLEntityNode)#a).getLine());
			}

	    )?
	;

add_expr
	:    mult_expr 
		(  a:ADD_OP!  
			add_expr
			{	
				#add_expr = #([CONST_EXP], #add_expr);
				#add_expr.setText(#a.getText());
				((IDLEntityNode)#add_expr).setLine(((IDLEntityNode)#a).getLine());
			}	
			
		)?
	;

mult_expr
	:   unary_expr 
		(  a:MULT_OP! 
			mult_expr
			{	
				#mult_expr = #([CONST_EXP], #mult_expr);
				#mult_expr.setText(#a.getText());
				((IDLEntityNode)#mult_expr).setLine(((IDLEntityNode)#a).getLine());
			}
		)?
	;

unary_expr
	:    a:ADD_OP! primary_expr
		{	
			#unary_expr = #([CONST_EXP], #unary_expr);
			#unary_expr.setText(#a.getText());
			((IDLEntityNode)#unary_expr).setLine(((IDLEntityNode)#a).getLine());
		}
	|    b:TILDE! primary_expr
		{	
			#unary_expr = #([CONST_EXP], #unary_expr);
			#unary_expr.setText(#b.getText());
			((IDLEntityNode)#unary_expr).setLine(((IDLEntityNode)#b).getLine());
		}
	|    c:primary_expr
	;


/* Node of type TPrimaryExp serves to avoid inf. recursion on tree parse */
primary_expr
	:   a:scoped_name  	
		{
			((IDLEntityNode)#a).setUsed(); 
		}              
	|   literal               
	|   LPAREN! const_exp RPAREN!     
	;

literal
	:   a:integer_literal!
		{   #literal = #([LITERAL], #literal);
			#literal.setText(#a.getText());
            ((IDLEntityNode)#literal).setLine(((IDLEntityNode)#a).getLine());
            ((LiteralNode)#literal).setLiteralType(IDLTypeUtil.INTEGER);
		}
	|   b:string_literal!
		{   #literal = #([LITERAL], #literal);
			#literal.setText(#b.getText());
            ((IDLEntityNode)#literal).setLine(((IDLEntityNode)#b).getLine());
            ((LiteralNode)#literal).setLiteralType(IDLTypeUtil.STRING);
		}
	|   i:wstring_literal!
		{   #literal = #([LITERAL], #literal);
			#literal.setText(#i.getText());
            ((IDLEntityNode)#literal).setLine(((IDLEntityNode)#i).getLine());
            ((LiteralNode)#literal).setLiteralType(IDLTypeUtil.WSTRING);
		}
	|   c:character_literal!
		{   #literal = #([LITERAL], #literal);
			#literal.setText(#c.getText());
            ((IDLEntityNode)#literal).setLine(((IDLEntityNode)#c).getLine());
            ((LiteralNode)#literal).setLiteralType(IDLTypeUtil.CHAR);
		}
	|   k:wcharacter_literal!
		{   #literal = #([LITERAL], #literal);
			#literal.setText(#k.getText());
            ((IDLEntityNode)#literal).setLine(((IDLEntityNode)#k).getLine());
            ((LiteralNode)#literal).setLiteralType(IDLTypeUtil.WCHAR);
		}
	|   d:floating_pt_literal!
		{   #literal = #([LITERAL], #literal);
			#literal.setText(#d.getText());
            ((IDLEntityNode)#literal).setLine(((IDLEntityNode)#d).getLine());
            ((LiteralNode)#literal).setLiteralType(IDLTypeUtil.FLOAT);
		}
	|   e:boolean_literal!
		{   #literal = #([LITERAL], #literal);
			#literal.setText(#e.getText());
            ((IDLEntityNode)#literal).setLine(((IDLEntityNode)#e).getLine());
            ((LiteralNode)#literal).setLiteralType(IDLTypeUtil.BOOLEAN);
		}
	|   h:fixed_pt_literal!
		{   #literal = #([LITERAL], #literal);
			#literal.setText(#h.getText());
            ((IDLEntityNode)#literal).setLine(((IDLEntityNode)#h).getLine());
            ((LiteralNode)#literal).setLiteralType(IDLTypeUtil.FIXED_CONST);
		}
	;

/* Literals from the lexer */

integer_literal
	:   INT
	|   OCTAL      
	|   HEX     
	;

context_string_literal
	:  a:STRING_LITERAL!
		{
			#context_string_literal = #([LITERAL], #context_string_literal);
            #context_string_literal.setText(#a.getText());
            ((IDLEntityNode)#context_string_literal).setLine(((IDLEntityNode)#a).getLine());
            ((LiteralNode)#context_string_literal).setLiteralType(IDLTypeUtil.STRING);
		}
	;	

string_literal
	:  a:STRING_LITERAL (b:string_literal
			{
				#a.setText(#a.getText()+#b.getText());
			}
		)?
	;

wstring_literal
	:  a:WSTRING_LITERAL (b:wstring_literal
			{
				#a.setText(#a.getText()+#b.getText());
			}
		)?
	;

fixed_pt_literal
	:   FIXED_PT_LITERAL
	;

character_literal
	:   CHAR_LITERAL
	;

wcharacter_literal
	:   WCHAR_LITERAL
	;

floating_pt_literal
	:   FLOAT
	;

identifier
	:   a:IDENT!
		{
			#identifier = #([SCOPED_NAME], #identifier);
			((IDLEntityNode)#identifier).setUsed();
                        // dl9rdz additions start here (escaped identifiers)
			String temp = #a.getText();
			if(temp.startsWith("_")) temp=temp.substring(1);
			#identifier.setText(temp);
                        // dl9rdz additions end here
            ((IDLEntityNode)#identifier).setLine(((IDLEntityNode)#a).getLine());
		}
	;

boolean_literal
	:   "TRUE"      
	|   "FALSE"     
	;

const_exp_in_brackets
	: LBRACK! const_exp RBRACK!
	;

array_decl
{
	AST down;//
	ArrayNode newNode, begin = null;
}
	:   (a:const_exp_in_brackets!
			{
				newNode = new ArrayNode();
				newNode.setLine(((IDLEntityNode)#a).getLine());
				newNode.setFirstChild(#a);
				if(#array_decl != null) {
					down = #array_decl.getFirstChild();//
					while(down.getNextSibling() != null) down = down.getNextSibling().getFirstChild();//
					down.setNextSibling(newNode);//
					//newNode.addChild(#array_decl);
					((IDLEntityNode)#array_decl).setUsed();
				}
				else #array_decl = newNode;//
				//#array_decl = newNode;
			}
		)*
	;				
/*
  type_dcl
  :  TYPEDEF^ a:type_declarator	
  {
  /* type_declarator is a sibling-list of TypeDeclNodes. The following
	 Java-code puts each TypeDeclNode of this list separately under an
	 own TypedefNode, i.e. splits up the list into single Typedefs.    
	 TypedefNode currentNode = (TypedefNode)#type_dcl;
	 TypedefNode newNode;
	 IDLEntityNode previousChild = (IDLEntityNode)#a;
	 IDLEntityNode currentChild = (IDLEntityNode)#a.getNextSibling();
	 while(currentChild != null) {
	 previousChild.setNextSibling(null);
	 newNode = new TypedefNode();
	 newNode.setLine(((IDLEntityNode)#type_dcl).getLine());
	 newNode.setFirstChild(currentChild);
	 previousChild = currentChild;
	 currentChild = (IDLEntityNode)(previousChild.getNextSibling());
	 previousChild.setNextSibling(null);
	 currentNode.setNextSibling(newNode);
	 currentNode = newNode;
	 }	
	 }
	 |  struct_type
	 |  union_type  
	 |  enum_type
	 |  NATIVE^ b:identifier
	 {
	 #type_dcl.setText(#b.getText());
	 }
	 ;
*/

type_dcl
{
	TypedefNode newNode, current = null;
	TypedefNode begin = null;
	AST down;
}
	:  TYPEDEF^ a:type_spec! d:identifier! e:array_decl!
        (COMMA! b:identifier! c:array_decl!
			{
				newNode = new TypedefNode();
				((IDLEntityNode)#a).setUsed();
				newNode.setLine(((IDLEntityNode)#b).getLine());
				newNode.setText(#b.getText());
				if(#c == null) newNode.addChild(astFactory.dupTree(#a)); 
				else {
					/* The following part adds the type_spec to the end of the 
					   array_decl i.e. adds the given type to the array. */
					((IDLEntityNode)#c).setUsed();
					newNode.addChild(#c);
					down = #c.getFirstChild();
					while(down.getNextSibling() != null) down = down.getNextSibling().getFirstChild();
					down.setNextSibling(astFactory.dupTree(#a));
				}
				if(begin == null) begin = newNode;
				else current.setNextSibling(newNode);
				current = newNode;
			}
        )*
		{
			//#type_dcl = #([TYPEDEF], #type_dcl);
			((IDLEntityNode)#type_dcl).setLine(((IDLEntityNode)#d).getLine());
			#type_dcl.setText(#d.getText());
			((IDLEntityNode)#a).setUsed();
			if(#e == null) #type_dcl.addChild(#a);
			else  {
				/* The following part adds the type_spec to the end of the 
				   array_decl i.e. adds the given type to the array. */
				((IDLEntityNode)#e).setUsed();
				#type_dcl.addChild(#e);
				down = #e.getFirstChild();
				while(down.getNextSibling() != null) down = down.getNextSibling().getFirstChild();
				down.setNextSibling(#a);
			}  
            if(begin != null) #type_dcl.setNextSibling(begin); 
		}
	|  struct_type
	|  union_type  
	|  enum_type
	|  NATIVE^ i:identifier!
		{
			#type_dcl.setText(#i.getText());
		}
	;

type_declarator
/* The added Java-code splits up the comma-separated identifier list, i.e. in the 
   created AST, only separate type declarations with only one child will occur.
*/
{
	TypeDeclNode newNode, current = null;
	TypeDeclNode begin = null;
	AST down;
}
	:   a:type_spec! d:identifier! e:array_decl!
        (COMMA! b:identifier! c:array_decl!
			{
				newNode = new TypeDeclNode();
				((IDLEntityNode)#a).setUsed();
				newNode.setLine(((IDLEntityNode)#b).getLine());
				newNode.setText(#b.getText());
				if(#c == null) newNode.addChild(astFactory.dupTree(#a)); 
				else {
					/* The following part adds the type_spec to the end of the 
					   array_decl i.e. adds the given type to the array. */
					((IDLEntityNode)#c).setUsed();
					newNode.addChild(#c);
					down = #c.getFirstChild();
					while(down.getNextSibling() != null) down = down.getNextSibling().getFirstChild();
					down.setNextSibling(astFactory.dupTree(#a));
				}
				if(begin == null) begin = newNode;
				else current.setNextSibling(newNode);
				current = newNode;
			}
        )*
		{
			#type_declarator = #([TYPE_DECL], #type_declarator);
			((IDLEntityNode)#a).setUsed();
			((IDLEntityNode)#type_declarator).setLine(((IDLEntityNode)#d).getLine());
			#type_declarator.setText(#d.getText());
			if(#e == null) #type_declarator.addChild(#a);
			else  {
				/* The following part adds the type_spec to the end of the 
				   array_decl i.e. adds the given type to the array. */
				((IDLEntityNode)#e).setUsed();
				#type_declarator.addChild(#e);
				down = #e.getFirstChild();
				while(down.getNextSibling() != null) down = down.getNextSibling().getFirstChild();
				down.setNextSibling(#a);
			}  
            if(begin != null) #type_declarator.setNextSibling(begin); 
		}
	;

type_spec
	: simple_type_spec
	| struct_type
	| union_type
	| enum_type
	;

simple_type_spec
	:   base_type_spec
	|   sequence_type
	|   string_type 
	|   wide_string_type
	|   fixed_pt_type
	|   scoped_name 
	;

base_type_spec
	:   int_or_float            
	|   char_type    
        |   wide_char_type
	|   boolean_type            
	|   octet_type               
	|   any_type
	|   object_type
        |   valuebase_type
	;

struct_type
	:   STRUCTURE^            
		name:identifier! 
		LCURLY! (member)+ RCURLY! 
		{
			#STRUCTURE.setText(#name.getText());
		}
	;

member
	:   type_declarator SEMI!
	;

union_type
	:   UNION^             
		name:identifier!
		"switch"! LPAREN! a:switch_type_spec RPAREN!   
		LCURLY! switch_body RCURLY!
		{
			#UNION.setText(#name.getText());
			((IDLEntityNode)#a).setUsed();
		}
	;

switch_type_spec
	:   integer_type
	|   char_type
	|   boolean_type
	|   enum_type
	|   scoped_name 
	;

switch_body
	:   (case_expr)+
	;

case_expr
{
	AST down;
}
	:   a:case_labels! b:type_spec! c:identifier! d:array_decl! SEMI!
        {
            #case_expr = #([CASE_EXP], #case_expr);
			#case_expr.setText(#c.getText());
			((IDLEntityNode)#b).setUsed();
			if(#d == null) #case_expr.setFirstChild(#b);
			else  {
				/* The following part adds the type_spec to the end of the 
				   array_decl i.e. adds the given type to the array. */
				((IDLEntityNode)#d).setUsed();
				#case_expr.setFirstChild(#d);
				down = #d.getFirstChild();
				while(down.getNextSibling() != null) down = down.getNextSibling().getFirstChild();
				down.setNextSibling(#b);
			}
			#case_expr.addChild(#a);
            ((IDLEntityNode)#case_expr).setLine(((IDLEntityNode)#b).getLine());
			
		}
	;

case_labels
	:    (case_label)+
	;

case_label
	:   "case"! const_exp COLON!
	|   "default" COLON!
	;

enum_type
	:   ENUM^ name:identifier! LCURLY! identifier (COMMA! identifier)*  RCURLY!
		{
			#ENUM.setText(#name.getText());
		}	
	;

sequence_type
	:   SEQUENCE^
		LT! a:simple_type_spec (COMMA! const_exp)? GT!
		{
			((IDLEntityNode)#a).setUsed();
		} 
	;

string_type
	:   STRING^ (LT! const_exp GT!)?
	;

wide_string_type
	:   WSTRING^ (LT! const_exp GT!)?
	;


fixed_pt_type
	:   FIXED^ LT! const_exp COMMA! a:integer_literal! GT!
		{   IDLEntityNode literalNode  = new LiteralNode();
			literalNode.setText(#a.getText());
            literalNode.setLine(((IDLEntityNode)#a).getLine());
            ((LiteralNode)literalNode).setLiteralType(IDLTypeUtil.INTEGER);
			#fixed_pt_type.addChild(literalNode);
		}
	;

attr_dcl
/* The added Java-code splits up the comma-separated identifier list, i.e. in the 
   created AST, only separate attribute declarations with only one child will occur.
*/
{
	AttrNode newNode, current = null; 
	AttrNode begin = null;
}
	:   (a:"readonly"!)?
		ATTRIBUTE^ c:param_type_spec
		d:identifier! 
		(COMMA! b:identifier! 
            { 
				newNode = new AttrNode();
				newNode.setText(#b.getText());  	
				((IDLEntityNode)#c).setUsed(); 
				newNode.setFirstChild(astFactory.dupTree(#c));
				newNode.setLine(((IDLEntityNode)#b).getLine());
				if(#a != null) newNode.setReadonly();
                if(begin == null) begin = newNode;
                else current.setNextSibling(newNode); 
				current = newNode;
            }
		)*
		{
            #ATTRIBUTE.setText(#d.getText());
			((IDLEntityNode)#c).setUsed(); 
			if(#a != null) #ATTRIBUTE.setReadonly();
            if(begin != null) #ATTRIBUTE.setNextSibling(begin);
		}
	;

except_dcl
	:   EXCEPT^         
		name:identifier! 
		LCURLY! (member)* RCURLY!
		{
			#EXCEPT.setText(#name.getText());
		}
	;

op_dcl
	:   (a:"oneway"!)? b:op_type_spec 
		name:identifier!
		parameter_dcls 
		(raises_expr)? (context_expr)?
		{
			#op_dcl = #([OP_DECL], #op_dcl);
			((IDLEntityNode)#b).setUsed();
            #op_dcl.setText(#name.getText());
            if(#a != null) ((OpDeclNode)#op_dcl).setOneway();
            ((IDLEntityNode)#op_dcl).setLine(((IDLEntityNode)#name).getLine());
        }
	;


parameter_dcls
	:  LPAREN! ( param_dcl (COMMA! param_dcl)* )? RPAREN!
	;

param_dcl
	:  a:param_attribute! 
		b:param_type_spec 
		name:identifier!
        {  
			#param_dcl = #([PARAM_DECL], #param_dcl);
			((IDLEntityNode)#b).setUsed();
			#param_dcl.setText(#name.getText());
			((ParamDeclNode)#param_dcl).setAttribute(#a.getText());
			((IDLEntityNode)#param_dcl).setLine(((IDLEntityNode)#name).getLine());
        }
	;

param_attribute
	:   "in"        
	|   "out"       
	|   "inout"     
	;

op_type_spec
	:   void_type    
	|   param_type_spec
	;

void_type
	:  x:"void"!
		{
			#void_type = #([BASETYPE], #void_type);
	        #void_type.setText("void");
			((IDLEntityNode)#void_type).setLine(((IDLEntityNode)#x).getLine());    
		}
	;

raises_expr
	:   RAISES^ LPAREN! scoped_name_list RPAREN!   
	;

context_expr
	:   CONTEXT^ LPAREN!  context_string_literal (COMMA! context_string_literal)* RPAREN!
	;

param_type_spec 
	:   base_type_spec
	|   string_type
	|   wide_string_type
	|   fixed_pt_type
	|   scoped_name    
	;


/* The following rule is necessary to decide, whether a type beginning with "long" is 
   to be considered an integer or floating point type without using general lookahead > 2
*/
int_or_float
	: {(LA(2) != LITERAL_double) && ((LA(1)==LITERAL_short) || (LA(1)==LITERAL_long) 
			|| (LA(1)==LITERAL_unsigned))}? integer_type 
	| {(LA(2) == LITERAL_double) || (LA(1)==LITERAL_double) || (LA(1)==LITERAL_float)}? floating_pt_type
	;

floating_pt_type
	:   a:"float"! 
		{
			#floating_pt_type = #([BASETYPE], #floating_pt_type);
            #floating_pt_type.setText("float");
			((IDLEntityNode)#floating_pt_type).setLine(((IDLEntityNode)#a).getLine());
		}            
	|   b:"double"!    
		{
			#floating_pt_type = #([BASETYPE], #floating_pt_type);
            #floating_pt_type.setText("double");
			((IDLEntityNode)#floating_pt_type).setLine(((IDLEntityNode)#b).getLine());
		}  
	|    c:"long"! "double"!   
		{
			#floating_pt_type = #([BASETYPE], #floating_pt_type);
			#floating_pt_type.setText("long_double");
			((IDLEntityNode)#floating_pt_type).setLine(((IDLEntityNode)#c).getLine());
		}  
	;

integer_type
	:  x:"short"!
		{
			#integer_type = #([BASETYPE], #integer_type);
	        #integer_type.setText("short");
			((IDLEntityNode)#integer_type).setLine(((IDLEntityNode)#x).getLine());    
		}
	|  a:"long"! (y:"long"!)?
		{
			#integer_type = #([BASETYPE], #integer_type);
			if(#y != null) #integer_type.setText("long_long"); 
			else #integer_type.setText("long");
			((IDLEntityNode)#integer_type).setLine(((IDLEntityNode)#a).getLine()); 
		}
	|  z:"unsigned"! (b:"short"! | c:"long"!) (d:"long"!)?
		{
			String type = null;
			if((#b != null) && (#d != null)) {
				System.out.println("error(line "+
					((IDLEntityNode)#b).getLine()+"): there is no data type unsigned short long");
				System.exit(0);
			}
			if(#b != null) {
				type = "unsigned_short"; 
			}
			if(#c != null) {
				type = "unsigned_long";
			}
			if(#d != null) type = type.concat("_long");
			#integer_type = #([BASETYPE], #integer_type); 
			#integer_type.setText(type);
			((IDLEntityNode)#integer_type).setLine(((IDLEntityNode)#z).getLine()); 
		}
	;

fixed_pt_const_type
	:	a:"fixed"!  
		{
			#fixed_pt_const_type = #([BASETYPE], #fixed_pt_const_type);
	        #fixed_pt_const_type.setText("fixed");
			((IDLEntityNode)#fixed_pt_const_type).setLine(((IDLEntityNode)#a).getLine());    
		}
	;

char_type
	:   a:"char"!             
		{
			#char_type = #([BASETYPE], #char_type);
			#char_type.setText("char");
			((IDLEntityNode)#char_type).setLine(((IDLEntityNode)#a).getLine()); 
		}          
	;

wide_char_type
	:   a:"wchar"!             
		{
			#wide_char_type = #([BASETYPE], #wide_char_type);
			#wide_char_type.setText("wchar");
			((IDLEntityNode)#wide_char_type).setLine(((IDLEntityNode)#a).getLine()); 
		}          
	;


any_type
	:   a:"any"!
		{
			#any_type = #([BASETYPE], #any_type);
			#any_type.setText("any");
			((IDLEntityNode)#any_type).setLine(((IDLEntityNode)#a).getLine()); 
		}          
	;

object_type
	:   a:"Object"!
		{
			#object_type = #([BASETYPE], #object_type);
            #object_type.setText("Object");
			((IDLEntityNode)#object_type).setLine(((IDLEntityNode)#a).getLine()); 
		}          
	;

valuebase_type
        :   a:"ValueBase"!
                {
                        #valuebase_type = #([BASETYPE], #valuebase_type);
            #valuebase_type.setText("ValueBase");
                        ((IDLEntityNode)#valuebase_type).setLine(((IDLEntityNode)#a).getLine());
                }
        ;


octet_type
	:   a:"octet"!
		{ 
            #octet_type = #([BASETYPE], #octet_type);
            #octet_type.setText("octet");
			((IDLEntityNode)#octet_type).setLine(((IDLEntityNode)#a).getLine());
		}
	;

boolean_type
	:   a:"boolean"!
		{
            #boolean_type = #([BASETYPE], #boolean_type);
            #boolean_type.setText("boolean");
			((IDLEntityNode)#boolean_type).setLine(((IDLEntityNode)#a).getLine());
        } 
	;

value_type
	: (c:"custom"! | a:"abstract"!)?  VALUETYPE^
	      name:identifier!
	      ( 
	        ( (inherit:COLON! scoped_name_list)?
		  (supp:SUPPORT! scoped_name_list)? 
	          body:LCURLY! (value_element)* RCURLY!
		)
                | box:type_spec!
              )?
	      {
		   #VALUETYPE.setText(#name.getText());
		   if(#inherit != null) {
			 #VALUETYPE.setInherits();
		   }
		   if(#supp != null) {
		 	 #VALUETYPE.setSupports();
		   }
		   if(#c != null) {
			 #VALUETYPE.setCustom();
		   }
		   if(#a != null) {
			 #VALUETYPE.setAbstractValue();
		   }
		   if(#box != null) {
			 #VALUETYPE.setBoxed();
		   } else if (#body != null) {
		   } else {
		         #VALUETYPE.setForwardDecl();
		   }
	      }
	;


value_support_spec
	:   SUPPORT^ scoped_name_list  
        { 
//#            #value_support_spec = #([SUPPORT], #value_support_spec);
//x            ((IDLEntityNode)#value_support_spec).setLine(((IDLEntityNode)#a).getLine());
        }
	| 
	;


value_element
	: export
	| (a:"public" | b:"private")! type_spec! elmt:identifier! SEMI!
	| init_dcl
	;

init_dcl
	: "factory"! name:identifier!
	  LPAREN! ( init_param (COMMA! init_param)* )? RPAREN! SEMI!
	;

init_param
	: "in"! 
	  b:param_type_spec
	  name:identifier!
        {  
		#init_param = #([PARAM_DECL], #init_param);
		((IDLEntityNode)#b).setUsed();
		#init_param.setText(#name.getText());
		((ParamDeclNode)#init_param).setAttribute("in");
		((IDLEntityNode)#init_param).setLine(((IDLEntityNode)#name).getLine());
        }
	;

/*******  LEXER  *********/


class IDLLexer extends Lexer;
options {
	exportVocab=IDL;
	k=4;
	defaultErrorHandler=false;
}

/*** IGNORED INPUT ***/ 

/* WHITESPACE */

WS      
	:       (' '
        |       '\t'
        |       '\n'  { newline(); }
        |       '\r')
		{ $setType(Token.SKIP); }
	;


/* COMMENTS */

SL_COMMENT 
	:
        "//" 
        (~'\n')* '\n'
        { $setType(Token.SKIP); newline(); }
	;

ML_COMMENT 
	: 
        "/*"
        (       
            //            STRING_LITERAL
            //    |       CHAR_LITERAL
            //    |     
	        '\n' { newline(); }
		|       '*' ~'/'
		|       ~'*'
        )*
        "*/"
        { $setType(Token.SKIP);  }
	;



/*** IDENTIFIERS IN TOKEN SECTION OF PARSER - SEE ABOVE ***/

IDENT         //may an identifier begin with "_" in special cases? 
	:      ('_')? ASCII_ALPHABETIC (ASCII_ALPHABETIC|'_'|'0'..'9')*
	;

protected ASCII_ALPHABETIC
	:	'a'..'z'|'A'..'Z'
	|	'\300'..'\317'|'\321'..'\326'|'\330'..'\334'|'\337'
	|	'\340'..'\357'|'\361'..'\366'|'\370'..'\374'|'\377'
	;
/*
protected ASCII_GRAPHIC
	:	'\041'..'\057'|'\072'..'\100'|'\133'..'\140'|'\173'..'\176'
	|	'\241'..'\277'|'\327'|'\367'
	;
*/

/*** KEYWORDS ARE USED DIRECTLY IN THE PARSER ***/

MULT_OP
	:    STAR
	|    DIV        
	|    MOD        
	;

/*** RESERVED CHARACTERS ***/

SEMI
options {
	paraphrase = ";";
}
	:       ';'
	;

//QUESTION
//options {
//  paraphrase = "?";
//}
//        :       '?'
//        ;

LPAREN
options {
	paraphrase = "(";
}
	:       '('
	;

RPAREN
options {
	paraphrase = ")";
}
	:       ')'
	;

LBRACK
options {
	paraphrase = "[";
}
	:       '['
	;

RBRACK
options {
	paraphrase = "]";
}
	:       ']'
	;

LCURLY
options {
	paraphrase = "{";
}
	:       '{'
	;

RCURLY
options {
	paraphrase = "}";
}
	:       '}'
	;

OR
options {
	paraphrase = "|";
}
	:       '|'
	;

XOR
options {
	paraphrase = "^";
}
	:       '^'
	;

AND
options {
	paraphrase = "&";
}
	:       '&'
	;

COLON
options {
	paraphrase = ":";
}
	:       ':'
	;

COMMA
options {
	paraphrase = ",";
}
	:       ','
	;

DOT
options {
	paraphrase = ".";
}
	:       '.'
	;

ASSIGN
options {
	paraphrase = "=";
}
	:       '='
	;

NOT
options {
	paraphrase = "!";
}
	:       '!'
	;


LT
options {
	paraphrase = "<";
}
	:       '<'
	;

SHIFT
options {
	paraphrase = "shift operator";
}
	: "<<"
	| ">>"
	;

GT
options {
	paraphrase = ">";
}
	:       '>'
	;

protected DIV
options {
	paraphrase = "/";
}
	:       '/'
	;

ADD_OP
options {
	paraphrase = "+ or -";
}
	:       '+'
	| 	'-'
	;

TILDE
options {
	paraphrase = "~";
}
	:       '~'
	;

protected STAR
options {
	paraphrase = "*";
}
	:       '*'
	;

protected MOD
options {
	paraphrase = "%";
}
	:       '%'
	;


SCOPEOP
options {
	paraphrase = "::";
}
	:       "::"
	;


PREPROC_DIRECTIVE
options {
	paraphrase = "a preprocessor directive";
}

	:
        '#' 
        (~'\n')* '\n'
/*        { $setType(Token.SKIP); }*/
         { newline(); }
	;


CHAR_LITERAL    
options {
	paraphrase = "a character literal";
}
	:
        '\'' 
        ( ESC | ~'\'' ) 
        '\''
	;

WCHAR_LITERAL    
options {
	paraphrase = "a character literal";
}
	:
        "L'" 
        ( ESC | ~'\'' ) 
        '\''
	;

/*
  CONTEXT_STRING_LITERAL
  :
  '"'    
  ('a'..'z')|('A'..'Z') ('_'|'.'|'a'..'z'|'A'..'Z'|'0'..'9')* ('*')?
  '"'
  ;
*/

STRING_LITERAL  
options {
	paraphrase = "a string literal";
}
	:
        '"' 
        (ESC|~'"')* 
        '"'
	;


WSTRING_LITERAL  
options {
	paraphrase = "a wstring literal";
}
	:
        "L\"" 
        (ESC|~'"')* 
        '"'
	;


protected
ESC     
options {
	paraphrase = "an escape sequence";
}
	:       '\\'
		(       'n'
		|       't'
		|       'v'     
		|       'b'
		|       'r'
		|       'f'
		|       'a'     
		|       '\\'
		|       '?'     
		|       '\''
		|       '"'
		|       OCTDIGIT
			(
				/* Since a digit can occur in a string literal,
                                 * which can follow an ESC reference, ANTLR
                                 * does not know if you want to match the digit
                                 * here (greedy) or in string literal.
                                 * The same applies for the next two decisions
                                 * with the warnWhenFollowAmbig option.
                                 */
				options {
					warnWhenFollowAmbig = false;
				}
			:       OCTDIGIT
				(
					options {
						warnWhenFollowAmbig = false;
					}
				:       OCTDIGIT
				)?
			)?
		|   'x' HEXDIGIT
			(
				options {
					warnWhenFollowAmbig = false;
				}
			:       HEXDIGIT
			)?
		)
	;

protected
VOCAB
options {
	paraphrase = "an escaped character value";
}
	:       '\3'..'\377'
	;

protected
DIGIT
options {
	paraphrase = "a digit";
}
	:       '0'..'9'
	;

protected
OCTDIGIT
options {
	paraphrase = "an octal digit";
}
	:       '0'..'7'
	;

protected
HEXDIGIT
options {
	paraphrase = "a hexadecimal digit";
}
	:       ('0'..'9' | 'a'..'f' | 'A'..'F')
	;


/* octal literals are detected by checkOctal */

HEX     
options {
	paraphrase = "a hexadecimal value";
}

	:    ("0x" | "0X") (HEXDIGIT)+         
	;


INT
options {
	paraphrase = "a integer, float or fixed literal";
}
	: (DIGIT)+
		(
			('.' (DIGIT)*
	        	(
					(
						('d'|'D')
						{$setType(FIXED_PT_LITERAL);}	
					)
				|
					(
						(('e' | 'E') ('+' | '-')? (DIGIT)+)?
						{$setType(FLOAT);}		
					)
				)
			)	
		|
          	(('e' | 'E') ('+' | '-')? (DIGIT)+ {$setType(FLOAT);})
		)?
		
	;

FLOAT
options {
	paraphrase = "a float or fixed literal";
}
	: 
		( '.' (DIGIT)+
			(
				(('d'|'D') {$setType(FIXED_PT_LITERAL);})
			|
				(('e' | 'E') ('+' | '-')? (DIGIT)+)
			)?
		)
	;
