package org.aspectix.IDLparser.pub;

import java.util.*;

/**
 * The representation of an IDLScopedName as a Stack.
 */
public class IDLScopeStack extends Stack {

    public IDLScopeStack() {
        super();
    }

    /**
     * Create a Stack from a String, using a given separator.
     * I.e. the String "::a::b::c" with separator "::" results in 
     * the stack null -- a -- b -- c.
     * @param text which shall be represented as stack
     * @param separator by which the text shall be tokenized
     * @return stack representation of the text
     */
    public static IDLScopeStack createStack(String text, String separator) {
        IDLScopeStack stack = new IDLScopeStack();
        int index;

        if(separator == null) {
            stack.push((Object) text);
            return stack;
        }

        while((index = text.indexOf(separator)) != -1) {
            stack.push((Object) text.substring(0, index));
            text = text.substring(index+separator.length());
        }
        stack.push((Object) text);
        return stack;
    }
    
    /**
     * Returns true, if the identifier represented by this stack
     * is not scoped. I.e. this stack contains only one stack element.
     * @return true, if this stack has only one element
     */
    public boolean isSimpleIdentifier() {
        if(isEmpty()) return false;
        Object o = pop();
        try {
            peek();
        } catch (EmptyStackException e) {
            push(o);
            return true;
        }
        push(o);
        return false;
    }
    
    /**
     * Returns true, if the identifier represented by this stack is 
     * a simple reference to a base interface member, like "::name" (see CORBA
     * specification). I.e. this stack has only one null plus one 
     * further element.
     * @return true, if this stack represents a base interface reference
     */
    public boolean isSimpleSuperCall() {
        if(isEmpty()) return false;
        if(isSimpleIdentifier()) return false;
        Object o = pop();
        if(toString().equals("")) {
            push(o);
            return true;
        }
        else {
            push(o);
            return false;
        }
    }


    /**
     * Returns true, if the identifier represented by this stack is 
     * a short reference to the given identifier. (E.g. "B::i" is a 
     * reference to "A::B::i".)
     * @param name which is potentially referenced by this stack
     * @return true if this stack is a reference to the given stack
     */
    public boolean isShortReferenceTo(IDLScopeStack name) {
        if (name.toString().endsWith(this.toString())) return true;
        else return false;
    }

    /**
     * Get an exact copy of this stack.
     * @return a copy of this stack
     */
    /*
    public Object clone() {
        IDLScopeStack result = new IDLScopeStack();
        IDLScopeStack helper = new IDLScopeStack();
        Object o;

        try {
            while(true) {
                o = pop();
                helper.push(o);
            }  
        } catch(EmptyStackException e) {
            try {
                while(true) {
                    o = helper.pop();
                    result.push(((String)o).concat(""));
                    push(o);
                } 
            } catch(EmptyStackException i) {
                return (Object)result;
            }
        }
    }
    */

    /**
     * Returns true, if this stack is equal to the given stack.
     * @param o stack to be compared with this stack
     * @return true if this stack equals the argument
     */
    public boolean equals(Object o) {
        String compare = ((IDLScopeStack)o).toString("::");
        return toString("::").equals(compare);
    }

    /**
     * Get the inverse stack of this stack.
     * @return an inverted stack
     */
    public IDLScopeStack getInvertedStack() {
        Object o;
        IDLScopeStack invertedStack = new IDLScopeStack();
        IDLScopeStack copyStack = (IDLScopeStack) this.clone();
        try {  
            while(true) {
                /* copy data from orig's copyStack
                   to invertedStack in reverse order */
                o = copyStack.pop();
                invertedStack.push(o);
                
            }
        } catch(EmptyStackException e) {
            /* copyStack is empty now - leave it to the garbage collection */
        }
        return invertedStack;
    }

    /**
     * Get a combined stack from two stacks.
     * @param appendix stack to be appended to this stack
     * @return new combined stack 
     */
    public IDLScopeStack append(IDLScopeStack appendix) {
        Object o;
        IDLScopeStack resultStack = (IDLScopeStack)this.clone();
        IDLScopeStack helperStack = new IDLScopeStack();
        IDLScopeStack copyStack = (IDLScopeStack) appendix.clone();
        try {  
            while(true) {
                /* copy data from appendix' copyStack
                   to helperStack in reverse order */
                o = copyStack.pop();
                helperStack.push(o);
            }
        } catch(EmptyStackException e) {
            /* copyStack is empty now - leave it to the garbage collection */
        }
        try {
            while(true) {
                o = helperStack.pop();
                resultStack.push(o);
            }
        } catch(EmptyStackException e) {
            /* helperStack is also empty now */
        }    
        return resultStack;
    }

    /**
     * Get a String representation of this stack. The String
     * "::" will be used to separate the stack elements from
     * each other. 
     * @return this stack's String representation
     */
    public String toString() {
        return toString("::");
    }

    /**
     * Get a String representation of this stack. The given
     * separator will be used to separate the stack elements 
     * from each other.  
     * @param separator to separate stack elements
     * @return this stack's String representation
     */
    public String toString(String separator) {
        Object o;
        String string = null;
        IDLScopeStack copyStack = (IDLScopeStack) this.clone(); 

        if(isEmpty()) return "";
        try {
            while(true) {
                o = copyStack.pop();
                if(string == null) string = (String)o;
                else string = (String)o + separator + string;
            }
        } catch(EmptyStackException e) {
            /* copyStack is empty now - leave it to the garbage collection */
        }
        return string;
    }

}
