package org.aspectix.IDLparser.tree;

import antlr.*;
import antlr.collections.AST;
import java.util.*;
import org.aspectix.IDLparser.util.*;
import org.aspectix.IDLparser.pub.*;

public class IDLEntityNode extends CommonAST implements Cloneable, IDLNode {

    
    /*** Constructor methods *******************************************/
    
    /* Constructors called directly by the parser */

    int summe,summe2;
    public IDLEntityNode() {  }

    public IDLEntityNode(Token tok) {
        super(tok);
        line = tok.getLine();
    } 

    /* Method for the IDLEntityNodeFactory which is called by the parser ******/
    public void initialize(Token tok) {
        super.initialize(tok);
        line = tok.getLine();
    }

    /*** The parser sets the nodes "context", if it has been
         "defined", "used" or both ("defined" and "user" are well-defined 
         terms in the IDL language specification) */

    private boolean used = false;
    
    public void setUsed() { used = true; }
    protected boolean used() { return used; }

    protected boolean shadowCopy() {
        return !definitionScope.equals(nodeScope);
    }

    /*** Iterator over node's children ********************************/
  

    IDLIterator iterator() { 
        return new IDLIterator(this);
    }


    /*** Methods to check IDL syntax *****************************************/

    public void checkClashes() throws IDLException {

        if(this instanceof IDLNamed) LookupUtil.instance().checkNameClash(this);
        
        Iterator i = iterator();
        while(i.hasNext()) {
            ((IDLEntityNode)i.next()).checkClashes();
        }
    }

    public void resolveReferences() throws IDLException {
        if(IDLTracer.instance().tracing()) 
            IDLTracer.instance().println("RESOLVING "+getClass().getName()
                                         +" "+getScopedName()+" in line "+getLine());
        Iterator i = iterator();
        while(i.hasNext()) {
            ((IDLEntityNode)i.next()).resolveReferences();
        }
    }

    public void checkIdentifierUsage() throws IDLException {
        if(IDLTracer.instance().tracing()) IDLTracer.instance().println("USAGE "+getScopedName());
        if(this instanceof IDLNamed) LookupUtil.instance().assertIdentNotUsed(this);
        IDLIterator i = iterator();
        while(i.hasNext()) {
            ((IDLEntityNode)i.next()).checkIdentifierUsage();
        }
    }

    protected void throwNotResolvedException() throws IDLException {
        throw new IDLException("error: before calling node specific methods to retrieve information "+
                               "checkIDLSyntax() must be called on the AST");
    }


    /*** Line control (for printing IDL error messages ) **********************/

    protected int line = -1;

    /**
     * Return this node's line number (-> IDL file).
     * @return this node's line number
     */
    public int getLine() {
	return line;
    }

    /** 
     * Set this node's line number (line in the IDL file).
     * @param line 
     */
    public void setLine(int line) {
        this.line = line;
    }



    /*** Sequence: In a single tree traversal, each node gets a
         sequence number corresponding to the occurence of 
         definitions in the IDL file *********************************************/
    
    private int sequence;

    public void setSequence() {
        sequence = LookupUtil.currentSequence++;

        Iterator i = iterator();
        while(i.hasNext()) {
            ((IDLEntityNode)i.next()).setSequence();
        }
    }

    public int sequence() {
        return sequence;
    }

    /*** Visibility modification: Inherited nodes, which are redefined
         in an interface will be set invisible **************************************/ 

    protected boolean visible = true;

    protected void setInvisible() {
        visible = false;
        IDLIterator i = iterator();

        while(i.hasNext()) {
            ((IDLEntityNode)i.next()).setInvisible();
        }
    }
    
    public boolean visible() {
        return visible;
    }    
    







    /*** Scope control *********************************************/

    protected IDLScopeStack nodeScope = null;
    protected IDLScopeStack definitionScope = null;

    /** 
     * Set the scope for this node and recursively for all of this node's children. 
     * (Scope forming nodes will change the passed scope on its way throug the AST.)
     * @param scope to be set on this node
     */
    protected void setScope(IDLScopeStack scope) {
        nodeScope = (IDLScopeStack) scope.clone();
        if(definitionScope == null) definitionScope = nodeScope;

        IDLIterator iterator = iterator();
        if(this instanceof IDLScopeForming) scope.push(getText());
        while(iterator.hasNext()) {
            ((IDLEntityNode)iterator.next()).setScope(scope);
        }
        if(this instanceof IDLScopeForming) scope.pop();   
    }

    /**
     * Return this node's scope.
     * @return node's scope 
     */
    public IDLScopeStack getScope() {
        if(nodeScope == null) throw new RuntimeException("error: node scope not set");
        return (IDLScopeStack)nodeScope.clone();
    }

    /**
     * Return the scope in which this node was defined. If a node has been 
     * inherited, his definition scope will be the name of the base interface.
     * @return node's definition scope
     */
    public IDLScopeStack getDefinitionScope() {
        return (IDLScopeStack)definitionScope.clone();
    }

    /**
     * Return the fully qualified name of this node.
     * @return scoped name of this node
     */
    public IDLScopeStack getScopedName() {  
        return getScope().append(IDLScopeStack.createStack(this.getText(), "::"));
    } 

    /**
     * Return this stack node's name (without appending its scope)
     * @return IDLScopeStack representing the localName
     */
    public String getLocalName()  {
        return getText();
    }

    public String assignedName() {
        IDLNode father = getFather();
        if(father == null) return null;
        if(father instanceof IDLTypedef) return ((IDLTypedef)father).getName();
        else return null;
    }



    /*** Debug method ********************************************************************/

    /** 
     * Print text, classname (if none - type) and line number for each node of
     * this AST. Asterisks will mark missing information (e.g. line not set). 
     */
    public void printParseInfo() {
        IDLIterator iterator = iterator();
        
        String classname = org.aspectix.IDLparser.grammar.IDLClassStrings.classes[getType()];
      
        if((getText() == null) || (getText().equals(""))) System.err.print("*****");
        else System.err.print("-----");
        if(classname == null) System.err.print("*****");
        else System.err.print("-----");
        if(getLine() != -1) System.err.print("----- ");
        else System.err.print("***** ");
        if(classname == null) System.err.println(getText()+" "+getType()+" line = "+getLine());
        else System.err.println(getText()+" "+classname+" line = "+getLine());
        while(iterator.hasNext()) ((IDLEntityNode)iterator.next()).printParseInfo();
    } 
    








    /*** Methods for tree navigation ************************************************/ 

    private IDLEntityNode left = null;
    private IDLEntityNode up = null;

    public IDLEntityNode getFather() {
        IDLEntityNode father = up;
        if(up != null) return up;
        else if(left != null) { 
            return left.getFather();
        }
        else return null;
    }

    protected  IDLEntityNode getSecondChild() {
        if(down == null) return null;
        else return (IDLEntityNode) down.getNextSibling();
    }

    private BaseAST getRight() {
        return this.right;
    }
    
    private void setUp(AST f) {
        if(f == null) return;
        up = (IDLEntityNode)f;
    } 

    private void setLeft(AST f) {
        if(f == null) return;
        left = (IDLEntityNode)f;
    }
    
    public void setFirstChild(AST c) {
        super.setFirstChild(c);
        if ( c==null ) return;
        ((IDLEntityNode)c).setUp(this);
    }  
    
    public void addChild(AST node) {
        if ( node==null ) return;
        BaseAST t = this.down;
        if(t != null) {
            while ( ((IDLEntityNode)t).getRight()!=null ) t = ((IDLEntityNode)t).getRight();
            ((IDLEntityNode)node).setLeft(t);
        }
        else ((IDLEntityNode)node).setUp(this);
        super.addChild(node);
    }
    
    public void setNextSibling(AST n) {
        super.setNextSibling(n);
        if ( n==null ) return;
        ((IDLEntityNode)n).setLeft(this);
    }


    /*** Our own special clone() method ****************************/

    public Object clone() throws CloneNotSupportedException {
        IDLEntityNode copy  = (IDLEntityNode) super.clone();
        IDLIterator iterator = iterator();    

        if(iterator.hasNext()) {
            copy.down = (BaseAST)((IDLEntityNode)iterator.next()).clone();
            
            IDLEntityNode child = (IDLEntityNode)copy.down;
            while(iterator.hasNext()) {
                IDLEntityNode next = (IDLEntityNode)iterator.next();
                child.up = null;
                child.right = (BaseAST)next.clone();
                ((IDLEntityNode)child.right).left = child;
                child = (IDLEntityNode)child.right;
            }
            
            ((IDLEntityNode)copy.down).up = copy;
        }

        return copy;
    }
   


    /*** Helper methods for LookupUtil *****************************/

    /* Recursively add all children, which match the given name 
       (inverted, relative to this node) to the given list. The
       order in which nodes are added to the list corresponds to
       the order of definition in the idl file.
    */
    void addDefinedChildrenWithInvertedName(List list, IDLScopeStack invertedName) {
        //System.err.println("*** "+getScopedName()+" adding children "+invertedName.toString("::"));
        if((invertedName == null) && (invertedName.isEmpty())) return;
        
        String next = (String) invertedName.peek();
        
        if(!next.equals(getLocalName())) return;
        
        if(invertedName.isSimpleIdentifier()) {
            if((this.visible) && (this instanceof IDLNamed)) list.add(this);
            return;
        }
        else {
            invertedName.pop();
            IDLIterator i = iterator();
            while(i.hasNext()) {
                ((IDLEntityNode)i.next()).addDefinedChildrenWithInvertedName(list, invertedName);
            }
            invertedName.push(next);
        }
    }


    /* 
     * Get an iterator over the surrounding scopes of a node.
     * Inner scopes occur first in the iterator. 
     * @param node the node whose surrounding nodes are to be searched
     * @return iterator over found surrounding nodes
     */
    Iterator getSurroundingScopes() {
        IDLEntityNode father = getFather();
        if(father == null) return IDLIteratorCreator.createEmpty();
        Iterator i = father.getSurroundingScopes();
        if(father instanceof IDLScopeForming) return IDLIteratorCreator.create(father, i);
        else return i;
    }

    /**
     * Get all used nodes in this scope (used nodes in 
     * enclosed scopes are not returned).
     * @return Iterator over the used nodes
     */
    public Iterator getUsedDescendants() {
        ArrayList list = new ArrayList();
        Iterator i = iterator();
        while(i.hasNext()) list.add(((IDLEntityNode)i.next()).getUsedRecursive());
        return IDLIteratorCreator.create(list);
    }
    
    /* Returns an Iterator containing the used nodes in the 
     * tree formed by this root node, doesn't include the 
     * descendants of enclosed scopes.
     */
    private Iterator getUsedRecursive() {
        ArrayList list = new ArrayList();
        if(this instanceof IDLScopeForming) {
            if(used() && !shadowCopy()) return IDLIteratorCreator.create(this);
            else return IDLIteratorCreator.createEmpty();
        }
        else {
            Iterator i = iterator();
            IDLEntityNode next;
            while(i.hasNext()) {
                next = (IDLEntityNode)i.next();
                if(next instanceof IDLScopeForming) { 
                    list.add(IDLIteratorCreator.create(next));
                }
                else list.add(next.getUsedRecursive());
            }
            if(used() && !shadowCopy()) return IDLIteratorCreator.create(this, list);
            else return IDLIteratorCreator.create(list);
        }
    }   


    /* Return the first node among the descendants of this node,
       which matches a given scoped name. 
    */
    IDLEntityNode getDescendantByAbsoluteName(IDLScopeStack scopedName) {
        if(getScopedName().equals(scopedName)) return this;
        Iterator i = iterator();
        IDLEntityNode next;
        while(i.hasNext()) {
            next = ((IDLEntityNode)i.next()).getDescendantByAbsoluteName(scopedName);
            if(next != null) return next;
        }
        return null;
    }

    
}
