package org.aspectix.IDLparser.tree;

import antlr.*;
import java.util.*;
import org.aspectix.IDLparser.grammar.*;
import org.aspectix.IDLparser.pub.*;
import org.aspectix.IDLparser.util.*;

/* Interface structure:

                          ____________ optional InheritNode
                         |
                         |____________ Declaration (export)
   InterfNode____________|
   (with text id         |____________ optional further Declaration (export)
   and flags inherits    |
   and forward decl.)    |____________ ...
                         |
                        ...
           
*/
public class InterfNode extends IDLEntityNode implements IDLInterf {
    
    private boolean inherit = false;
    private boolean forwardDecl = false;
    private boolean localif = false;
    private boolean abstractif = false;

    public InterfNode() {  }

    public InterfNode(Token tok) { 
        super(tok);  
        line = tok.getLine();
    }

    private boolean inherited = false;

    private void inherit() throws IDLException {
        if(forwardDecl) return;
        if(inherited) return;

        if(inherit) {
            IDLIterator i = iterator();
            /* get the inheritance node */
            IDLEntityNode node = (IDLEntityNode)i.next();        
            if(!(node instanceof InheritNode))
                throw new IDLException("error(line "+LocationInfo.getLocationInfo(getLine())+") : interface "+getScopedName()+
                                       " expected inheritance specification - found "+
                                       node.getClass().getName()+" "+node.getScopedName());

            /* for all scoped names in the inheritance specification... */
            IDLIterator inheritIterator = node.iterator();
            Iterator nodeIterator;
            IDLEntityNode base;
            IDLEntityNode nameNode;
            ArrayList baseInterfaces = new ArrayList();
            while(inheritIterator.hasNext()) {
                nameNode =  (IDLEntityNode)inheritIterator.next();

                /* ...get the according base interface */
                base = ((ScopedNameNode)nameNode).getReference();

                /* base not found? */
                if((base == null) || (!(base instanceof InterfNode)))
                    throw new IDLException("error(line "+LocationInfo.getLocationInfo(nameNode.getLine())+"): "+nameNode.getScopedName()+
                                           " is not a valid interface to inherit from.");

                /* base just a forward declaration? */
                if(((InterfNode)base).forwardDecl())
                    throw new IDLException("error(line "+LocationInfo.getLocationInfo(getLine())+"): "+base.getScopedName()+
                                           " is just a forward declaration (line "+base.getLine()+
                                           "), not a valid base interface");

                /* attempt to inherit from ourselves? */
                if(base.getScopedName().equals(getScopedName()))
                    throw new IDLException("error(line "+LocationInfo.getLocationInfo(getLine())+"): "+getScopedName()+
                                           " may not inherit from itself");

                /* already inherited from this base? */
                if(baseInterfaces.contains(base)) 
                    throw new IDLException("error(line "+LocationInfo.getLocationInfo(getLine())+") : "+getScopedName()+
                                           " can't inherit from "+nameNode.getScopedName()+" twice");
                baseInterfaces.add(base);
        
                /* get an iterator over all nodes that we inherit from the base interface */
                nodeIterator = ((InterfNode)base).getVisibleMembers();

                /* inherit from all nodes in the iterator  */
                while(nodeIterator.hasNext())   
                    inheritMember((IDLEntityNode)nodeIterator.next());
            }  
            inherited = true;
        }
          
    }

    public void resolveReferences() throws IDLException {
        if(IDLTracer.instance().tracing()) IDLTracer.instance().println("RESOLVING interface "+getScopedName()+" in line "+getLine());
     
        Iterator i = iterator();
        if(inherit) ((IDLEntityNode)i.next()).resolveReferences();
        inherit();
        while(i.hasNext()) {
            ((IDLEntityNode)i.next()).resolveReferences();
        }  

        /* if this is a forward declaration, check if definition follows */
        if(forwardDecl) {
            InterfNode definition = LookupUtil.instance().lookupInterfaceDefinition(getScopedName());
            if(definition == null) 
                throw new IDLException("error(line "+LocationInfo.getLocationInfo(getLine())+"): unsatisfied forward declaration "+
                                       getScopedName());
            if(definition.sequence() < sequence()) 
                throw new IDLException("error(line "+LocationInfo.getLocationInfo(getLine())+"): forward declaration "+getScopedName()+
                                       " ocurred after interface definition - line "+definition.getLine());
        }
    }

    IDLEntityNode resolveBaseReference(ScopedNameNode node) throws IDLException {
        IDLEntityNode next, result = null;
        IDLScopeStack gname = node.getScopedName();
        String lname = node.getLocalName();
        //System.err.println("resolveBaseReference called on interface "+getScopedName()+
        //                 " with name "+node.getScopedName());
        inherit();
        IDLIterator i = iterator();
        if(!inherit) return null;
        i.next();
        while(i.hasNext()) {
            next = (IDLEntityNode)i.next();
            if(!next.visible()) {
                if(next.getLocalName().equals(lname)) {
                    if(result != null) 
                        throw new IDLException("error(line "+LocationInfo.getLocationInfo(+node.getLine())+"): ambiguous reference to "+
                                               "inherited member "+gname+" - found in lines "+result.getLine()+
                                               " and "+next.getLine());
                    result = next;
                }
            }
        }
        return result;
    }
    

    /*** Type Interface ***/

    public IDLType getEffectiveType() { 
        return this;
    }

    public Object matchValue(IDLNode constExp) throws IDLException {
        throw new IDLException("error(line "+LocationInfo.getLocationInfo(constExp.getLine())+
                               "): "+((IDLEntityNode)constExp).getText()+
                               " cannot be assigned to type interface "+getScopedName());
    }

    public int getTypeId() {
        return IDLTypeUtil.INTERFACE;
    }


    /*** InterfNode specific methods ***/

    public void setInherit() {
        inherit = true;
    }

    public boolean inherits() {
        return inherit;
    }

    public void setForwardDecl() {
        forwardDecl = true;
    }

    public boolean forwardDecl() {
        return forwardDecl;
    }

    public void setLocalIf() {
	localif = true;
    }

    public boolean localIf() {
	return localif;
    }

    public void setAbstractIf() {
	abstractif = true;
    }

    public boolean abstractIf() {
	return abstractif;
    }

    public String getName() {
        return getText();
    }

    /**
     * Get all direct base interfaces of this interface
     * @return Iterator over the interfaces
     */
    public Iterator getDirectBaseInterfaces() throws IDLException {
        ArrayList list = new ArrayList();
        if(!inherit) return list.iterator();

        /* our first child is the inherit node */
        IDLIterator iterator = ((IDLEntityNode)getFirstChild()).iterator(); 
        ScopedNameNode next;
        
        while(iterator.hasNext()) {    
            next = (ScopedNameNode)iterator.next();
            list.add(next.getReference());
        }
        return list.iterator();
    }

    /** 
     * Get all nodes which are accessible in this interface without scope
     * resolution, i.e. all members which are defined directly in this
     * interface plus all members which have been inherited 
     * unambiguously but not overridden. 
     * @return Iterator over all nodes of interest
     */   
    public Iterator getVisibleMembers() {
        ArrayList list = new ArrayList();
       
        IDLIterator iterator = iterator();
        IDLEntityNode next;

        /* skip inheritance specification node; it's no longer interesting here */
        if(inherits()) iterator.next();
        
        while(iterator.hasNext()) {
            next = (IDLEntityNode)iterator.next();
            if(next.visible()) list.add(next);
        }
        return list.iterator();
    }

    public Iterator getVisibleInheritedMembers() {
        ArrayList list = new ArrayList();
       
        IDLIterator iterator = iterator();
        IDLEntityNode next;

        /* skip inheritance specification node; it's no longer interesting here */
        if(inherits()) iterator.next();
        
        while(iterator.hasNext()) {
            next = (IDLEntityNode)iterator.next();
            if((!next.getDefinitionScope().equals(next.getScope())) && next.visible()) list.add(next);
        }
        return list.iterator();
    }

    public Iterator getLocallyDefinedMembers() {
        ArrayList list = new ArrayList();
       
        IDLIterator iterator = iterator();
        IDLEntityNode next;

        /* skip inheritance specification node; it's no longer interesting here */
        if(inherits()) iterator.next();
        
        while(iterator.hasNext()) {
            next = (IDLEntityNode)iterator.next();
            if(next.getDefinitionScope().equals(next.getScope())) list.add(next);
        }
        return list.iterator();
    }

    
    /** 
     * Add a member (IDLEntityNode) to the list of inherited nodes.
     * @param node to be inherited
     * @exception IDLExcption if there are any collisions with redefined or
     * previously inherited members.
     */
    private void inheritMember(IDLEntityNode node) throws IDLException {
        IDLEntityNode current;
        String currentName, nodeName;
        boolean invisible = false;
        IDLIterator iterator = iterator(); 
        if(inherits()) iterator.next();

        /* check for collisions with self-defined or already inherited  members */
        while(iterator.hasNext()) {
            current = (IDLEntityNode)iterator.next();
            currentName = current.getLocalName();
            nodeName = node.getLocalName();


            /* is there already a member with the same name? */
            if(currentName.toLowerCase().equals(nodeName.toLowerCase())) {  
        
                /* the names differ in case? */
                if(!currentName.equals(nodeName)) 
                    throw new IDLException("error(line "+LocationInfo.getLocationInfo(current.getLine())+"): members "+
                                           IDLClassStrings.classes[node.getType()]+" "+
                                           nodeName+" "+LocationInfo.getLocationInfo(node.getLine())+" and "+
                                           IDLClassStrings.classes[current.getType()]+
                                           " "+currentName+" in interface "+getText()+
                                           " - names differ in case");
                
                /* found members of different type? */
                if(node.getType() != current.getType()) 
                    throw new IDLException("error(line "+LocationInfo.getLocationInfo(node.getLine())+"): interface "+
                                           getScopedName()+" member "+nodeName+
                                           " collides with member from line "+current.getLine()+
                                           " (have different types)");
                
                
                /* found nodes the same - inherited via different inheritance path? */
                if(node.getDefinitionScope().equals(current.getDefinitionScope())) return;
                
                
                /* operation or attribute? - redefining not allowed */
                if((node.getType() == IDLTokenTypes.OP_DECL) || (node.getType() == IDLTokenTypes.ATTRIBUTE)) {
                    throw new IDLException("error(line "+LocationInfo.getLocationInfo(getLine())+"): colliding members "+nodeName
                                           +" "+LocationInfo.getLocationInfo(current.getLine())+" and "+LocationInfo.getLocationInfo(node.getLine()));    
                }
		

                /* ok, no collision - node must be set invisible (it's shadowed or ambiguous) */
                invisible = true;

                /* If old node has been inherited, set it invisible,    */
                /* since the new node has the same name.                */
                /* (constants and typdefs may be ambiguously inherited) */
                if(!current.getDefinitionScope().equals(current.getScope())) current.setInvisible();
            }
        }
        
        /* no collisions - add shadow copy of node to our children */
        IDLEntityNode memberClone = null;
        try {
            memberClone = (IDLEntityNode)node.clone();
        } catch (CloneNotSupportedException e) { /* can't happen */ }
        if(invisible) memberClone.setInvisible();
        memberClone.setNextSibling(null);
        memberClone.setScope(getScopedName());
        //memberClone.setLineRecursive(getLine());
        addChild(memberClone);
        // System.err.println("interface "+getScopedName()+" inherited "+memberClone.getScopedName()+
        //                 " newS is "+getScopedName()+", defS is "+memberClone.getDefinitionScope()+
        //                 " new line is "+LocationInfo.getLocationInfo(memberClone.getLine())+" visible is "+memberClone.visible());
    }
    
            
}
