package org.aspectix.IDLparser.tree;

import java.io.*;
import java.util.*;
import java.math.BigDecimal;
import org.aspectix.IDLparser.util.*;
import org.aspectix.IDLparser.pub.*;

public class LookupUtil {

    public static int currentSequence = 0;

    /* TreeSet holding CORBA keywords for case insensitve comparism with identifiers */
    private static final TreeSet keywords = new TreeSet();
    static {
        keywords.addAll(Arrays.asList(new Object[] {
            "any", "attribute", "boolean", "case", "char", "const",
            "context", "default", "double", "enum", "exception", "false",
            "fixed", "float", "in", "inout", "interface", "long", "module",
            "object", "octet", "oneway", "out", "raises", "readonly",
            "sequence", "short", "string", "struct", "switch", "true",
            "typedef", "unsigned", "union", "void", "wchar", "wstring"
        }));
    }

    /*** Singleton functionality ***/

    private static LookupUtil instance = null;

    private LookupUtil() {}

    public static LookupUtil instance() {
        if(instance == null) instance = new LookupUtil();
        return instance;
    }


    /*** Tree search functionality ***/

    private IDLEntityNode ast = null;

    /**
     * Set the AST to be searched by this TreeUtil
     * @param ast the abstract syntax tree
     */
    public void setAST(IDLEntityNode ast) {
        this.ast = ast;
    } 

    /**
     * Check, if a given BigDecimal represents a positive integer value.
     * @param value to be checked
     * @param line where the number has been parsed
     * @exception if value is not a positive interger
     */
    public static void assertPositiveIntValue(BigDecimal value, int line) throws IDLException {
        if(value.signum() != 1) 
            throw new IDLException("error(line "+line+
                                   "): value must be positive (found "+value.toString()+")");
        if(value.scale() != 0)
            throw new IDLException("error(line "+line+
                                   "): value must be integer (found "+value.toString()+")");
    }

    /**
     * Search the tree for an interface definition with a given absolute name.
     * Forward declarations are ignored in the search.
     * @param absoluteName of the interface
     * @return the definition of the interface with that name
     */
    public InterfNode lookupInterfaceDefinition(IDLScopeStack absoluteName) {
        Iterator i = getDefinedNodesByAbsoluteName(absoluteName);
        IDLEntityNode next, prev = null;
        while(i.hasNext()) {
            next = (IDLEntityNode)i.next();
            if(next instanceof InterfNode) {
                if(!((InterfNode)next).forwardDecl()) return (InterfNode)next;
            }
        }
        return null;
    }

    /**
     * Assert, that the name of a given node has not been used in unqualified
     * form before (sequence number) the give node.
     * @param nameNode to be compared with preceding uses
     * @exception if name has been used before
     */
    void assertIdentNotUsed(IDLEntityNode nameNode) throws IDLException {
        //System.err.println("asserting ident "+nameNode.getScopedName()+" not used in line "+nameNode.getLine());
        Iterator scopes = nameNode.getSurroundingScopes();
        if(!scopes.hasNext()) return;
        IDLEntityNode scope = (IDLEntityNode)scopes.next();
        //System.err.println("searching scope "+scope.getScopedName());
        Iterator used = scope.getUsedDescendants();
        IDLEntityNode next;
        while(used.hasNext()) {
            next = (IDLEntityNode)used.next();
            //System.err.println("next is "+next.getScopedName()+" in line "+next.getLine());
            if( (next instanceof ScopedNameNode) && 
                (next.sequence() < nameNode.sequence()) &&
                (((ScopedNameNode)next).usesUnqualified(nameNode.getLocalName())) )
		// TODO: This fails on compilint PortableServer at POA definition . why?
		System.err.println("error (line "+LocationInfo.getLocationInfo(nameNode.getLine())+"): identifier "+
                                       nameNode.getLocalName()+" may not be redefined after having "+
                                       "been used in scope "+scope.getScopedName()+
                                       " at line "+LocationInfo.getLocationInfo(next.getLine()));
/*
                throw new IDLException("error (line "+LocationInfo.getLocationInfo(nameNode.getLine())+"): identifier "+
                                       nameNode.getLocalName()+" may not be redefined after having "+
                                       "been used in scope "+scope.getScopedName()+
                                       " at line "+LocationInfo.getLocationInfo(next.getLine()));
*/
        }
    }

    /**
     * Check, if a given node clashes with a previously defined node
     * in the ast. (Modules don't clash, interfaces don't clash with 
     * forward declarations.)
     * @param nameNode to be checked
     * @exception if name collision occured
     */
    public void checkNameClash(IDLEntityNode nameNode) throws IDLException {
        /* check, if identifier collides with IDL keyword (case insensitive) */
	/* dl9rdz: this check is disabled. 
	 *         not complaining about IDL keyword collisions does not
	 *         matter, and it is allowed to use idl keywords, if escaped
	 *         via a '_' (which currently is removed by the parser)
        if(keywords.contains((Object) nameNode.getText().toLowerCase())) {
            throw new IDLException("error(line "+LocationInfo.getLocationInfo(nameNode.getLine())+"): "+nameNode.getText()+
                                   " - CORBA keywords may not be used as identifiers, even if"+
                                   " spelled differently in case");
        }  
	*/
        Iterator existing = getDefinedNodesByAbsoluteName(nameNode.getScopedName());
        while(existing.hasNext()) {
            IDLEntityNode exists = (IDLEntityNode) existing.next();
            if(exists.sequence() >= nameNode.sequence()) continue;
		// Hack for OOVS: method overloading
	    if((nameNode instanceof OpDeclNode) && (exists instanceof OpDeclNode)) continue;
	    if((nameNode instanceof ParamDeclNode) && (exists instanceof ParamDeclNode)) continue;
	   System.err.println(nameNode.getClass().toString());

            if((nameNode instanceof ModuleNode) && (exists instanceof ModuleNode)) continue;
            if((nameNode instanceof InterfNode) && (exists instanceof InterfNode) &&
               (((InterfNode)exists).forwardDecl())) continue;
            if((nameNode instanceof ValueTypeNode) && (exists instanceof ValueTypeNode) &&
               (((ValueTypeNode)exists).forwardDecl())) continue;

            throw new IDLException(LocationInfo.getLocationInfo(nameNode.getLine())+": "+nameNode.getScopedName()+
                                   " previously defined in line "+(LocationInfo.getLocationInfo(exists.getLine())));
        }
    }

    /**
     * Try to finde the name of the given nodes among the surrounding nodes
     * of the node. This is useful for allowing recursiv type definitions 
     * by sequences.
     * @param node whose name shall be found
     * @return the recursively referenced node
     */
    public IDLEntityNode resolveRecursiveReference(IDLEntityNode node) {
        Iterator i = node.getSurroundingScopes();
        IDLEntityNode scope;
        while(i.hasNext()) {
            scope = (IDLEntityNode)i.next();
            if(!(scope instanceof IDLNamed)) continue;
            if(node.getScopedName().isShortReferenceTo(scope.getScopedName())) return scope;                           
        }
        return null;
    }

    private boolean trace = false; ///
    //private boolean trace = true; ///

    /** 
     * Find the node in the tree, which is referenced by a given scoped name
     * @param nameNode name to be found
     * @exception if reference is ambiguous
     */
    IDLEntityNode resolveReference(ScopedNameNode nameNode) throws IDLException {
        IDLEntityNode next, result = null;
        Iterator i, j;
        IDLScopeStack gname = nameNode.getScopedName();

        if(trace) System.err.println("- resolveReference called with "+gname+ ///
                                     " on node "+nameNode.getScopedName()); ///
        
        /* name beginning with "::" */
        if(gname.toString("::").startsWith("::")) {

            /* try to find name at root scope, will throw exception, if ambiguous */
            result = handleAmbiguity(getDefinedNodesByAbsoluteName(nameNode.getScopedName()), nameNode);
            if(result != null) {                                                 ///
                if(trace) System.err.println("- found "+result.getScopedName()); ///
            }                                                                    ///
           
            /* name of the form ::a */
            if(gname.isSimpleSuperCall()) {        
                /* try to resolve as reference to base interface */
                InterfNode interf = getSurroundingInterface(nameNode);
                if(interf != null) {
                    if(trace) System.err.println("- interface context: "+interf.getScopedName()); ///
                    /* retrieve referenced node from interface, will throw exception, if ambiguous */
                    next = interf.resolveBaseReference(nameNode);
                    if(next != null) {
                        if(result != null) 
                            throw new IDLException("error (line "+LocationInfo.getLocationInfo(nameNode.getLine())+" ): ambiguous reference to "+
                                                   gname+" - found in lines "+LocationInfo.getLocationInfo(result.getLine())+" and "
						   +LocationInfo.getLocationInfo(next.getLine()));
                        if(trace) System.err.println("- found "+next.getScopedName()); ///
                        result = next;
                    } 
                }
            }
        }
        else {
            /* search for name outward in scopes */
            i = nameNode.getSurroundingScopes();
            while(i.hasNext()) {
                next = (IDLEntityNode)i.next();
                if(trace) System.err.println("- searching scope "+next.getScopedName());
                /* try to find name in next scope, will throw exception, if ambiguous */
                result = resolveChildByRelativeName(next, nameNode);
                if(result != null) break; /* don't search farther out in scopes if name found */
            }  
            if(result == null) {
                InterfNode interf = getSurroundingInterface(nameNode);
                if(interf != null) {
                    if(trace) System.err.println("- interface context: "+interf.getScopedName()); ///
                    /* retrieve referenced node from interface, will throw exception, if ambiguous */
                    next = interf.resolveBaseReference(nameNode);
                    if(next != null) {
                        if(result != null) 
                            throw new IDLException("error (line "+LocationInfo.getLocationInfo(nameNode.getLine())+" ): ambiguous reference to "+
                                                   gname+" - found in lines "+LocationInfo.getLocationInfo(result.getLine())+" and "
						   +LocationInfo.getLocationInfo(next.getLine()));
                        if(trace) System.err.println("- found "+next.getScopedName()); ///
                        result = next;
                    } 
                } 
            }
        }     
        if(trace) {                                                                        ///
            if(result != null) System.err.println("- returning "+result.getScopedName());  ///
            else System.err.println("- returning null");                                   ///
        }                                                                                  ///
        return result;
    }

    /* Get a descendant of a node by name relative to the node, i.e. node
     * A::B::C::d is called node C::d relative to node C.
     */
    private IDLEntityNode resolveChildByRelativeName(IDLEntityNode scope, IDLEntityNode nameNode) throws IDLException {
        ArrayList list = new ArrayList();
        Iterator i = scope.iterator();
        while(i.hasNext()) {
            ((IDLEntityNode)i.next()).addDefinedChildrenWithInvertedName(list, nameNode.getScopedName().getInvertedStack());
        }        
        i = list.iterator();
        return handleAmbiguity(i, nameNode);
    }
     

    /*
     * Get the next outer scope of a node, which is an interface.
     * @param node the node whose surrounding scopes shall be searched
     * @return surrounding interface, null if there is none
     */
    private InterfNode getSurroundingInterface(IDLEntityNode node) {
        Iterator i = node.getSurroundingScopes();
        IDLEntityNode next;
        while(i.hasNext()) {
            next = (IDLEntityNode)i.next();
            if(next instanceof InterfNode) return (InterfNode)next;
        }
        return null;
    }
    
    /* 
     * Get all descendants of the ast which match a given name.
     * @param name absolute name of the searched node
     * @return iterator over matching descendants
     */
    private Iterator getDefinedNodesByAbsoluteName(IDLScopeStack name) {
        ArrayList list = new ArrayList();
        ast.addDefinedChildrenWithInvertedName(list, name.getInvertedStack());
        return list.iterator();
    } 


    private IDLEntityNode handleAmbiguity(Iterator nodes, IDLEntityNode nameNode) throws IDLException {
        IDLEntityNode next, result = null;
        while(nodes.hasNext()) {    
            next = (IDLEntityNode)nodes.next();
            if(next.sequence() < nameNode.sequence()) {
                if(trace) System.err.println("- next is "+next.getScopedName());
                if(result != null) {
                    result = chooseAmbiguous(result, next, nameNode);
                    if(trace) System.err.println("- found "+result.getScopedName()); ///
                }
                else result = next;
            }
        }
        return result;
    }


    private IDLEntityNode chooseAmbiguous(IDLEntityNode first, IDLEntityNode second, IDLEntityNode nameNode) throws IDLException {
        int sequence = nameNode.sequence();
        int fseq = first.sequence();
        int sseq = second.sequence();
        if((fseq < sequence) && (sseq < sequence)) {
            if((first instanceof InterfNode) && (second instanceof InterfNode)) {
                if((fseq < sseq) && (((InterfNode)first).forwardDecl())) return second;
                else if((sseq < fseq) && (((InterfNode)second).forwardDecl())) return first;
            }
            throw new IDLException("error (line "+LocationInfo.getLocationInfo(nameNode.getLine())+" ): ambiguous reference to "+
                                   nameNode.getScopedName()+" - found in lines "+LocationInfo.getLocationInfo(first.getLine())+
                                   " and "+LocationInfo.getLocationInfo(second.getLine()));

        }
        else if((fseq >= sequence) && (sseq < sequence)) return second;
        else if((sseq >= sequence) && (fseq < sequence)) return first;
        else return null;
    }

}
