package org.aspectix.IDLparser.util;

import java.util.*;
import org.aspectix.IDLparser.pub.*;

public class IDLIteratorCreator {

    /**
     * Create a combined iterator from a single IDLEntityNode and an Iterator.
     * @param first node to be returned first by the iterator
     * @param second iterator holding all following objects
     */
    public static Iterator create(IDLNode first, Iterator second) {
        if(second == null) return create(first);
        final IDLNode myfirst = first;
        final Iterator mysecond = second;
        return new Iterator() {
                private boolean firstReturned = false;
                
                public boolean hasNext() {
                    if((myfirst != null) && (!firstReturned)) return true;
                    else return mysecond.hasNext();
                }

                public Object next() {
                    if((myfirst != null) && (!firstReturned)) {
                        firstReturned = true;
                        return myfirst;
                    }
                    else return mysecond.next();
                }

                public void remove() {
                    throw new UnsupportedOperationException();
                }
            };
    } 


    /**
     * Create a combined iterator, which returns all elements from
     * a List of Iterators. 
     * @param list holding the iterators
     */
    public static Iterator create(List list) {
        if(list == null) return createEmpty();
        final List mylist = list; 
        return new Iterator() {
                private int current = 0;

                public boolean hasNext() {
                   if((mylist != null) && (current != mylist.size())) {
                        if(((Iterator)mylist.get(current)).hasNext()) return true;
                        else {
                            current++;
                            if(current == mylist.size()) return false;
                            return ((Iterator)mylist.get(current)).hasNext(); 
                        }
                    }
                    else return false;
                }
                
                public Object next() {
                    /* check, if there are still Iterators available in List mylist */
                    if(current != mylist.size()) {
                        /* if current Iterator has more elements -> return next */
                        if(((Iterator)mylist.get(current)).hasNext()) return ((Iterator)mylist.get(current)).next();
                        /* else try to get a further Iterator from List mylist */
                        else {
                            current++;
                            /* if no more Iterators -> throw Exception */
                            if(current == mylist.size()) throw new NoSuchElementException();
                            /* if no elements in next Iterator -> throw Exception */
                            if (!((Iterator)mylist.get(current)).hasNext()) throw new NoSuchElementException();
                            /* else -> return next element */
                            return ((Iterator)mylist.get(current)).next();
                        }
                    }
                    else throw new NoSuchElementException();
                }
                
                public void remove() {
                    throw new UnsupportedOperationException();
                }
                
            };
    }

    /**
     * Create a combined iterator from a single IDLEntityNode and a List
     * containing more Iterators.
     * @param first node to be returned first by the iterator
     * @param more list holding all following iterators
     */
    public static Iterator create(IDLNode first, List more) {
        if(first == null) return create(more);
        if(more == null) return create(first);
        final IDLNode myfirst = first;
        final List mymore = more; 
        return new Iterator() {
                private boolean firstReturned = false;
                private int current = 0;

                public boolean hasNext() {
                    if(!firstReturned) return true;
                    else if(current != mymore.size()) {
                        if(((Iterator)mymore.get(current)).hasNext()) return true;
                        else {
                            current++;
                            if(current == mymore.size()) return false;
                            return ((Iterator)mymore.get(current)).hasNext(); 
                        }
                    }
                    else return false;
                }
                
                public Object next() {
                    /* return first element, if not yet done */
                    if(!firstReturned) {
                        firstReturned = true;
                        return myfirst;
                    }
                    /* check, if there are still Iterators available in List mymore */
                    else if(current != mymore.size()) {

                        /* if current Iterator has more elements -> return next */
                        if(((Iterator)mymore.get(current)).hasNext()) return ((Iterator)mymore.get(current)).next();
                        /* else try to get a further Iterator from List mymore */
                        else {
                            current++;
                            /* if no more Iterators -> throw Exception */
                            if(current == mymore.size()) throw new NoSuchElementException();
                            /* if no elements in next Iterator -> throw Exception */
                            if (!((Iterator)mymore.get(current)).hasNext()) throw new NoSuchElementException();
                            /* else -> return next element */
                            return ((Iterator)mymore.get(current)).next();
                        }
                    }
                    else throw new NoSuchElementException();
                }
                
                public void remove() {
                    throw new UnsupportedOperationException();
                }
                
            };
    }


    /**
     * Create an iterator containing only one single IDLEntityNode.
     * @param single node to be returned by the iterator
     */
    public static Iterator create(IDLNode single) {
        if(single == null) return createEmpty();
        final IDLNode mysingle = single;
        return new Iterator() {
                private boolean hasNext = true;
                public boolean hasNext() { return hasNext; } 
                public Object next() {
                    if(hasNext) {
                        hasNext = false;
                        return mysingle;
                    }
                    else throw new NoSuchElementException();
                }
                public void remove() {throw new UnsupportedOperationException();}
            };
    }

    /** 
     * Create an empty iterator.
     */
    public static Iterator createEmpty() {
        return new Iterator() {
                public boolean hasNext() { return false; }
                public Object next() { throw new NoSuchElementException(); }
                public void remove() { throw new UnsupportedOperationException(); }
            };
    }

}
