/***********************************************************HeaderBegin*******
 *                                                                         
 * File:        bitio.c
 *
 * Author:      T.W. M.F., K.S.
 * Created:     23-July-96 
 *                                                                         
 * Description: source code of routines for buffering and bit-io
 *
 * Notes: 
 *
 * Modified:  
 *
 ***********************************************************HeaderEnd*********/



#include "global.h"


/***********************************************************CommentBegin******
 *****************************************************************************
 *
 * -- AllocBitstr -- Alloc bitstream with buffer but without filepointer
 *
 * Author:           T.W.
 *
 * Created:          23-July-96		
 *	
 * Purpose:          Creates a new 'Bitstr' with a buffer size of 'n' bytes
 *                   and sets the filepointer to NULL. 
 *
 * Arguments in:     int      n     size of buffer in byte (min. sizeof(int)).
 *
 * Arguments in/out: -
 *
 * Arguments out:    -
 *
 * Return values:    Bitstr *       pointer to allocated bitstream structure.
 *                                  NULL in case of error.
 *
 * Example:          b = AllocBitstr(1024);
 *
 * Side effects:     -
 *
 * Description:      The memory of a 'Bitstr' structure is allocated.
 *                   No file is opened!
 *                   If a file shall be associated with the bitstream, use
 *                   FopenBitstr! 
 *                   This function is included in FopenBitstr.
 *
 * See also:         FopenBitstr
 *	
 * Modified:         -
 *	
 *****************************************************************************/
Bitstr *AllocBitstr(int n)
/***********************************************************CommentEnd********/
{
  Bitstr *b = (Bitstr *) malloc(sizeof(Bitstr));

  n = MAX(n, sizeof(int));

  b->size = n * 8;
  b->ind = 0;
  b->actualSize = 0;
  b->b = (UChar *) malloc(n * sizeof(UChar));
  b->fp = NULL;

  return(b);
}


/***********************************************************CommentBegin******
 *****************************************************************************
 *
 * -- FreeBitstr -- Free a bitstream structure (file is  n o t  closed!)
 *
 * Author:           T.W.
 *
 * Created:          23-July-96		
 *	
 * Purpose:          Frees a 'Bitstr' and sets the pointer to NULL.
 *
 * Arguments in:     -
 *
 * Arguments in/out: Bitstr **b_p    pointer to bitstream structure
 *
 * Arguments out:    -
 *
 * Return values:    -
 *
 * Example:          FreeBitstr(&b);
 *
 * Side effects:     -
 *
 * Description:      The memory of a 'Bitstr' structure is freed.
 *                   No file closing is done!
 *                   If a file is associated with the bitstream, use
 *                   FcloseBitstr! 
 *                   This function is included in FcloseBitstr.
 *
 * See also:         FcloseBitstr
 *	
 * Modified:         -
 *	
 *****************************************************************************/
void FreeBitstr(Bitstr **b_p)
/***********************************************************CommentEnd********/
{
  free((*b_p)->b);
  free(*b_p);
  *b_p = NULL;
}


/***********************************************************CommentBegin******
 *****************************************************************************
 *
 * -- CheckInBufferN -- Read in bytes from the file
 *
 * Author:           T.W.
 *
 * Created:          23-July-96		
 *	
 * Purpose:          Reads in bytes from the file addressed by 'b->fp' into the
 *                   bit-stream 'b'.
 *
 * Arguments in:     -
 *
 * Arguments in/out: Bitstr *b    bitstream structure
 *
 * Arguments out:    -
 *
 * Return values:    FALSE  (0)  on 'EOF' or 'b->fp == NULL'
 *                   TRUE   (1)  on success
 *
 * Example:          CheckInBuffer(b);
 *
 * Side effects:     -
 *
 * Description:      The buffer is filled with bytes from the file.
 *                   This function is called automatically if you use
 *                   ShowBit or ShowCodeword.
 *
 * See also:         CheckOutBuffer, GetBit, GetCodeword, GetCodewordIndex
 *	
 * Modified:         -
 *	
 *****************************************************************************/
int CheckInBufferN(Bitstr *b)
/***********************************************************CommentEnd********/
{
  int i;
  int currentByte = b->ind/8;
  int bufferSizeByte = b->actualSize/8;

  if (b->fp == NULL)
    return(FALSE);
  else if (feof(b->fp))
    return(FALSE);
 
  for(i=currentByte;i<bufferSizeByte;++i) {
    b->b[i-currentByte] = b->b[i];
  }

  b->actualSize = 8 * 
    (fread(&b->b[bufferSizeByte-currentByte],1,currentByte,b->fp) + 
     bufferSizeByte-currentByte);
  b->ind = b->ind%8;

  return(TRUE);
}


/***********************************************************CommentBegin******
 *****************************************************************************
 *
 * -- CheckInBuffer -- Read in bytes from the file
 *
 * Author:           T.W.
 *
 * Created:          23-July-96		
 *	
 * Purpose:          Reads in bytes from the file addressed by 'b->fp' into the
 *                   bit-stream 'b'.
 *
 * Arguments in:     -
 *
 * Arguments in/out: Bitstr *b    bitstream structure
 *
 * Arguments out:    -
 *
 * Return values:    FALSE  (0)  on 'EOF' or 'b->fp == NULL'
 *                   TRUE   (1)  on success
 *
 * Example:          CheckInBuffer(b);
 *
 * Side effects:     -
 *
 * Description:      The buffer is filled with bytes from the file.
 *                   The filling starts at the beginning of the buffer, i.e.
 *                   all content is overwritten.
 *                   The buffer indicater 'b->ind' is reset.
 *                   This function is called automatically if you use
 *                   GetBit, GetCodeword or GetCodewordIndex.
 *
 * See also:         CheckOutBuffer, GetBit, GetCodeword, GetCodewordIndex
 *	
 * Modified:         -
 *	
 *****************************************************************************/
int CheckInBuffer(Bitstr *b)
/***********************************************************CommentEnd********/
{
  if (b->fp == NULL)
    return(FALSE);
  else if (feof(b->fp))
    return(FALSE);
 
  /* actualSize in bit only a multiple of 8! might be incorrect at EOF! */
  /* needs improvement! */
  b->actualSize = 8 * fread(&b->b[0],1,b->size/8,b->fp);
  b->ind = 0;

  return(TRUE);
}


/***********************************************************CommentBegin******
 *****************************************************************************
 *
 * -- ExtendBuffer -- Extend the buffer memory
 *
 * Author:           K.S.
 *
 * Created:          13-Aug-96
 *	
 * Purpose:          Extends the buffer memory of a bitstream.
 *                   The buffer content is copied to the new buffer.
 *
 * Arguments in:     int    n     extension in bytes
 *
 * Arguments in/out: Bitstr *b    bitstream structure
 *
 * Arguments out:    -
 *
 * Return values:    FALSE  (0)  on error
 *                   TRUE   (1)  on success
 *
 * Example:          ExtendBuffer(1024, b);
 *
 * Side effects:     -
 *
 * Description:      'b->b' is reallocated
 *
 * See also:         CheckOutBuffer
 *	
 * Modified:         -
 *	
 *****************************************************************************/
int ExtendBuffer(int n, Bitstr *b)
/***********************************************************CommentEnd********/
{
  UChar *bp;

  if ((bp = (UChar *)malloc((b->size + n) * sizeof(UChar))) == NULL) {
    fprintf(stderr, "ExtendBuffer: Can't allocate buffer with size %d\n",
	    (b->size + n) * sizeof(UChar));
    return(FALSE);
  }
  memcpy(bp, b->b, b->size * sizeof(UChar));
  free(b->b);
  b->b = bp;
  b->size += n;

  return(TRUE);
}


/***********************************************************CommentBegin******
 *****************************************************************************
 *
 * -- ByteAlign -- Set the stream indicator to next byte
 *
 * Author:           T.W.
 *
 * Created:          23-July-96
 *	
 * Purpose:          Sets the stream indicator to the first bit of the
 *                   next byte. The skipped bits of the bitstream are
 *                   undefined.
 *
 * Arguments in:     Bitstr *b    bitstream structure
 *
 * Arguments in/out: -
 *
 * Arguments out:    -
 *
 * Return values:    -
 *
 * Example:          ByteAlign(b);
 *
 * Side effects:     -
 *
 * Description:      The bitstream indicator 'b->ind' is set to the beginning
 *                   of the next byte.
 *
 * See also:         ByteAlign0, BitsToByteAlignment
 *
 * Modified:         -
 *
 *****************************************************************************/
void ByteAlign(Bitstr *b)
/***********************************************************CommentEnd********/
{
  b->ind = ((b->ind+7)/8)*8;
}


/***********************************************************CommentBegin******
 *****************************************************************************
 *
 * -- BitsToByteAlignment -- Return remaining bits in current byte
 *
 * Author:           Niko Faerber
 *
 * Created:          08-March-97
 *	
 * Purpose:          Calculate the number of bits to complete the current byte
 *
 * Arguments in:     Bitstr *bs : bitstream
 *
 * Arguments in/out: -
 *
 * Arguments out:    -
 *
 * Return values:    int    n   : number of bits to complete the current byte
 *
 * Example:          n = BitsToByteAlignment(bs);
 *
 * Side effects:     -
 *
 * Description:      -
 *
 * See also:         -
 *	
 * Modified:         -
 *	
 *****************************************************************************/
int BitsToByteAlignment(Bitstr *bs)
/***********************************************************CommentEnd********/
{
  int n;
  
  n = 8-(bs->ind)%8;
  if(n==8) n=0;

  return n;
}


/***********************************************************CommentBegin******
 *****************************************************************************
 *
 * -- AbsBitPosIn -- Return absolute bit position of an input bitstream
 *
 * Author:           Niko Faerber
 *
 * Created:          08-March-97
 *	
 * Purpose:          Return the absolute bit position of an input bitstream
 *                   that reads data from a file
 *
 * Arguments in:     bs
 *
 * Arguments in/out: -
 *
 * Arguments out:    -
 *
 * Return values:    absolute bit position
 *
 * Example:          pos = AbsBitPosIn(bs);
 *
 * Side effects:     -
 *
 * Description:      If no bitstream file is open, '0' is returned.
 *
 * See also:         -
 *	
 * Modified:         -
 *	
 *****************************************************************************/
int AbsBitPosIn(Bitstr *bs)
/***********************************************************CommentEnd********/
{
  if (bs->fp != NULL)
    return(ftell(bs->fp)*8 - bs->actualSize + bs->ind);
  else
    return 0;
}

/***********************************************************CommentBegin******
 *****************************************************************************
 *
 * -- GetBit -- Read bit from bitstream
 *
 * Author:           T.W.
 *
 * Created:          23-July-96		
 *	
 * Purpose:          Reads a bit '*rbit' of the bit-stream 'b' at the current
 *                   pointer stage 'b->ind' of the bit-stream. 'GetBit' returns 1
 *                   on success, else 0.
 *
 * Arguments in:     Bitstr *b         bitstream structure
 *
 * Arguments in/out: int    *rbit_p    pointer to int that is filled with
 *                                     the bit value.
 *
 * Arguments out:    -
 *
 * Return values:    FALSE  (0)  on error
 *                   TRUE   (1)  on success
 *
 * Example:          GetBit(b, &rbit);
 *
 * Side effects:     -
 *
 * Description:      -
 *
 * See also:         PutBit
 *	
 * Modified:         -
 *	
 *****************************************************************************/
int GetBit(Bitstr *b, int *rbit_p)
/***********************************************************CommentEnd********/
{
  int quot;
  int rem;
  
  if( b->ind == b->actualSize ) {
    if(!CheckInBuffer(b))
      return(FALSE);
  }

  quot = b->ind / 8;
  rem = b->ind % 8;
  *rbit_p = READBIT(b->b[quot],7-rem);
  
  /* *rbit = (b->b[b->buffPtr/8]>>(7-b->buffPtr%8))&1; */

  /* should check here for EOF on bitstream */
  ++(b->ind);

  return(TRUE);  
}


/***********************************************************CommentBegin******
 *****************************************************************************
 *
 * -- ShowBit -- Show bit from bitstream
 *
 * Author:           T.W.
 *
 * Created:          19-Nov-96		
 *	
 * Purpose:          Shows a bit '*rbit' of the bit-stream 'b' at the current
 *                   pointer stage 'b->ind' + offset. 'ShowBit' returns 1
 *                   on success, else 0.
 *
 * Arguments in:     Bitstr *b  : bitstream structure
 *                   int offset : offset to 'b->ind' 
 *
 * Arguments in/out: int *rbit_p: pointer to int that is filled with
 *                                the bit value.
 *
 * Arguments out:    -
 *
 * Return values:    FALSE  (0)  on error
 *                   TRUE   (1)  on success
 *
 * Example:          ShowBit(b, offset, &rbit);
 *
 * Side effects:     -
 *
 * Description:      -
 *
 * See also:         ShowBits, GetBit, PutBit
 *	
 * Modified:         -
 *	
 *****************************************************************************/
int ShowBit(Bitstr *b, int offset, int *rbit_p)
/***********************************************************CommentEnd********/
{
  int quot;
  int rem;
  
  /* Buffer not loaded yet */
  if(b->actualSize==0) {
    if(!CheckInBuffer(b))
      return(FALSE);
  } else {    
    if( b->ind+offset >= b->actualSize ) {
      if(!CheckInBufferN(b))
	return(FALSE);
    }
  }

  quot = (b->ind+offset) / 8;
  rem = (b->ind+offset) % 8;
  *rbit_p = READBIT(b->b[quot],7-rem);
  
  return(TRUE);  
}


/***********************************************************CommentBegin******
 *****************************************************************************
 *
 * -- ShowBits -- Show bits from bitstream
 *
 * Author:           K.S.
 *
 * Created:          7-Aug-97
 *	
 * Purpose:          Shows 'n' bits '*rbit' of the bitstream 'b' at the current
 *                   pointer stage 'b->ind' + offset. 'ShowBits' returns 1
 *                   on success, else 0.
 *
 * Arguments in:     Bitstr *b  : bitstream structure
 *                   int offset : offset to 'b->ind'
 *                   int n      : number of bits (max. sizeof(int)).
 *
 * Arguments in/out: int *rbit_p: pointer to int that is filled with
 *                                the bit value.
 *
 * Arguments out:    -
 *
 * Return values:    FALSE  (0)  on error
 *                   TRUE   (1)  on success
 *
 * Example:          ShowBits(b, offset, n, &rbit);
 *
 * Side effects:     -
 *
 * Description:      -
 *
 * See also:         ShowBit, GetBit, PutBit
 *	
 * Modified:         -
 *	
 *****************************************************************************/
int ShowBits(Bitstr *b, int offset, int n, int *rbit_p)
/***********************************************************CommentEnd********/
{
  int ind;


  ind = b->ind + offset;

  /* Buffer not loaded yet */
  if(b->actualSize == 0) {
    if(!CheckInBuffer(b))
      return(FALSE);
  } else {
    if(ind + n >= b->actualSize ) {
      if(!CheckInBufferN(b))
	return(FALSE);
    }
  }

  *rbit_p = 0;
  for (ind = b->ind + offset, n--; n >= 0; ind++, n--)
    *rbit_p |= ((READBIT(b->b[ind / 8], (7-(ind%8)))) << n);
  
  return(TRUE);  
}


/***********************************************************CommentBegin******
 *****************************************************************************
 *
 * -- ConsumeBits -- Consume bits from bitstream
 *
 * Author:           K.S.
 *
 * Created:          7-Aug-97
 *	
 * Purpose:          Consumes 'n' bits of the bitstream 'b' at the current
 *                   pointer stage 'b->ind'.
 *
 * Arguments in:     Bitstr *b  : bitstream structure
 *                   int n      : number of bits.
 *
 * Arguments in/out: -
 *
 * Arguments out:    -
 *
 * Return values:    FALSE  (0)  on error
 *                   TRUE   (1)  on success
 *
 * Example:          ConsumeBits(b, n);
 *
 * Side effects:     -
 *
 * Description:      -
 *
 * See also:         ConsumeBitsX, ShowBits, ShowBit, GetBit, PutBit
 *	
 * Modified:         -
 *	
 *****************************************************************************/
int ConsumeBits(Bitstr *b, int n)
/***********************************************************CommentEnd********/
{
  /* Buffer not loaded yet */
  if(b->actualSize == 0) {
    if(!CheckInBuffer(b))
      return(FALSE);
  } else {
    if(b->ind + n >= b->actualSize) {
      if(!CheckInBufferN(b))
	return(FALSE);
    }
  }

  b->ind += n;
  
  return(TRUE);  
}


/***********************************************************CommentBegin******
 *****************************************************************************
 *
 * -- GetCodeword -- Read codeword
 *
 * Author:           T.W.
 *
 * Created:          23-July-96		
 *	
 * Purpose:          Reads a codeword from the bit-stream 'b' of 'N' bits
 *                   at the current pointer stage. Returns the codeword
 *                   using 'cw'. 'cw' has to be an allocated array of at
 *                   least N bytes. 'GetCodeword' returns 1 on success,
 *                   else 0.
 *
 * Arguments in:     Bitstr *b         bitstream structure
 *                   int    n          number of bits
 *
 * Arguments in/out: char   *cw        read codeword as a string of '0' and '1'
 *
 * Arguments out:    -
 *
 * Return values:    FALSE  (0)  on error
 *                   TRUE   (1)  on success
 *
 * Example:          GetCodeword(b, 8, cw);
 *
 * Side effects:     -
 *
 * Description:      For cw at least (n+1 * sizeof(char)) memory must
 *                   have been allocated.
 *
 * See also:         PutCodeword, GetCodewordIndex, GetBit
 *	
 * Modified:         -
 *	
 *****************************************************************************/
int GetCodeword(Bitstr *b, int n, char *cw)
/***********************************************************CommentEnd********/
{
  int i;
  int bit;
 
  cw[0] = '\0';
 
  for(i=0;i<n;++i) {
    if(!GetBit(b, &bit)) return(FALSE);
    if(bit){
      strcat(cw,"1");
    } else {
      strcat(cw,"0");
    }
  }  
  return(TRUE);
}


/***********************************************************CommentBegin******
 *****************************************************************************
 *
 * -- ShowCodeword -- Show codeword
 *
 * Author:           T.W.
 *
 * Created:          23-July-96		
 *	
 * Purpose:          Shows a codeword from the bit-stream 'b' of 'N' bits
 *                   at the current pointer stage. Returns the codeword
 *                   using 'cw'. 'cw' has to be an allocated array of at
 *                   least N bytes. 'ShowCodeword' returns 1 on success,
 *                   else 0.
 *
 * Arguments in:     Bitstr *b         bitstream structure
 *                   int    n          number of bits
 *
 * Arguments in/out: char   *cw        read codeword as a string of '0' and '1'
 *
 * Arguments out:    -
 *
 * Return values:    FALSE  (0)  on error
 *                   TRUE   (1)  on success
 *
 * Example:          ShowCodeword(b, 8, cw);
 *
 * Side effects:     -
 *
 * Description:      For cw at least (n+1 * sizeof(char)) memory must
 *                   have been allocated. The buffersize must be at least n+1.
 *
 * See also:         GetCodeword, PutCodeword, GetCodewordIndex, GetBit
 *	
 * Modified:         -
 *	
 *****************************************************************************/
int ShowCodeword(Bitstr *b, int n, char *cw)
/***********************************************************CommentEnd********/
{
  int i;
  int bit;

  if ((CEILING(n/8)+1)>(b->size/8)) return(FALSE);

  cw[0] = '\0';
 
  for(i=0;i<n;++i) {
    if(!ShowBit(b, i, &bit)) return(FALSE);
    if(bit){
      strcat(cw,"1");
    } else {
      strcat(cw,"0");
    }
  }  
  return(TRUE);
}


/***********************************************************CommentBegin******
 *****************************************************************************
 *
 * -- GetCodewordIndex -- Reads codeword and search in set of codewords
 *
 * Author:           T.W.
 *
 * Created:          23-July-96		
 *	
 * Purpose:          Reads a codeword of the bit-stream 'b' at the pointer
 *                   stage of the bit-stream. Searches the whole set 'ct[]'
 *                   of 'n' strings and returns the index '*index_p'.
 *                   (Assumption: ct[] is a prefix-code).
 *                   'GetCodewordIndex' returns 1 on success, else 0.
 *
 * Arguments in:     Bitstr *b         bitstream structure
 *                   int    n          number of bits
 *                   char   *ct[]      code; codeword strings
 *
 * Arguments in/out: int    *index_p   integer in which the index is returned
 *
 * Arguments out:    -
 *
 * Return values:    FALSE  (0)  on error
 *                   TRUE   (1)  on success
 *
 * Example:          GetCodewordIndex(b, ct, n, *index_p);
 *
 * Side effects:     -
 *
 * Description:      -
 *
 * See also:         PutCodeword, GetCodeword
 *	
 * Modified:         -
 *	
 *****************************************************************************/
int GetCodewordIndex(Bitstr *b, char *ct[], int n, int *index_p)
/***********************************************************CommentEnd********/
{
  int found = 0;
  int bit;
  int i=0;
  char *cw = malloc(100*sizeof(char));
    
  cw[0] = '\0';
 
  while(!found) {
    GetBit(b, &bit);
    if(bit){
      strcat(cw,"1");
    } else {
      strcat(cw,"0");
    }
       
    i = 0;
    while(i<n && !found) {
      if(strcmp(cw,ct[i])==0) {
        found = 1;
        *index_p = i;
        free(cw);
        return(TRUE);
      } else {
        ++i;
      }
    }
  }
  free(cw);
  return(FALSE);
}


/***********************************************************CommentBegin******
 *****************************************************************************
 *
 * -- GetNumber -- Read some bits in a number
 *
 * Author:           K.S.
 *
 * Created:          30-Jan-97		
 *	
 * Purpose:          Reads 'n' bits from the bit-stream 'b'
 *                   at the current pointer stage. Returns the number
 *                   in the integer '*num_p'. 'num_p' has to be allocated
 *                   before. 'GetNumber' returns 1 on success,
 *                   else 0.
 *
 * Arguments in:     Bitstr *b         bitstream structure
 *                   int    n          number of bits
 *
 * Arguments in/out: int    *num_p     read number as a bit pattern
 *
 * Arguments out:    -
 *
 * Return values:    FALSE  (0)  on error
 *                   TRUE   (1)  on success
 *
 * Example:          GetNumber(b, 8, *num_p);
 *
 * Side effects:     -
 *
 * Description:      For '*num_p' memory must have been allocated.
 *
 * See also:         PutNumber, GetCodeword, GetCodewordIndex, GetBit
 *	
 * Modified:         -
 *	
 *****************************************************************************/
int GetNumber(Bitstr *b, int n, int *num_p)
/***********************************************************CommentEnd********/
{
  int i;
  int bit;
 
  *num_p = 0;
 
  for(i = n - 1; i >= 0; i--) {
    if(!GetBit(b, &bit))
      return(FALSE);
    *num_p |= (bit << i);
  }  
  return(TRUE);
}


