/***********************************************************HeaderBegin*******
 *                                                                         
 * File:        bitioX.c
 *
 * Author:      K.S.
 * Created:     23-Aug-97
 *
 * Description: source code of routines for fast buffering and bit-io
 *              without checking the buffersize.
 *
 * Notes: 
 *
 * Modified:  
 *
 ***********************************************************HeaderEnd*********/


#include "global.h"


/***********************************************************CommentBegin******
 *****************************************************************************
 *
 * -- GetBitX -- Read bit from bitstream without checking the buffersize
 *
 * Author:           K.S.
 *
 * Created:          23-Aug-97
 *	
 * Purpose:          Reads a bit '*rbit' of the bit-stream 'b' at the current
 *                   pointer stage 'b->ind' of the bit-stream without
 *                   regarding the buffersize and the file.
 *                   'GetBit' returns 1 on success, else 0.
 *
 * Arguments in:     Bitstr *b         bitstream structure
 *
 * Arguments in/out: int    *rbit_p    pointer to int that is filled with
 *                                     the bit value.
 *
 * Arguments out:    -
 *
 * Return values:    FALSE  (0)  on error
 *                   TRUE   (1)  on success
 *
 * Example:          GetBitX(b, &rbit);
 *
 * Side effects:     -
 *
 * Description:      -
 *
 * See also:         GetBit, PutBitX
 *	
 * Modified:         -
 *	
 *****************************************************************************/
int GetBitX(Bitstr *b, int *rbit_p)
/***********************************************************CommentEnd********/
{
  int quot = b->ind >> 3;
  int rem = b->ind & 0x7;
  

  ++(b->ind);

  *rbit_p = READBIT(b->b[quot],7-rem);
  
  return(TRUE);  
}


/***********************************************************CommentBegin******
 *****************************************************************************
 *
 * -- ShowBitX -- Show bit from bitstream without checking the buffersize
 *
 * Author:           K.S.
 *
 * Created:          27-Aug-97
 *	
 * Purpose:          Shows a bit '*rbit' of the bit-stream 'b' at the current
 *                   pointer stage 'b->ind' + offset. 'ShowBit' returns 1
 *                   on success, else 0.
 *                   Buffersize and file are not regarded.
 *
 * Arguments in:     Bitstr *b  : bitstream structure
 *                   int offset : offset to 'b->ind' 
 *
 * Arguments in/out: int *rbit_p: pointer to int that is filled with
 *                                the bit value.
 *
 * Arguments out:    -
 *
 * Return values:    FALSE  (0)  on error
 *                   TRUE   (1)  on success
 *
 * Example:          ShowBitX(b, offset, &rbit);
 *
 * Side effects:     -
 *
 * Description:      -
 *
 * See also:         ShowBit, ShowBitsX, GetBitX, PutBitX
 *	
 * Modified:         -
 *
 *****************************************************************************/
int ShowBitX(Bitstr *b, int offset, int *rbit_p)
/***********************************************************CommentEnd********/
{
  int quot = (b->ind+offset) >> 3;
  int rem = (b->ind+offset) & 0x7;
  
  *rbit_p = READBIT(b->b[quot],7-rem);
  
  return(TRUE);  
}


/***********************************************************CommentBegin******
 *****************************************************************************
 *
 * -- ShowBitsX -- Show bits from bitstream without checking the buffersize
 *
 * Author:           K.S.
 *
 * Created:          27-Aug-97
 *	
 * Purpose:          Shows 'n' bits '*rbit' of the bitstream 'b' at the current
 *                   pointer stage 'b->ind' + offset. 'ShowBits' returns 1
 *                   on success, else 0.
 *                   Buffersize and file are not regarded.
 *
 * Arguments in:     Bitstr *b  : bitstream structure
 *                   int offset : offset to 'b->ind'
 *                   int n      : number of bits
 *                                (max. sizeof(int) || sizeof(long) - 7).
 *
 * Arguments in/out: int *rbit_p: pointer to int that is filled with
 *                                the bit value.
 *
 * Arguments out:    -
 *
 * Return values:    FALSE  (0)  on error
 *                   TRUE   (1)  on success
 *
 * Example:          ShowBitsX(b, offset, n, &rbit);
 *
 * Side effects:     -
 *
 * Description:      -
 *
 * See also:         ShowBit, ShowBitsX, ShowBitX, GetBitW, PutBitX
 *	
 * Modified:         -
 *	
 *****************************************************************************/
int ShowBitsX(Bitstr *b, int offset, int n, int *rbit_p)
/***********************************************************CommentEnd********/
{
  int  ind = b->ind + offset;
  int  i;
  int  bitPEnd = 7 - ((ind + n - 1) & 0x7);
  int  byteInd = ind >> 3;
  int  mask = ((0x1 << n) - 1);
  long l = 0;


  for (i = (ind & 0x7) + n; i > 0; i -= 8) {
    l <<= 8;
    l |= b->b[byteInd++];
  }
  *rbit_p = (int)((l >> bitPEnd) & mask);

  return(TRUE);  
}


/***********************************************************CommentBegin******
 *****************************************************************************
 *
 * -- ConsumeBitsX -- Consume bits from bitstream without checking buffersize
 *
 * Author:           K.S.
 *
 * Created:          27-Aug-97
 *	
 * Purpose:          Consumes 'n' bits of the bitstream 'b' at the current
 *                   pointer stage 'b->ind'. 'ShowBits' returns 1
 *                   on success, else 0.
 *                   Buffersize and file are not regarded.
 *
 * Arguments in:     Bitstr *b  : bitstream structure
 *                   int n      : number of bits.
 *
 * Arguments in/out: -
 *
 * Arguments out:    -
 *
 * Return values:    FALSE  (0)  on error
 *                   TRUE   (1)  on success
 *
 * Example:          ConsumeBitsX(b, n);
 *
 * Side effects:     -
 *
 * Description:      -
 *
 * See also:         ConsumeBits, ShowBitsX, ShowBitX, GetBitX, PutBitX
 *	
 * Modified:         -
 *	
 *****************************************************************************/
int ConsumeBitsX(Bitstr *b, int n)
/***********************************************************CommentEnd********/
{
  b->ind += n;
  
  return(TRUE);  
}



/***********************************************************CommentBegin******
 *****************************************************************************
 *
 * -- GetCodewordX -- Read codeword without regarding buffersize
 *
 * Author:           K.S.
 *
 * Created:          27-Aug-97
 *	
 * Purpose:          Reads a codeword from the bit-stream 'b' of 'N' bits
 *                   at the current pointer stage. Returns the codeword
 *                   using 'cw'. 'cw' has to be an allocated array of at
 *                   least N bytes. 'GetCodeword' returns 1 on success,
 *                   else 0.
 *                   Buffersize and file are not regarded (-> 'GetBitX()').
 *
 * Arguments in:     Bitstr *b         bitstream structure
 *                   int    n          number of bits
 *
 * Arguments in/out: char   *cw        read codeword as a string of '0' and '1'
 *
 * Arguments out:    -
 *
 * Return values:    FALSE  (0)  on error
 *                   TRUE   (1)  on success
 *
 * Example:          GetCodewordX(b, 8, cw);
 *
 * Side effects:     -
 *
 * Description:      For cw at least (n+1 * sizeof(char)) memory must
 *                   have been allocated.
 *
 * See also:         GetCodeword, PutCodewordX, GetCodewordIndexX, GetBitX
 *	
 * Modified:         -
 *
 *****************************************************************************/
int GetCodewordX(Bitstr *b, int n, char *cw)
/***********************************************************CommentEnd********/
{
  int i;
  int bit;
 
  cw[0] = '\0';
 
  for(i=0;i<n;++i) {
    if(!GetBitX(b, &bit)) return(FALSE);
    if(bit){
      strcat(cw,"1");
    } else {
      strcat(cw,"0");
    }
  }  
  return(TRUE);
}


/***********************************************************CommentBegin******
 *****************************************************************************
 *
 * -- ShowCodewordX -- Show codeword without regarding buffersize
 *
 * Author:           K.S.
 *
 * Created:          27-Aug-97
 *	
 * Purpose:          Shows a codeword from the bit-stream 'b' of 'N' bits
 *                   at the current pointer stage. Returns the codeword
 *                   using 'cw'. 'cw' has to be an allocated array of at
 *                   least N bytes. 'ShowCodeword' returns 1 on success,
 *                   else 0.
 *                   Buffersize and file are not regarded (-> 'ShowBitX()').
 *
 * Arguments in:     Bitstr *b         bitstream structure
 *                   int    n          number of bits
 *
 * Arguments in/out: char   *cw        read codeword as a string of '0' and '1'
 *
 * Arguments out:    -
 *
 * Return values:    FALSE  (0)  on error
 *                   TRUE   (1)  on success
 *
 * Example:          ShowCodewordX(b, 8, cw);
 *
 * Side effects:     -
 *
 * Description:      For cw at least (n+1 * sizeof(char)) memory must
 *                   have been allocated. The buffersize must be at least n+1.
 *
 * See also:         ShowCodeword, GetCodewordX, GetCodewordIndexX, GetBitX
 *	
 * Modified:         -
 *	
 *****************************************************************************/
int ShowCodewordX(Bitstr *b, int n, char *cw)
/***********************************************************CommentEnd********/
{
  int i;
  int bit;

  if ((CEILING(n/8)+1)>(b->size/8)) return(FALSE);

  cw[0] = '\0';
 
  for(i=0;i<n;++i) {
    if(!ShowBitX(b, i, &bit)) return(FALSE);
    if(bit){
      strcat(cw,"1");
    } else {
      strcat(cw,"0");
    }
  }  
  return(TRUE);
}


/***********************************************************CommentBegin******
 *****************************************************************************
 *
 * -- GetCodewordIndexX -- Reads codeword and search in set of codewords (no f)
 *
 * Author:           K.S.
 *
 * Created:          27-Aug-97
 *	
 * Purpose:          Reads a codeword of the bit-stream 'b' at the pointer
 *                   stage of the bit-stream. Searches the whole set 'ct[]'
 *                   of 'n' strings and returns the index '*index_p'.
 *                   (Assumption: ct[] is a prefix-code).
 *                   'GetCodewordIndexS' returns 1 on success, else 0.
 *                   Buffersize and file are not regarded (-> 'GetBitX()').
 *
 * Arguments in:     Bitstr *b         bitstream structure
 *                   int    n          number of bits
 *                   char   *ct[]      code; codeword strings
 *
 * Arguments in/out: int    *index_p   integer in which the index is returned
 *
 * Arguments out:    -
 *
 * Return values:    FALSE  (0)  on error
 *                   TRUE   (1)  on success
 *
 * Example:          GetCodewordIndexX(b, ct, n, *index_p);
 *
 * Side effects:     -
 *
 * Description:      -
 *
 * See also:         GetCodewordIndex, PutCodewordX, GetCodewordX
 *	
 * Modified:         -
 *
 *****************************************************************************/
int GetCodewordIndexX(Bitstr *b, char *ct[], int n, int *index_p)
/***********************************************************CommentEnd********/
{
  int found = 0;
  int bit;
  int i=0;
  char *cw = malloc(100*sizeof(char));
    
  cw[0] = '\0';
 
  while(!found) {
    GetBitX(b, &bit);
    if(bit){
      strcat(cw,"1");
    } else {
      strcat(cw,"0");
    }
       
    i = 0;
    while(i<n && !found) {
      if(strcmp(cw,ct[i])==0) {
        found = 1;
        *index_p = i;
        free(cw);
        return(TRUE);
      } else {
        ++i;
      }
    }
  }
  free(cw);
  return(FALSE);
}



/***********************************************************CommentBegin******
 *****************************************************************************
 *
 * -- GetNumberX -- Read some bits in a number without regarding buffersize
 *
 * Author:           K.S.
 *
 * Created:          27-Aug-97
 *	
 * Purpose:          Reads 'n' bits from the bit-stream 'b'
 *                   at the current pointer stage. Returns the number
 *                   in the integer '*num_p'. 'num_p' has to be allocated
 *                   before. 'GetNumberX' returns 1 on success,
 *                   else 0.
 *
 * Arguments in:     Bitstr *b         bitstream structure
 *                   int    n          number of bits
 *
 * Arguments in/out: int    *num_p     read number as a bit pattern
 *
 * Arguments out:    -
 *
 * Return values:    FALSE  (0)  on error
 *                   TRUE   (1)  on success
 *
 * Example:          GetNumberX(b, 8, *num_p);
 *
 * Side effects:     -
 *
 * Description:      For '*num_p' memory must have been allocated.
 *
 * See also:         GetNumber, PutNumberX, GetCodewordX, GetCodewordIndexX,
 *                   GetBitX
 *
 * Modified:         -
 *
 *****************************************************************************/
int GetNumberX(Bitstr *b, int n, int *num_p)
/***********************************************************CommentEnd********/
{
  int  i;
  int  bitPEnd = 7 - ((b->ind + n - 1) & 0x7);
  int  byteInd = b->ind >> 3;
  int  mask = ((0x1 << n) - 1);
  long l = 0;


  for (i = (b->ind & 0x7) + n; i > 0; i -= 8) {
    l <<= 8;
    l |= b->b[byteInd++];
  }
  *num_p = (int)((l >> bitPEnd) & mask);

  b->ind += n;

  return(TRUE);  
}
