#include <stdio.h>

#include "global.h"



/***********************************************************CommentBegin******
 *****************************************************************************
 *
 * -- H263gotoNextSync -- Read from bitstream until next H.263 sync
 *
 * Author:           K.S.
 *
 * Created:          16-Sep-97
 *
 * Purpose:          Read from bitsream until the (byte aligned) sync word
 *                   "0000 0000 0000 0000 1" is found.
 *
 * Arguments in:     -
 *
 * Arguments in/out: Bitstr *bs     bitstream
 *
 * Arguments out:    -
 *
 * Return values:    number of consumed bits or '-1' if no sync found.
 *
 * Example:          H263gotoNextSync(bitstr);
 *
 * Side effects:     The size of the bitstream is not checked.
 *
 * Description:      'bs->ind' is set to the beginning of the next
 *                   byte aligned H.263 sync word.
 *
 * See also:         -
 *
 * Modified:         -
 *
 *****************************************************************************/
int H263gotoNextSync(Bitstr *bs)
/***********************************************************CommentEnd********/
{
  ByteAlign(bs);

  {
    Int  stop = bs->actualSize - 16;
    Int  n = bs->ind;
    Int  indStart = n;
    Byte *p = bs->b + (n >> 3);

    for (; n < stop; n += 8, p++)
      if ((p[0] == 0x00) && (p[1] == 0x00) && ((p[2] & 0x80) == 0x80)) {
	bs->ind = n;
	return(n - indStart);
      }
    return(-1);
  }
}

/*
 * similar to the above, but does not touch ind
 */
int H263LookupAnotherSync(Bitstr *bs)
{
  {
    Int  stop = bs->actualSize - 16;
    Int  n = (bs->ind & ~7) + 8;

    Int  indStart = n;
    Byte *p = bs->b + (n >> 3);

    for (; n < stop; n += 8, p++)
      if ((p[0] == 0x00) && (p[1] == 0x00) && ((p[2] & 0x80) == 0x80)) {
	return(n - indStart);
      }
    return(-1);
  }
}


/***********************************************************CommentBegin******
 *****************************************************************************
 *
 * -- GetPTYPEX -- Read PTYPE from a bitstream using bitioX functions
 *
 * Author:           K.S.
 *
 * Created:          16-Sep-97
 *
 * Purpose:          Read PTYPE from a bitsream and write into picture struct.
 *                   Also sets single flags in picture struct that are
 *                   extracted from PTYPE. The bitstream size is not checked
 *                   (bitioX functions).
 *
 * Arguments in:     -
 *
 * Arguments in/out: Bitstr *bs      bitstream
 *
 * Arguments out:    pic->ptype
 *                   pic->splitScreenIndicator
 *                   pic->documentCameraIndicator
 *                   pic->freezePictureRelease
 *                   pic->sourceFormat
 *                   pic->pictureCodingType
 *                   pic->umv
 *                   pic->sac
 *                   pic->ap
 *                   pic->pb
 *               
 * Return values:    TRUE  (1) on success
 *                   FALSE (0) on failure
 *
 * Example:          GetPTYPEX(bs, pic);
 *
 * Side effects:     -
 *
 * Description:      -
 *
 * See also:         -
 *
 * Modified:         -
 *
 *****************************************************************************/
Int GetPTYPEX(Bitstr *bs, H263picStruct *pic)
/***********************************************************CommentEnd********/
{
  Int bit;


  ShowBitsX(bs, 0, 13, &bit);
  pic->ptype = bit;
  GetBitX(bs,&bit);
  if(bit==0) {
    return 0;
  }
  GetBitX(bs,&bit);
  if(bit==1) {
    return 0;
  }
  GetBitX(bs,&pic->splitScreenIndicator);
  GetBitX(bs,&pic->documentCameraIndicator);
  GetBitX(bs,&pic->freezePictureRelease);
  GetNumberX(bs,3,&pic->sourceFormat);
  if(pic->sourceFormat == 0 || pic->sourceFormat == 6 ||
     pic->sourceFormat == 7) {
    return 0;
  }
  GetBitX(bs,&pic->pictureCodingType);
  GetBitX(bs,&pic->umv);
  GetBitX(bs,&pic->sac);
  GetBitX(bs,&pic->ap);
  GetBitX(bs,&pic->pb);
  if(pic->pictureCodingType==0 && pic->pb==1) {
    return 0;
  }

  return 1;
}




/***********************************************************CommentBegin******
 *****************************************************************************
 *
 * -- H263getPictureHeaderWoSync -- Read the picture header without sync
 *
 * Author:           K.S.
 *
 * Created:          16-Sep-97
 *
 * Purpose:          Read the picture header from the bitstream and set the
 *                   appropriate variables in the picture struct.
 *
 * Arguments in:     -
 *
 * Arguments in/out: Bitstr  *bs    bitstream
 *                   pic->ptype
 *
 * Arguments out:    pic->tr
 *                   pic->splitScreenIndicator
 *                   pic->documentCameraIndicator
 *                   pic->freezePictureRelease
 *                   pic->sourceFormat
 *                   pic->pictureCodingType
 *                   pic->umv
 *                   pic->sac
 *                   pic->ap
 *                   pic->pb
 *                   pic->pquant
 *                   pic->currentQuant
 *                   pic->cpm
 *                   pic->psbi                     : only read if CPM=1
 *                   pic->trb                      : only read if PB=1
 *                   pic->dbquant                  : only read if PB=1
 *                   pic->numberOfGOBs             : number of GOB in one
 *                                                   picture
 *                   pic->linesOfMBs               : MB lines in one GOB
 *                   pic->numberOfMBs              : number of MBs in one GOB
 *                   pic->widthFrame               : width in pixels
 *                   pic->heightFrame              : heigth in pixels
 *                   pic->numberOfBLKs             : max. number of Blocks in
 *                                                   MB (6 or 12)
 *                   pic->numberOfMVs              : max. number of MVs in MB
 *                                                   (0,1,4 or 5)
 *
 * Return values:    TRUE   (1)  on success
 *                   FALSE  (0)  on failure
 *
 * Example:          H263getPictureHeaderWoSync(bs, pic)
 *
 * Side effects:     -
 *
 * Description:      Reads the picture header and sets the variables listed in
 *                   "Arguments out" above. If an error occures 0 is returned.
 *
 *                   The value of unused variables (psbi, trb, dbquant)
 *                   is not defined. 
 *
 * See also:         -
 *
 * Modified:         -
 *
 *****************************************************************************/
Int H263getPictureHeaderWoSync(Bitstr *bs, H263picStruct *pic)
/***********************************************************CommentEnd********/
{
  Int pei;

  /* TR */
  if(!GetNumberX(bs, 8, &pic->tr)) return 0;
  /* PTYPE */
  if(!GetPTYPEX(bs,pic)) return 0;
  /* PQUANT */
  if(!GetNumberX(bs,5, &pic->pquant)) return 0;
  pic->currentQuant = pic->pquant;
  /* CPM */
  if(!GetBitX(bs,&pic->cpm)) return 0;
  /* PSBI */
  if(pic->cpm) {
    if(!GetNumberX(bs, 2, &pic->psbi)) return 0;
  }
  /* TRB, DBQUANT */
  if(pic->pb) {
    if(!GetNumberX(bs, 3, &pic->trb)) return 0;
    if(!GetNumberX(bs, 2, &pic->dbquant)) return 0;
  }
  /* PEI, PSPARE */
  if(!GetBitX(bs, &pei)) return 0;
  while(pei) {
    if(!ConsumeBitsX(bs, 8)) return 0;
    if(!GetBitX(bs, &pei)) return 0;
  }

  /* set other usefull variables
   */
  switch(pic->sourceFormat) {
  case SQCIF: {
    pic->numberOfGOBs = 6; 
    pic->numberOfMBs = 8 * 1;
    pic->widthFrame = SQCIF_WIDTH;
    pic->heightFrame = SQCIF_HEIGHT;
    break; }
  case QCIF:  {
    pic->numberOfGOBs = 9;
    pic->numberOfMBs = 11 * 1;
    pic->widthFrame = QCIF_WIDTH;
    pic->heightFrame = QCIF_HEIGHT;
    break; }
  case CIF:   {
    pic->numberOfGOBs = 18;
    pic->numberOfMBs = 22 * 1;
    pic->widthFrame = CIF_WIDTH;
    pic->heightFrame = CIF_HEIGHT;
    break; }
  case CIF4:  {
    pic->numberOfGOBs = 18;
    pic->numberOfMBs = 44 * 2; 
    pic->widthFrame = 2 * CIF_WIDTH;
    pic->heightFrame = 2 * CIF_HEIGHT;
    break; }
  case CIF16: {
    pic->numberOfGOBs = 18;
    pic->numberOfMBs = 88 * 4; 
    pic->widthFrame = 4 * CIF_WIDTH;
    pic->heightFrame = 4 * CIF_HEIGHT;
    break; }
  default:    return 0;
  }

  if(pic->pb) 
    pic->numberOfBLKs=12; 
  else 
    pic->numberOfBLKs=6;

  switch(pic->pictureCodingType) {
  case PICTURE_CODING_TYPE_INTER:
    pic->numberOfMVs=1;
    if(pic->ap) pic->numberOfMVs+=3;
    if(pic->pb) pic->numberOfMVs++;
    break;
  case PICTURE_CODING_TYPE_INTRA:
    pic->numberOfMVs=0;
    break;
  }

  return 1;
}



/***********************************************************CommentBegin******
 *****************************************************************************
 *
 * -- H263getHeader -- Get an H.263 picture- or GOB-header
 *
 * Author:           K.S.
 *
 * Created:          16-Aug-97
 *
 * Purpose:          Get an H.263 picture- or GOB-header and fill the
 *                   information into the 'pic' struct.
 * 
 * Arguments in:     Bitstr        *bs     H.263 bitstream to be decoded
 *
 * Arguments in/out: H263picStruct *pic    All the information retrieved from
 *                                         the parsed header is filled into
 *                                         this struct.
 *
 * Arguments out:    Int           *gn_p   GOB number
 *
 * Return values:    number of parsed bits or
 *                   0   if EOS or
 *                   -1  if error
 *
 * Example:          H263getHeader(bs, &gn, pic);
 *
 * Side effects:     -
 *
 * Description:      -
 *
 * See also:         -
 *
 * Modified:         -
 *
 *****************************************************************************/
Int H263getHeader(Bitstr *bs, Int *gn_p, H263picStruct *pic)
/***********************************************************CommentEnd********/
{
  Int           bsIndStart = bs->ind;
  H263gobStruct *gob;


  /* Skip sync word */
  ConsumeBits(bs, 17);

  /* Get GOB number */
  if(!GetNumberX(bs, 5, gn_p))
    return(-1);

  if (*gn_p == 0) {    /* First GOB --> Picture header */
    H263getPictureHeaderWoSync(bs, pic);
    pic->previousGFID = -1;
  } else if (*gn_p == 31) {  /* EOS */
    return(0);
  } else {             /* GOB header */
    gob = pic->gob[*gn_p];
    /* GSBI */
    if(pic->cpm) {
      if(!GetNumberX(bs, 2, &(gob->gsbi)))
	return(-1);
    }
    /* GFID */
    if(!GetNumberX(bs, 2, &(gob->gfid)))
      return(-1);
    if (pic->previousGFID >= 0)
      if (gob->gfid == pic->previousGFID)
	pic->ptype = pic->previousPTYPE;
      else
	return(-2);
    pic->previousGFID = gob->gfid;
    /* GQUANT */
    if(!GetNumberX(bs, 5, &(gob->gquant)))
      return(-1);
    pic->currentQuant = gob->gquant;
  }

  return(bs->ind - bsIndStart);
}


/***********************************************************CommentBegin******
 *****************************************************************************
 *
 * -- GetLevelsX -- Read DCT coefficient levels from bitstream
 *
 * Author:           K.S.
 *
 * Created:          16-Sep-97
 *
 * Purpose:          Read DCT coefficient levels from the bitstream without
 *                   checking the bitstream size (bitioX routines).
 *
 * Arguments in:     -
 *
 * Arguments in/out: Bitstr   *bs      bitstream
 *
 * Arguments out:    Int      *levels  DCT coefficient levels
 *
 * Return values:    TRUE  (1) on success
 *                   FALSE (0) on failure
 *
 * Example:          GetLevelsX(bs, &levels);
 *
 * Side effects:     Changes state of the bitstream
 *
 * Description:      No zigzag-scanning is performed.
 *
 * See also:         -
 *
 * Modified:         -
 *
 *****************************************************************************/
Int GetLevelsX(Bitstr *bs, Int *levels)
/***********************************************************CommentEnd********/
{
  Int last, run, level;
  Int pos;


  memset(levels, 0, 64 * sizeof(int));
  pos=0;
  do {
    if(!GetLastRunLevelX(bs, &last, &run, &level)) {
      return 0;
    }
    pos+=run;
    if(pos>63) {
      return 0;
    }
    levels[pos] = level;
    pos++;
  } while(!last);

  return 1;
}


/***********************************************************CommentBegin******
 *****************************************************************************
 *
 * -- GetLastRunLevelX -- Read one last-run-level event from bitstream
 *
 * Author:           K.S.
 *
 * Created:          16-Sep-97
 *
 * Purpose:          Read one last, run, DCT coefficient level from the
 *                   bitstream without checking the bitstream size
 *                   (bitioX routines).
 *
 * Arguments in:     -
 *
 * Arguments in/out: Bitstr   *bs      bitstream
 *
 * Arguments out:    Int      *last    flag if this is the last coefficient
 *                                     in this block.
 *                   Int      *run     zero run from last DCT coefficient
 *                                     != 0.
 *                   Int      *level   DCT coefficient level
 *
 * Return values:    TRUE  (1) on success
 *                   FALSE (0) on failure
 *
 * Example:          GetLastRunLevelX(bs, &last, &run, &level);
 *
 * Side effects:     Changes state of the bitstream
 *
 * Description:      -
 *
 * See also:         -
 *
 * Modified:         -
 *
 *****************************************************************************/
Int GetLastRunLevelX(Bitstr *bs, Int *last, Int *run, Int *level)
/***********************************************************CommentEnd********/
{
  Int sign, n;


  n = H263parseTcoef(bs);
  if(n == 102) {   /* Escape */
    if(!GetBitX(bs,last))
      return 0;
    if(!GetNumberX(bs, 6, run))
      return 0;
    if(!GetNumberX(bs, 8, level))
      return 0;
    if (*level == 0 || *level == 0x80)
      return(0);
    if (*level > 127)
      *level -= 256;
  }
  else {
    if(!GetBitX(bs, &sign))
      return 0;
    if (n < 58) {
      *last = 0;
      if (n < 12) {
	*run = 0;
	*level = n + 1;
      } else if (n < 18) {
	*run = 1;
	*level = n - 11;
      } else if (n < 22) {
	*run = 2;
	*level = n - 17;
      } else if (n < 25) {
	*run = 3;
	*level = n - 21;
      } else if (n < 28) {
	*run = 4;
	*level = n - 24;
      } else if (n < 31) {
	*run = 5;
	*level = n - 27;
      } else if (n < 34) {
	*run = 6;
	*level = n - 30;
      } else if (n < 42) {
	*run = n / 2 - 10;
	*level = (n & 0x1) + 1;
      } else {
	*run = n - 31;
	*level = 1;
      }
    } else {
      *last = 1;
      if (n < 61) {
	*run = 0;
	*level = n - 57;
      } else if (n < 63) {
	*run = 1;
	*level = n - 60;
      } else {
	*run = n - 61;
	*level = 1;
      }
    }
    *level = sign ? -(*level) : *level;
  }

  return 1;
}


/***********************************************************CommentBegin******
 *****************************************************************************
 *
 * -- GetMVBaseModeX -- Read motion vector from bitstream
 *
 * Author:           K.S.
 *
 * Created:          16-Sep-97
 *
 * Purpose:          Read a motion vector from the bitstream without
 *                   checking the bitstream size (bitioX routines).
 *
 * Arguments in:     MVector        *pv    motion vector predictor
 *
 * Arguments in/out: Bitstr         *bs    bitstream
 *                   MVector        *mv    Resulting motion vector.
 *
 * Arguments out:    -
 *
 * Return values:    TRUE  (1) on success
 *                   FALSE (0) on failure
 *
 * Example:          GetMVBaseModeX(bs, pv, mv);
 *
 * Side effects:     Changes state of the bitstream.
 *
 * Description:      -
 *
 * See also:         -
 *
 * Modified:         -
 *
 *****************************************************************************/
Int GetMVBaseModeX(Bitstr *bs, MVector *pv, MVector *mv)
/***********************************************************CommentEnd********/
{
  Int   n, na, nz, nh;


  mv->sF = 2;

  /* read MVDs */
  /* x */
  n = H263parseMvd(bs);
  n -= 32;
  n += 2 * pv->x + pv->sx;
  if(n < -32)
    n += 64;
  else if (n > 31)
    n -= 64;
  nz = (n < 0);
  na = ABS(n);
  nh = na & 0x1;
  mv->sx = nh;
  na = na >> 1;
  mv->x = nz ? (-(na + nh)) : na;
  /* y */
  n = H263parseMvd(bs);
  n -= 32;
  n += 2 * pv->y + pv->sy;
  if(n < -32)
    n += 64;
  else if (n > 31)
    n -= 64;
  nz = (n < 0);
  na = ABS(n);
  nh = na & 0x1;
  mv->sy = nh;
  na = na >> 1;
  mv->y = nz ? (-(na + nh)) : na;

  return 1;
}


/***********************************************************CommentBegin******
 *****************************************************************************
 *
 * -- H263ParseDecodePictureBaseMode -- Decode picture in baseline mode
 *
 * Author:           K.S.
 *
 * Created:          15-Sep-97
 *
 * Purpose:          Decode picture in baseline mode
 * 
 * Arguments in:     Bitstr     *bs         bitstream
 *                   Picture    *refPict    previously decoded color image.
 *                                          Reference picture.
 *
 * Arguments in/out: H263picStruct *pic     contains all data from bitstream.
 *                                          Contains current state of decoder.
 *
 * Arguments out:    Picture    *pict       decoded color image.
 *
 * Return values:    -1                      if error
 *                   number of consumed bits if success
 *
 * Example:          H263ParseDecodePictureBaseMode(bs, pic, refPict, pict);
 *
 * Side effects:     -
 *
 * Description:      'bs->ind' has to point to the beginning of a
 *                   H.263 sync word.
 *
 * See also:         -
 *
 * Modified:         -
 *
 *****************************************************************************/
Int H263ParseDecodePictureBaseMode(Bitstr *bs, H263picStruct *pic,
				   Picture *refPict, Picture *pict)
/***********************************************************CommentEnd********/
{
  Int            bsIndStart = bs->ind;
  Int            gn;


  for(gn = 0; gn < pic->numberOfGOBs; gn++) {
    if (H263ParseDecodeGOBBaseMode(bs, pic, refPict, pict) <= 0)
      return(-1);
    ByteAlign(bs);
  }

  /* save PTYPE for GFID-Check */
  pic->previousPTYPE = pic->ptype;

  return(bs->ind - bsIndStart);
}


/***********************************************************CommentBegin******
 *****************************************************************************
 *
 * -- H263ParseDecodeGOBBaseMode -- Parse and dec. an GOB in baselinemode
 *
 * Author:           K.S.
 *
 * Created:          05-Aug-97
 *
 * Purpose:          Parse and decode one H.263 GOB in baseline mode
 *                   including picture- or GOB-header
 * 
 * Arguments in:     Bitstr        *bs       H.263 bitstream to be decoded
 *                   Picture       *refPict  previous (reference) color image.
 *
 * Arguments in/out: H263picStruct *pic      From this struct are needed:
 *                                           pic->mb    memory for macroblocks.
 *                                                      MB-types and MVs are
 *                                                      returned within.
 *
 * Arguments out:    Picture    *pict        currently decoded color image.
 *
 * Return values:    number of parsed bits or -1 if error
 *
 * Example:          H263ParseDecodeGOBBaseMode(bs, pic, prevPict, cPict);
 *
 * Side effects:     -
 *
 * Description:      Allowed picture formats are SQCIF, QCIF and CIF.
 *                   'bs->ind' has to point to the beginning of a
 *                   H.263 sync word.
 *
 * See also:         H263ParseDecodeGOBDataBaseMode
 *
 * Modified:         -
 *
 *****************************************************************************/
Int H263ParseDecodeGOBBaseMode(Bitstr *bs, H263picStruct *pic,
			       Picture *refPict, Picture *pict)
/***********************************************************CommentEnd********/
{
  Int bsIndStart = bs->ind;
  Int gn;
  Int n;


  /* Get header */
  n = H263getHeader(bs, &gn, pic);
  if (n <= 0)
    return(n);

  if(pic->sac == 1 || pic->umv == 1 || pic->ap == 1 || pic->pb == 1) {
    return(-1);
  }
  if (pict->w != pic->widthFrame || pict->h != pic->heightFrame)
    return(-1);

  if (H263ParseDecodeGOBDataBaseMode(gn, bs, pic, refPict, pict) <= 0)
    return(-1);

  return(bs->ind - bsIndStart);
}

Int H263ParseDecodeGOB(Bitstr *bs, Picture *prevPict, Picture *pict)
{
  static H263picStruct *ps = NULL;

  if (!ps)
    {
      ps = H263newPicStruct();
      ps->numberOfGOBs = pict->h / MACROBLOCK_SIZE;
      ps->numberOfMBs = pict->w / MACROBLOCK_SIZE;
      ps->numberOfMVs = 1;
      ps->numberOfBLKs = 6;
      H263allocPicStruct(ps);
    }

  return H263ParseDecodeGOBBaseMode(bs, ps, prevPict, pict);
}


/***********************************************************CommentBegin******
 *****************************************************************************
 *
 * -- H263ParseDecodeGOBDataBaseMode -- Parse and dec. an GOB in baselinemode
 *
 * Author:           K.S.
 *
 * Created:          05-Aug-97
 *
 * Purpose:          Parse and decode one H.263 GOB in baseline mode
 * 
 * Arguments in:     Int           gn      GOB number
 *                   Bitstr        *bs     H.263 bitstream to be decoded
 *                   Picture   *prevPict   previous (reference) color image.
 *
 * Arguments in/out: H263picStruct *pic    From this struct are needed:
 *                                         pic->mb    memory for macroblocks.
 *                                                    MB-types and MVs are
 *                                                    returned within.
 *                                         pic->umv   (shall be 0)
 *                                         pic->pictureCodingType:
 *                                                PICTURE_CODING_TYPE_INTRA or
 *                                                PICTURE_CODING_TYPE_INTER
 *                                         pic->currentQuant
 *                                                current quantizer
 *                                                (pquant or gquant)
 *                                                It may be changed by dquant.
 *
 * Arguments out:    Picture    *pict      currently decoded color image.
 *
 * Return values:    number of parsed bits or -1 if error
 *
 * Example:          H263ParseDecodeGOBDataBaseMode(gn, bs, pic,
 *                                                  prevPict, cPict);
 *
 * Side effects:     -
 *
 * Description:      Allowed picture formats are SQCIF, QCIF and CIF.
 *
 * See also:         -
 *
 * Modified:         -
 *
 *****************************************************************************/
Int H263ParseDecodeGOBDataBaseMode(Int gn, Bitstr *bs, H263picStruct *pic,
				   Picture *prevPict, Picture *pict)
/***********************************************************CommentEnd********/
{
  Int            bsIndStart = bs->ind;
  Int            n, mbn;
  Int            i, index;
  Int            x, y;
  Int            w, wc, hc;
  Int            nMbHor;
  Byte           *p1, *p2;
  Short          *pr;
  H263mbStruct   *mb;
  H263blkStruct  *h263blk;
  Int            coeff[BLOCK_SIZE*BLOCK_SIZE];
  Short          res[BLOCK_SIZE*BLOCK_SIZE];
  Block          blk, blkc;
  MVector        mv, pv;


  w = pict->w;
  wc = pict->w / 2;
  hc = pict->h / 2;
  nMbHor = pict->w / MACROBLOCK_SIZE;
  blk.w = MACROBLOCK_SIZE;
  blk.h = MACROBLOCK_SIZE;
  blkc.w = MACROBLOCK_SIZE / 2;
  blkc.h = MACROBLOCK_SIZE / 2;
  mv.sF = 2;
  mv.t = 0;
  pv.sF = 2;
  pv.t = 0;
  ResetMVector(&pv);

  for (i = 0; i < nMbHor; ++i) {
    mbn = gn * nMbHor + i;
    mb = pic->mb[mbn];

    if (pic->pictureCodingType == PICTURE_CODING_TYPE_INTER) {
      /* COD */
      if(!GetBitX(bs, &n))
	return(-1);
      /****************/
      /* Mode uncoded */
      /****************/
      if (n == 1) {
	ResetMVector(&pv);
	mb->type = MODE_UNCODED;
	index = (i + gn * w) * MACROBLOCK_SIZE;
	p1 = pict->y + index;
	p2 = prevPict->y + index;
	for (y = 0; y < MACROBLOCK_SIZE; y++) {
	  memcpy(p1, p2, MACROBLOCK_SIZE * sizeof(Byte));
	  p1 += w * sizeof(Byte);
	  p2 += w * sizeof(Byte);
	}
	/* Color components */
	index = (gn * wc + i) * (MACROBLOCK_SIZE / 2);
	p1 = pict->u + index;
	p2 = prevPict->u + index;
	for (y = 0; y < MACROBLOCK_SIZE / 2; y++) {
	  memcpy(p1, p2, MACROBLOCK_SIZE / 2 * sizeof(Byte));
	  p1 += wc * sizeof(Byte);
	  p2 += wc * sizeof(Byte);
	}
	p1 = pict->v + index;
	p2 = prevPict->v + index;
	for (y = 0; y < MACROBLOCK_SIZE / 2; y++) {
	  memcpy(p1, p2, MACROBLOCK_SIZE / 2 * sizeof(Byte));
	  p1 += wc * sizeof(Byte);
	  p2 += wc * sizeof(Byte);
	}
	continue;
      }
    }

    /* MCBPC */
    switch (pic->pictureCodingType) {
    case PICTURE_CODING_TYPE_INTRA:
      do {
	n = H263parseMcbpcI(bs);
      } while (n == 8);     /* While stuffing */
      mb->type = (n & 0x4) ? 4 : 3;
      break;
    case PICTURE_CODING_TYPE_INTER:
      do {
	n = H263parseMcbpcP(bs);
      } while (n == 0x14);     /* While stuffing */
      mb->type = (n >> 2);
      break;
    }
    mb->cbp[5] = (n & 0x1);
    mb->cbp[4] = ((n >> 1) & 0x1);
    if (mb->type == MODE_INTER4V)   /* Inter4v only in AP mode*/
      return(-1);

    /* CBPY */
    n = H263parseCbpyI(bs);
    if (mb->type == MODE_INTER || mb->type == MODE_INTER_Q ||
	mb->type == MODE_INTER4V) {
      n = ~n;
    }
    mb->cbp[3] = (n & 0x1);
    mb->cbp[2] = ((n >> 1) & 0x1);
    mb->cbp[1] = ((n >> 2) & 0x1);
    mb->cbp[0] = ((n >> 3) & 0x1);

    /* DQUANT */
    if((mb->type == MODE_INTRA_Q) || (mb->type == MODE_INTER_Q)) {
      if(!GetNumberX(bs, 2, &n))
	return(-1);
      n = ((n & 0x2) ? n-- : ((n == 0) ? -1 : -2));   /* Mapping */
      pic->currentQuant += n;
      if (pic->currentQuant < 1 || pic->currentQuant > 31)
	return(-1);
    }
    mb->quant = pic->currentQuant;

    switch (mb->type) {
    case MODE_INTRA:
    case MODE_INTRA_Q:
      ResetMVector(&pv);
      blk.x0 = i * MACROBLOCK_SIZE;
      blk.y0 = gn * MACROBLOCK_SIZE;

      /* Block 0 */
      /* Intra DC */
      if(!GetNumberX(bs, 8, &n))
	return(-1);
      if (n == 0 || n == 0x80)
	return(-1);
      h263blk = mb->blk[0];
      h263blk->intradc = ((n == 0xff) ? 1024 : (n * 8));
      /* T coeffs */
      if(mb->cbp[0]) {
	if(!GetLevelsX(bs, h263blk->level))
	  return(-1);
      } else
	memset(h263blk->level, 0, 64 * sizeof(Int));
      H263dequant(h263blk->level, h263blk->intradc, mb->quant, mb->type,
		  mb->cbp[0], coeff);
      IdctFast(coeff, res);
      p1 = pict->y + blk.x0 + blk.y0 * w;
      pr = res;
      for (y = 0; y < BLOCK_SIZE; y++) {
	for (x = 0; x < BLOCK_SIZE; x++, pr++) {
	  *(p1++) = CLIP(*pr, 0, 255);
	}
	p1 += w - BLOCK_SIZE;
      }
      /* Block 1 */
      /* Intra DC */
      if(!GetNumberX(bs, 8, &n))
	return(-1);
      if (n == 0 || n == 0x80)
	return(-1);
      h263blk = mb->blk[1];
      h263blk->intradc = ((n == 0xff) ? 1024 : (n * 8));
      /* T coeffs */
      if(mb->cbp[1]) {
	if(!GetLevelsX(bs, h263blk->level))
	  return(-1);
      } else
	memset(h263blk->level, 0, 64 * sizeof(Int));
      H263dequant(h263blk->level, h263blk->intradc, mb->quant, mb->type,
		  mb->cbp[1], coeff);
      IdctFast(coeff, res);
      p1 = pict->y + blk.x0 + BLOCK_SIZE + blk.y0 * w;
      pr = res;
      for (y = 0; y < BLOCK_SIZE; y++) {
	for (x = 0; x < BLOCK_SIZE; x++, pr++) {
	  *(p1++) = CLIP(*pr, 0, 255);
	}
	p1 += w - BLOCK_SIZE;
      }
      /* Block 2 */
      /* Intra DC */
      if(!GetNumberX(bs, 8, &n))
	return(-1);
      if (n == 0 || n == 0x80)
	return(-1);
      h263blk = mb->blk[2];
      h263blk->intradc = ((n == 0xff) ? 1024 : (n * 8));
      /* T coeffs */
      if(mb->cbp[2]) {
	if(!GetLevelsX(bs, h263blk->level))
	  return(-1);
      } else
	memset(h263blk->level, 0, 64 * sizeof(Int));
      H263dequant(h263blk->level, h263blk->intradc, mb->quant, mb->type,
		  mb->cbp[2], coeff);
      IdctFast(coeff, res);
      p1 = pict->y + blk.x0 + (blk.y0 + BLOCK_SIZE) * w;
      pr = res;
      for (y = 0; y < BLOCK_SIZE; y++) {
	for (x = 0; x < BLOCK_SIZE; x++, pr++) {
	  *(p1++) = CLIP(*pr, 0, 255);
	}
	p1 += w - BLOCK_SIZE;
      }
      /* Block 3 */
      /* Intra DC */
      if(!GetNumberX(bs, 8, &n))
	return(-1);
      if (n == 0 || n == 0x80)
	return(-1);
      h263blk = mb->blk[3];
      h263blk->intradc = ((n == 0xff) ? 1024 : (n * 8));
      /* T coeffs */
      if(mb->cbp[3]) {
	if(!GetLevelsX(bs, h263blk->level))
	  return(-1);
      } else
	memset(h263blk->level, 0, 64 * sizeof(Int));
      H263dequant(h263blk->level, h263blk->intradc, mb->quant, mb->type,
		  mb->cbp[3], coeff);
      IdctFast(coeff, res);
      p1 = pict->y + blk.x0 + BLOCK_SIZE + (blk.y0 + BLOCK_SIZE) * w;
      pr = res;
      for (y = 0; y < BLOCK_SIZE; y++) {
	for (x = 0; x < BLOCK_SIZE; x++, pr++) {
	  *(p1++) = CLIP(*pr, 0, 255);
	}
	p1 += w - BLOCK_SIZE;
      }

      /* Color components */
      blkc.x0 = i * (MACROBLOCK_SIZE / 2);
      blkc.y0 = gn * (MACROBLOCK_SIZE / 2);

      /* U Block */
      /* Intra DC */
      if(!GetNumberX(bs, 8, &n))
	return(-1);
      if (n == 0 || n == 0x80)
	return(-1);
      h263blk = mb->blk[4];
      h263blk->intradc = ((n == 0xff) ? 1024 : (n * 8));
      /* T coeffs */
      if(mb->cbp[4]) {
	if(!GetLevelsX(bs, h263blk->level))
	  return(-1);
      } else
	memset(h263blk->level, 0, 64 * sizeof(Int));
      H263dequant(h263blk->level, h263blk->intradc, mb->quant, mb->type,
		  mb->cbp[4], coeff);
      IdctFast(coeff, res);
      p1 = pict->u + blkc.x0 + blkc.y0 * wc;
      pr = res;
      for (y = 0; y < BLOCK_SIZE; y++) {
	for (x = 0; x < BLOCK_SIZE; x++, pr++) {
	  *(p1++) = CLIP(*pr, 0, 255);
	}
	p1 += wc - BLOCK_SIZE;
      }

      /* V Block */
      /* Intra DC */
      if(!GetNumberX(bs, 8, &n))
	return(-1);
      if (n == 0 || n == 0x80)
	return(-1);
      h263blk = mb->blk[5];
      h263blk->intradc = ((n == 0xff) ? 1024 : (n * 8));
      /* T coeffs */
      if(mb->cbp[5]) {
	if(!GetLevelsX(bs, h263blk->level))
	  return(-1);
      } else
	memset(h263blk->level, 0, 64 * sizeof(Int));
      H263dequant(h263blk->level, h263blk->intradc, mb->quant, mb->type,
		  mb->cbp[5], coeff);
      IdctFast(coeff, res);
      p1 = pict->v + blkc.x0 + blkc.y0 * wc;
      pr = res;
      for (y = 0; y < BLOCK_SIZE; y++) {
	for (x = 0; x < BLOCK_SIZE; x++, pr++) {
	  *(p1++) = CLIP(*pr, 0, 255);
	}
	p1 += wc - BLOCK_SIZE;
      }
      break;
    case MODE_INTER:
    case MODE_INTER_Q:
      /* MVD */
      if(!GetMVBaseModeX(bs, &pv, &mv))
	return(-1);
      CopyMVector(&mv, &pv);
      CopyMVector(&mv, mb->mv[0]);

      blk.x0 = i * MACROBLOCK_SIZE;
      blk.y0 = gn * MACROBLOCK_SIZE;
      HalfPelBMCBaseModeFast(prevPict->y, w, pict->h, mv,
			     blk.x0, blk.y0, MACROBLOCK_SIZE, pict->y);
      /* Block 0 */
      if(mb->cbp[0]) {
	h263blk = mb->blk[0];
	if(!GetLevelsX(bs, h263blk->level))
	  return(-1);
	H263dequant(h263blk->level, h263blk->intradc, mb->quant, mb->type,
		    mb->cbp[0], coeff);
	IdctFast(coeff, res);
	p1 = pict->y + blk.x0 + blk.y0 * w;
	pr = res;
	for (y = 0; y < BLOCK_SIZE; y++) {
	  for (x = 0; x < BLOCK_SIZE; x++, pr++) {
	    *pr += *p1;
	    *(p1++) = CLIP(*pr, 0, 255);
	  }
	  p1 += w - BLOCK_SIZE;
	}
      }
      /* Block 1 */
      if(mb->cbp[1]) {
	h263blk = mb->blk[1];
	if(!GetLevelsX(bs, h263blk->level))
	  return(-1);
	H263dequant(h263blk->level, h263blk->intradc, mb->quant, mb->type,
		    mb->cbp[1], coeff);
	IdctFast(coeff, res);
	p1 = pict->y + blk.x0 + BLOCK_SIZE + blk.y0 * w;
	pr = res;
	for (y = 0; y < BLOCK_SIZE; y++) {
	  for (x = 0; x < BLOCK_SIZE; x++, pr++) {
	    *pr += *p1;
	    *(p1++) = CLIP(*pr, 0, 255);
	  }
	  p1 += w - BLOCK_SIZE;
	}
      }
      /* Block 2 */
      if(mb->cbp[2]) {
	h263blk = mb->blk[2];
	if(!GetLevelsX(bs, h263blk->level))
	  return(-1);
	H263dequant(h263blk->level, h263blk->intradc, mb->quant, mb->type,
		    mb->cbp[2], coeff);
	IdctFast(coeff, res);
	p1 = pict->y + blk.x0 + (blk.y0 + BLOCK_SIZE) * w;
	pr = res;
	for (y = 0; y < BLOCK_SIZE; y++) {
	  for (x = 0; x < BLOCK_SIZE; x++, pr++) {
	    *pr += *p1;
	    *(p1++) = CLIP(*pr, 0, 255);
	  }
	  p1 += w - BLOCK_SIZE;
	}
      }
      /* Block 3 */
      if(mb->cbp[3]) {
	h263blk = mb->blk[3];
	if(!GetLevelsX(bs, h263blk->level))
	  return(-1);
	H263dequant(h263blk->level, h263blk->intradc, mb->quant, mb->type,
		    mb->cbp[3], coeff);
	IdctFast(coeff, res);
	p1 = pict->y + blk.x0 + BLOCK_SIZE + (blk.y0 + BLOCK_SIZE) * w;
	pr = res;
	for (y = 0; y < BLOCK_SIZE; y++) {
	  for (x = 0; x < BLOCK_SIZE; x++, pr++) {
	    *pr += *p1;
	    *(p1++) = CLIP(*pr, 0, 255);
	  }
	  p1 += w - BLOCK_SIZE;
	}
      }

      /* Color components */
      DivideBy2HalfPelMVector(&mv,&mv);
      blkc.x0 = i * (MACROBLOCK_SIZE / 2);
      blkc.y0 = gn * (MACROBLOCK_SIZE / 2);
      /* U Block */
      HalfPelBMCBaseModeFast(prevPict->u, wc, hc, mv, blkc.x0, blkc.y0,
			     MACROBLOCK_SIZE / 2, pict->u);
      if(mb->cbp[4]) {
	h263blk = mb->blk[4];
	if(!GetLevelsX(bs, h263blk->level))
	  return(-1);
	H263dequant(h263blk->level, h263blk->intradc, mb->quant, mb->type,
		    mb->cbp[4], coeff);
	IdctFast(coeff, res);
	p1 = pict->u + blkc.x0 + blkc.y0 * wc;
	pr = res;
	for (y = 0; y < BLOCK_SIZE; y++) {
	  for (x = 0; x < BLOCK_SIZE; x++, pr++) {
	    *pr += *p1;
	    *(p1++) = CLIP(*pr, 0, 255);
	  }
	  p1 += wc - BLOCK_SIZE;
	}
      }
      /* V Block */
      HalfPelBMCBaseModeFast(prevPict->v, wc, hc, mv, blkc.x0, blkc.y0,
			     MACROBLOCK_SIZE / 2, pict->v);
      if(mb->cbp[5]) {
	h263blk = mb->blk[5];
	if(!GetLevelsX(bs, h263blk->level))
	  return(-1);
	H263dequant(h263blk->level, h263blk->intradc, mb->quant, mb->type,
		    mb->cbp[5], coeff);
	IdctFast(coeff, res);
	p1 = pict->v + blkc.x0 + blkc.y0 * wc;
	pr = res;
	for (y = 0; y < BLOCK_SIZE; y++) {
	  for (x = 0; x < BLOCK_SIZE; x++, pr++) {
	    *pr += *p1;
	    *(p1++) = CLIP(*pr, 0, 255);
	  }
	  p1 += wc - BLOCK_SIZE;
	}
      }
      break;
    }
  }


  return(bs->ind - bsIndStart);
}
