#include "global.h"



/***********************************************************CommentBegin******
 *****************************************************************************
 *
 * -- HalfPelBMCBaseModeFast -- Half pel Block Motion Compensation
 *
 * Author:           K.S.
 *
 * Created:          13-Nov-96
 *
 * Purpose:          Half pel block motion compensation for one Block from
 *                   one image to another
 * 
 * Arguments in:     Image      *img     input image
 *                   MVector    *mv      motion vector to displace block
 *                   Block      *blk     block the should be displaced
 *
 * Arguments in/out: Image      *mcImg   motion compensated image
 *
 * Arguments out:    -
 *
 * Return values:    0 if error
 *                   1 if ok
 *
 * Example:          HalfPelBMCompensation(img, mv, blk, mcImg);
 *
 * Side effects:     -
 *
 * Description:      * The motion vector 'mv' points from 'mcImg' to 'img'.
 *                   * The block 'blk' is located in 'mcImg'.
 *                   * The block 'blk' must be inside 'mcImg'. This is not
 *                     checked.
 *
 * See also:         -
 *	
 * Modified:         -
 *	
 *****************************************************************************/
Void HalfPelBMCBaseModeFast(Byte *ip, int w, int h, MVector mv, Int xb, Int yb,
			    Int s, Byte *mcIp)
/***********************************************************CommentEnd********/
{
  Int x;                                /* Horizontal loop variable */
  Int y;                                /* Vertical loop variable */
  Int ws = w - s;
  Byte *mcp, *ip1, *ip2;


  if (mv.sx == 0 && mv.sy == 0) {  /* Fullpel */
    if (mv.x == 0 && mv.y == 0) {  /* No displacement at all */
      for (y = yb; y < yb + s; y++)
	memcpy(mcIp + xb + y * w, ip + xb + y * w, s);
    } else {
      mcp = mcIp + xb + yb * w;
      ip1 = ip + xb + mv.x + (yb + mv.y) * w;
      for(y = 0; y < s; y++) {
	memcpy(mcp, ip1, s);
	ip1 += w;
	mcp += w;
      }
    }
  } else if (mv.sy == 0) {  /* Halfpel only in x: (A + B + 1) / 2 */
    Short value;            /* Tmp integer variable for interpolation */
    
    mcp = mcIp + xb + yb * w;
    ip1 = ip + xb + mv.x + (yb + mv.y) * w;
    for(y = 0; y < s; y++) {
      for(x = 0; x < s; x++) {
	value = *ip1;
	value += *(++ip1);
	value++;
	value /= 2;
	*(mcp++) = (Byte)value;
      }
      ip1 += ws;
      mcp += ws;
    }
  } else if (mv.sx == 0) {  /* Halfpel only in y: (A + C + 1) / 2 */
    Short value;            /* Tmp integer variable for interpolation */

    mcp = mcIp + xb + yb * w;
    ip1 = ip + xb + mv.x + (yb + mv.y) * w;
    ip2 = ip1 + w;
    for(y = 0; y < s; y++) {
      for(x = 0; x < s; x++) {
	value = *(ip1++);
	value += *(ip2++);
	value++;
	value /= 2;
	*(mcp++) = (Byte)value;
      }
      ip1 += ws;
      ip2 += ws;
      mcp += ws;
    }
  } else {                /* Halfpel in x and y: (A + B + C + D + 2) / 4 */
    Short value;            /* Tmp integer variable for interpolation */

    mcp = mcIp + xb + yb * w;
    ip1 = ip + xb + mv.x + (yb + mv.y) * w;
    ip2 = ip1 + w;
    for(y = 0; y < s; y++) {
      for(x = 0; x < s; x++) {
	value = *ip1;
	value += *(++ip1);
	value += *ip2;
	value += *(++ip2);
	value += 2;
	value /= 4;
	*(mcp++) = (Byte)value;
      }
      ip1 += ws;
      ip2 += ws;
      mcp += ws;
    }
  }
}


/***********************************************************CommentBegin******
 *****************************************************************************
 *
 * -- HalfPelBMCFast -- Half pel Block Motion Compensation
 *
 * Author:           K.S.
 *
 * Created:          13-Nov-96
 *
 * Purpose:          Half pel block motion compensation for one Block from
 *                   one image to another
 * 
 * Arguments in:     Image      *img     input image
 *                   MVector    *mv      motion vector to displace block
 *                   Block      *blk     block the should be displaced
 *
 * Arguments in/out: Image      *mcImg   motion compensated image
 *
 * Arguments out:    -
 *
 * Return values:    0 if error
 *                   1 if ok
 *
 * Example:          HalfPelBlockMotionCompensation(img, mv, blk, mcImg);
 *
 * Side effects:     -
 *
 * Description:      * The motion vector 'mv' points from 'mcImg' to 'img'.
 *                   * The block 'blk' is located in 'mcImg'.
 *                   * The block 'blk' must be inside 'mcImg'. This is not
 *                     checked.
 *
 * See also:         -
 *	
 * Modified:         -
 *	
 *****************************************************************************/
Void HalfPelBMCFast(Byte *ip, int w, int h, MVector mv,
		    Int xb, Int yb, Int s, Byte *mcIp)
/***********************************************************CommentEnd********/
{
  Int x;                                /* Horizontal loop variable */
  Int y;                                /* Vertical loop variable */
  Int xiMax = w - 1;               /* max. x in input image */
  Int yiMax = h - 1;               /* max. y in input image */
  Int ws = w - s;
  Int x0, y0, x1, y1;                   /* image indices for interpolation */
  Byte *mcp;


  if (mv.sx == 0 && mv.sy == 0) {  /* Fullpel */
    if (mv.x == 0 && mv.y == 0) {  /* No displacement at all */
      for (y = yb; y < yb + s; y++)
	memcpy(mcIp + y * w, ip + y * w, s);
    } else {
      mcp = mcIp + xb + yb * w;
      for(y = 0; y < s; y++) {
	for(x = 0; x < s; x++) {
	  x0 = xb + x;
	  y0 = yb + y;
	  *(mcp++) = ip[CLIP(y0 + mv.y, 0, yiMax) * w +
		       CLIP(x0 + mv.x, 0, xiMax)];
	}
	mcp += ws;
      }
    }
  } else if (mv.sy == 0) {  /* Halfpel only in x: (A + B + 1) / 2 */
    Int value;            /* Tmp integer variable for interpolation */
    
    mcp = mcIp + xb + yb * w;
    for(y = 0; y < s; y++) {
      for(x = 0; x < s; x++) {
	x0 = xb + x + mv.x;
	x1 = CLIP(x0 + 1, 0, xiMax);
	x0 = CLIP(x0, 0, xiMax);
	y0 = CLIP(yb + y + mv.y, 0, yiMax) * w;
	value = (Int)(ip[y0 + x0]);
	value += (Int)(ip[y0 + x1]);
	value++;
	value /= 2;
	*(mcp++) = (Byte)value;
      }
      mcp += ws;
    }
  } else if (mv.sx == 0) {  /* Halfpel only in y: (A + C + 1) / 2 */
    Int value;            /* Tmp integer variable for interpolation */

    mcp = mcIp + xb + yb * w;
    for(y = 0; y < s; y++) {
      for(x = 0; x < s; x++) {
	x0 = CLIP(xb + x + mv.x, 0, xiMax);
	y0 = yb + y + mv.y;
	y1 = CLIP(y0 + 1, 0, yiMax) * w;
	y0 = CLIP(y0, 0, yiMax) * w;
	value = (Int)(ip[y0 + x0]);
	value += (Int)(ip[y1 + x0]);
	value++;
	value /= 2;
	*(mcp++) = (Byte)value;
      }
      mcp += ws;
    }
  } else {                /* Halfpel in x and y: (A + B + C + D + 2) / 4 */
    Int value;            /* Tmp integer variable for interpolation */

    mcp = mcIp + xb + yb * w;
    for(y = 0; y < s; y++) {
      for(x = 0; x < s; x++) {
	x0 = xb + x + mv.x;
	x1 = CLIP(x0 + 1, 0, xiMax);
	x0 = CLIP(x0, 0, xiMax);
	y0 = yb + y + mv.y;
	y1 = CLIP(y0 + 1, 0, yiMax) * w;
	y0 = CLIP(y0, 0, yiMax) * w;
	value = (Int)(ip[y0 + x0]);
	value += (Int)(ip[y0 + x1]);
	value += (Int)(ip[y1 + x0]);
	value += (Int)(ip[y1 + x1]);
	value += 2;
	value /= 4;
	*(mcp++) = (Byte)value;
      }
      mcp += ws;
    }
  }
}
